﻿use strict;
use utf8;

use letter;
use errata;

#***********************************************************
# パッケージconverter
#***********************************************************
package converter;

my $COMP_CHECK_FOR_USE_TAG = 0;

my $CR_CODE = 0xe000;		# 改行コード。
my $TAG_CODE = 0xfffe;		# タグ開始コード。
my $EOM_CODE = 0xffff;		# EOMコード。
my $CMP_CODE = 0xf100;		# 圧縮文字列コード。海外版の独自拡張。

my $JOSHI_NONE = 0;
my %JOSHI;
my $EMPTY = [];
my @JOSHI_LIST = (
	[ "は" ],				# 1
	[ "を" ],				# 2
	[ "が" ],				# 3
	[ "と" ],				# 4
	[ "に" ],				# 5
	[ "01", "０１", "い" ],	# 6
	[ "あ" ],				# 7
	$EMPTY, $EMPTY, $EMPTY, $EMPTY, $EMPTY, $EMPTY, $EMPTY, $EMPTY, $EMPTY,
	$EMPTY,
	[ "(色0)を" ]			#18
);

my %proc;

#===========================================================
# パッケージ初期化。
#===========================================================
{
	my ($str, $cnt, $c, %tmp, %kor);

	$str = $letter::KOR_ONLY;
	while (($c = substr($str, 0, 1, "")) ne "") {
		$kor{$c} = 1;
	}

	$str = $letter::LETTER;
	$cnt = 1;
	while (($c = substr($str, 0, 1, "")) ne "") {
		if ($c ne '_') {
			$proc{"LANG_JAPAN"}{$c} = create_normal_conv($cnt, $c, 1, 1);
			$proc{"LANG_KOREA"}{$c} = create_normal_conv($cnt, $c, $kor{$c}, 1);
		}
		$cnt++;
	}
	$str = $letter::LETTER_KOR;
	$cnt = 0x401;
	while (($c = substr($str, 0, 1, "")) ne "") {
		if ($c ne '_') {
			$proc{"LANG_KOREA"}{$c} = create_normal_conv($cnt, $c, 1, 1);
		}
		$cnt++;
	}
	$tmp{"▽"} = \&scroll_wait_conv;
	$tmp{"▼"} = \&normal_wait_conv;
	$tmp{"\n"} = \&newline_conv;
	$tmp{"\["} = create_wrapper_end_wait(\&tag_conv);
	$tmp{"_"} = create_underscore_conv(\%letter::UNDERSCORE);
	for my $i (keys %tmp) {
		$proc{"LANG_JAPAN"}{$i} = $tmp{$i};
		$proc{"LANG_KOREA"}{$i} = $tmp{$i};
	}

	# 助詞リストを作成。
	$cnt = 1;
	for my $i (@JOSHI_LIST) {
		for my $j (@{$i}) {
			$JOSHI{$j} = $cnt;
		}
		$cnt++;
	}
}

#===========================================================
# オブジェクト生成。
#===========================================================
sub new {
	my ($class, $lang) = @_;

	bless {
		proc => $proc{$lang}
	}, $class;
}

#===========================================================
# コンバート実行。
#===========================================================
sub exec {
	my ($self, $text, $repo, $except, $use_tag_list, $scroll, $win_con,
		$need_wait, $limit, $line_limit, $comm) = @_;
	my ($res, $c, @str, $str);

	# 変換処理。
	$self->{last_color} = 0;
	$self->{skip_cr} = 0;
	$self->{end_wait} = 0;
	$self->{except} = $except;
	$self->{use_tag_list} = {};
	$self->{scroll} = $scroll;
	$self->{win_con} = $win_con;
	$self->{line_num} = 1;
	$self->{line_limit} = $line_limit;
	$self->{line_over} = 0;
	$self->{comm} = $comm;

	while (($c = substr($text, 0, 1, "")) ne "") {
		my $sub;

		$sub = $self->{proc}{$c};
		if (!defined($sub)) {
			$$repo .= errata::out("変換不可", $c);
			next;
		}
		$res->{data} .= $self->$sub(\$text, $repo);
	}
	if (defined($use_tag_list)) {
		my (@tag_list, %use_tag_list);

		@tag_list = split(/\//, $use_tag_list);
		for my $i (@tag_list) {
			$i =~ s/-(\d+)//;
			$use_tag_list{$i} = $1;
		}
		for my $i (keys %{$self->{use_tag_list}}) {
			if (!exists $use_tag_list{$i}) {
				$$repo .= errata::out("不正タグ", $i);
			} elsif ($COMP_CHECK_FOR_USE_TAG) {
				if ($use_tag_list{$i} < $self->{use_tag_list}{$i}) {
					$$repo .= errata::out("使いすぎ", $i);
				} elsif ($self->{use_tag_list}{$i} < $use_tag_list{$i}) {
					$$repo .= errata::out("未タグ", $i);
				}
				delete $use_tag_list{$i};
			}
		}
		if ($COMP_CHECK_FOR_USE_TAG) {
			for my $i (keys %use_tag_list) {
				$$repo .= errata::out("未タグ", $i);
			}
		}
	}
	if (defined($need_wait)) {
		if ($need_wait eq "TRUE") {
			if ($self->{end_wait} != 1) {
				$$repo .= errata::out("終待ち", "ウェイト無し");
			}
		} else {
			if ($self->{end_wait} == 1) {
				$$repo .= errata::out("終待ち", "ウェイト有り");
			}
		}
	}

	# デフォルトカラーに戻されていない単語
	if ($self->{last_color} != 0) {
		$res->{data} .= $self->tag_bin('FF:00:色:0');
	}

	if (defined($limit)) {
		my $length;

		$length = length($res->{data}) / 2;
		if ($limit < $length) {
			$$repo .= errata::out("文字数超", $limit, $length);
			$res->{data} = substr($res->{data}, 0, $length * 2);
		}
	}

	$res->{data} .= pack('S', $EOM_CODE);

	# １文字２バイト固定なので
	$res->{length} = length($res->{data}) / 2;

	# 日本語でローカライズを考慮して行っていた長さ２倍の件は、
	# 海外版では削除する。

	return $res;
}

#===========================================================
# ポケモン文字圧縮。
#===========================================================
sub pokeStrCompress {
	my ($src, $org_fname, $msg_id) = @_;
	my ($dst, $left, $code, $c, $need_add);
	my $CW = 9;
	my $D_EOM = (1 << $CW) - 1;
	my $MSK_ERR = ~$D_EOM & 0xffff;

	# $srcは、必ず長さが1以上でEOMで終端する文字の必要あり。
	# また、最後にEOMが続くような場合は、符号化文字列にもEOMが続く。
	$dst = "";
	# 最初の$codeの値は使用されないが、念のため0に。
	$code = 0;
	# $leftの初期値は0。このため、必ず先頭に余分な$codeが付くことになる。
	$left = 0;
	while (($c = substr($src, 0, 2, "")) ne "") {
		my $d = unpack("S", $c);

		if ($d eq 0xffff) {
			# 残りの部分までビットセットされるために。
			$d = 0x7fffffff;
		} elsif (($d & $MSK_ERR) || $d == $D_EOM) {
			# タグや改行コード、ウェイトなどの特殊文字には未対応。
			# また、$D_EOMは符号文字列でのEOMのため使えません。
			die "圧縮対象文字コードが不適切。\n";
		}
		if ($left < $CW) {
			# $leftは0～($CW-1)。
			# 下位ビット（全くない場合もある）をエンコードしてパック。
			# EOMでも既存ビットがあるため、最上位ビットはクリアする。
			$code |= ($d << (15 - $left)) & 0x7fff;
			$dst .= pack("S", $code);
			# 残り上位ビットを次の初期値に。
			# EOMの場合は、これだけのcodeになるため、最上位ビットをクリアしない。
			$code = ($d >> $left) & 0xffff;
			$left = 15 - ($CW - $left);
			# このケースで終わった場合、最後に本来のEOMが付くことになる。
			$need_add = 0;
		} else {
			# $leftは$CW～14。
			# EOMの場合は既存ビットが存在するため、最上位ビットはクリアする。
			$code |= ($d << (15 - $left)) & 0x7fff;
			$left -= $CW;
			# このケースで終わった場合、本来のEOMを付ける必要がある。
			$need_add = 1;
		}
	}
	# 最後の$codeをパック。
	$dst .= pack("S", $code);
	if ($need_add) {
		$dst .= pack("S", 0xffff);
	}
	# 先頭の余分な$codeを取り除く。
	substr($dst, 0, 2, "");
	# 圧縮文字列のマークを付ける。
	$dst = pack("S", $CMP_CODE) . $dst;

	return $dst;
}

#---------------------------------------------------------------
# private
#---------------------------------------------------------------

#===============================================================
# 終了ノーマルウェイト後の状態を判断するラッパー。
#===============================================================
sub create_wrapper_end_wait {
	my ($conv) = @_;
	my $sub;

	$sub = sub {
		my ($self, $str, $repo) = @_;

		if ($self->{end_wait} == 1) {
			$self->{end_wait} = 2;
		}
		if (!$self->{line_over}) {
			if (defined($self->{line_limit}) && $self->{line_limit} < $self->{line_num}) {
				$$repo .= errata::out("行数超", $self->{line_limit}, $self->{win_con});
				$self->{line_over} = 1;
			}
		}
		&$conv;
	};
	return $sub;
}

#===============================================================
# 通常の文字変換。
#===============================================================
sub create_normal_conv {
	my ($conv, $c, $ose, $comm) = @_;
	my $sub;

	$sub = sub {
		my ($self, $str, $repo) = @_;

		if (!$self->{except}) {
			if (!$ose) {
				$$repo .= errata::out("未使用", $c);
			} elsif ($self->{comm} && !$comm) {
				$$repo .= errata::out("不通信", $c);
			}
		}
		return pack('S', $conv);
	};
	return create_wrapper_end_wait($sub);
}

#===============================================================
# スクロールウェイト変換。
#===============================================================
sub scroll_wait_conv {
	my ($self, $str, $repo) = @_;

	$self->{end_wait} = 1;
	if (defined($self->{scroll}) && $self->{scroll} ne "TRUE") {
		$$repo .= errata::out("ウェイト", $self->{win_con});
		return "";
	}
	$self->{skip_cr} = 1;		# 次の改行は無視する
	return pack('S', 0x25bd);
}

#===============================================================
# ノーマルウェイト変換。
#===============================================================
sub normal_wait_conv {
	my ($self, $str, $repo) = @_;

	$self->{end_wait} = 1;
	if (defined($self->{scroll}) && $self->{scroll} ne "TRUE") {
		$$repo .= errata::out("ウェイト", $self->{win_con});
		return "";
	}
	$self->{line_num} = 1;
	$self->{skip_cr} = 1;		# 次の改行は無視する
	return pack('S', 0x25bc);
}

#===============================================================
# 改行変換。
#===============================================================
sub newline_conv {
	my ($self) = @_;
	my $out;

	if ($self->{skip_cr} == 0) {
		$self->{line_num}++;
		$out = pack('S', $CR_CODE);
	} else {
		$self->{skip_cr} = 0;
		$out = "";
	}
	return $out;
}

#===============================================================
# _XXX_変換。
#===============================================================
sub create_underscore_conv {
	my ($underscore_key) = @_;
	my (%underscore_proc, $sub, %test);

	for my $i (keys %{$underscore_key}) {
		$underscore_proc{$i} = create_underscore_normal_conv($underscore_key->{$i});
	}
	$underscore_proc{"CTRL-C"} = \&underscore_ctrl_c_conv;
	for my $i (keys %underscore_proc) {
		if (defined($test{$i})) {
			die "error: _${i}_が被ってます。\n";
		}
		$test{$i} = 1;
	}
	$sub = sub {
		my ($self, $text, $repo) = @_;
		my ($key, $proc);

		if (${$text} !~ s/^([^_]+)_//) {
			$$repo .= errata::out("_無");
			return "";
		}
		$key = $1;
		$proc = $underscore_proc{$key};
		if (defined($proc)) {
			return $self->$proc();
		} else {
			my ($val);

			$val = $self->underscore_other_conv($key);
			if (!defined($val)) {
				$$repo .= errata::out("_未定義", $key);
				return "";
			}
			return $val;
		}
	};
	return create_wrapper_end_wait($sub);
}

#===============================================================
# _XXX_通常変換。
#===============================================================
sub create_underscore_normal_conv {
	my ($val) = @_;
	my ($sub);

	$sub = sub {
		return $val;
	};
	return $sub;
}

#===============================================================
# _CTRL-C_通常変換。
#===============================================================
sub underscore_ctrl_c_conv {
	my ($self) = @_;

	return $self->tag_bin('FF:00:色:255');
}

#===============================================================
# _XXX_その他変換。
#===============================================================
sub underscore_other_conv {
	return undef;
}

#===============================================================
# タグ変換。
#===============================================================
sub tag_conv {
	my ($self, $str, $repo) = @_;

	if (${$str} !~ s/^([^\[\]]+)\]//) {
		$$repo .= errata::out("タグ終無");
		return "";
	}
	return $self->tag_bin($1, $repo);
}

#===============================================================
# タグ文字列をバイナリ化
# input 0: タグ文字列
# return バイナリパックされたタグデータ
#===============================================================
sub tag_bin {
	my ($self, $tag_str, $repo) = @_;

	my @elems = split(/:/, $tag_str);

	my $tag_val;
	my $tag_type;
	my $tag_param;

	if (@elems < 3) {
		$$repo .= errata::out("タグ要少");
		return "";
	}
	if ($elems[0] == 3 || $elems[0] == 4) {
		my ($sub, $text, $dum);

		$sub = $self->{proc}{'_'};
		$text = $letter::TAG{"[$elems[0]:$elems[1]:"};
		if (!defined($text)) {
			$$repo .= errata::out("タグ無");
			return "";
		}

		if ($tag_str !~ s/^([\dA-F]+:[\dA-F]+):[^:]+/$1/) {
			$$repo .= errata::out("タグ異常");
		}
		if ($elems[0] == 3) {
			$self->{use_tag_list}{$tag_str}++;
		}

		return $self->$sub(\$text, \$dum);
	} elsif ($elems[0] == 1) {
		if (@elems == 4) {
			push @elems, $JOSHI_NONE;
		} elsif (@elems == 5) {
			if (!defined($JOSHI{$elems[4]})) {
				$$repo .= errata::out("助詞", $elems[4]);
				return "";
			}
			$elems[4] = $JOSHI{$elems[4]};
		} else {
			$$repo .= errata::out("タグパラ");
			return "";
		}
		$tag_str =~ s/:[^:]*$//;
	}

	if ($tag_str !~ s/^([\dA-F]+:[\dA-F]+):[^:]+/$1/) {
		$$repo .= errata::out("タグ異常");
	}
	$self->{use_tag_list}{$tag_str}++;

	# 最初は開始コード（2byte）
	$tag_val = pack('S', $TAG_CODE);

	# 次にタグ種類コード（2byte）
	$tag_type = hex($elems[0]) * 256 + hex($elems[1]);
	$tag_val .= pack('S', $tag_type);

	# 次にパラメータ数（2byte）
	my $elem_max = @elems - 3;
	$tag_val .= pack('S', $elem_max);

	# パラメータ１つあたり2byte
	for my $i (@elems[-$elem_max .. -1]) {
		$tag_param = int($i);
		if ($tag_type == 0xff00) {
			$self->{last_color} = $tag_param;
		}
		$tag_val .= pack('S', $tag_param);
	}

	return $tag_val;
}

1;
