﻿use strict;
use XML::DOM;
use utf8;

#***********************************************************
# パッケージgmm
#***********************************************************
package gmm;

our %STR_TBL = (
	'japanese' => {
		LANG_JAPAN => '日本語',
		LANG_ENGLISH => 'English',
		LANG_FRANCE => 'French',
		LANG_GERMANY => 'German',
		LANG_ITALY => 'Italian',
		LANG_SPAIN => 'Spanish',
		LANG_KOREA => 'Korean',
		WIN_CNT => 'ウィンドウコンテキスト名',
		CMT_ENG => 'コメント（英語）',
		CMT_ADD => 'コメント（追加）',
		SCROLL => 'スクロール'
	},
	'english' => {
		LANG_JAPAN => "Japanese",
		LANG_ENGLISH => 'English',
		LANG_FRANCE => 'French',
		LANG_GERMANY => 'German',
		LANG_ITALY => 'Italian',
		LANG_SPAIN => 'Spanish',
		LANG_KOREA => 'Korean',
		WIN_CNT => 'WindowContextName',
		CMT_ENG => 'Comment(English)',
		CMT_ADD => 'comment(add)',
		SCROLL => 'Scroll'
	}
);

my $parser = new XML::DOM::Parser;

#===========================================================
# オブジェクト生成。
#===========================================================
sub new {
	my ($class, $fname) = @_;
	my ($doc, $body, @row, $lang, $tmp);

	$doc = $parser->parsefile($fname);
	($tmp) = $doc->getElementsByTagName('gmml', 0);
	($body) = $tmp->getElementsByTagName('body', 0);
	@row = $body->getElementsByTagName('row', 0);
	$lang = $body->getAttribute('language');

	bless {
		doc => $doc,
		row => \@row,
		lang => $lang
	}, $class;
}

#===========================================================
# オブジェクト削除。
#===========================================================
sub delete {
	my ($self) = @_;

	$self->{doc}->dispose();
}

#===========================================================
# ファイル出力。
#===========================================================
sub print {
	my ($self, $fname) = @_;

	if (!open(OUT, ">$fname")) {
		die "error: open";
	}
	binmode(OUT, "utf8");
	$self->{doc}->printToFileHandle(\*OUT);
	close(OUT);
}

#===========================================================
# 新規イテレータ取得。
#===========================================================
sub newIter {
	my ($self, $lang) = @_;

	return gmm::iterator->new($self, $lang);
}

#***********************************************************
# パッケージgmm::iterator
#***********************************************************
package gmm::iterator;

#===========================================================
# オブジェクト生成。
#===========================================================
sub new {
	my ($class, $gmm, $lang) = @_;
	my $over = 0;

	if (@{$gmm->{row}} == 0) {
		$over = 1;
	}
	return bless {
		gmm => $gmm,
		id => 0,
		over => $over,
		cache => {},
		lang => $lang
	}, $class;
}

#===========================================================
# 次に進める。
#===========================================================
sub next {
	my ($self) = @_;

	$self->{id}++;
	$self->{cache} = {};

	if (@{$self->{gmm}{row}} <= $self->{id}) {
		$self->{over} = 1;
	}
}

#===========================================================
# 範囲オーバーしていないか？
#===========================================================
sub over {
	my ($self) = @_;

	return $self->{over};
}

#===========================================================
# テキストを取得。
#===========================================================
sub getText {
	my ($self, $lang) = @_;
	my $msg_node;

	if (!defined($lang)) {
		$lang = $self->{lang};
	}
	$msg_node = $self->get_msg_node($lang);
	if ($msg_node == 0) {
		return "";
	}
	return $msg_node->getData();
}

#===========================================================
# メッセージIDを取得。
#===========================================================
sub getMsgId {
	my ($self) = @_;
	my $row = $self->{gmm}{row}[$self->{id}];

	return $row->getAttribute('id');
}

#===========================================================
# 背景カラーを取得。
#===========================================================
sub getBackColor {
	my ($self) = @_;
	my $row = $self->{gmm}{row}[$self->{id}];

	return $row->getAttribute('back-color');
}

#===========================================================
# 背景カラーを設定。
#===========================================================
sub setBackColor {
	my ($self, $val) = @_;
	my $row = $self->{gmm}{row}[$self->{id}];

	return $row->setAttribute('back-color', $val);
}

#===========================================================
# テキストを設定。
#===========================================================
sub setText {
	my ($self, $text, $lang) = @_;
	my $msg_node;

	if (!defined($lang)) {
		$lang = $self->{lang};
	}
	$msg_node = $self->get_msg_node($lang);
	if ($msg_node == 0) {
		# 基本的にコンバータに使用するので、追加の必要はない。
		# ただし、日本語のノードがない場合には対処する必要がある。
		# 本当にあり得るかはコンバータをチェックすること。
		die "error: no msg_node";
	}
	$msg_node->setData($text);
}

#===========================================================
# 元ファイル名を取得。
#===========================================================
sub getOrgFileName {
	my ($self) = @_;

	return $self->get_attr('org_file');
}

#===========================================================
# タグリストを取得。
#===========================================================
sub getUseTagList {
	my ($self) = @_;

	return $self->get_attr('use_tag_list');
}

#===========================================================
# タグリストを設定。
#===========================================================
sub setUseTagList {
	my ($self, $val) = @_;

	return $self->set_attr('use_tag_list', $val);
}

#===========================================================
# スクロール設定を取得。
#===========================================================
sub getScroll {
	my ($self) = @_;

	return $self->get_attr($gmm::STR_TBL{$self->{gmm}{lang}}{SCROLL});
}

#===========================================================
# ウィンドウコンテキストを取得。
#===========================================================
sub getWindowContext {
	my ($self) = @_;

	return $self->get_attr($gmm::STR_TBL{$self->{gmm}{lang}}{WIN_CNT});
}

#===========================================================
# ウィンドウコンテキストを設定。
#===========================================================
sub setWindowContext {
	my ($self, $val) = @_;

	$self->set_attr($gmm::STR_TBL{$self->{gmm}{lang}}{WIN_CNT}, $val);
}

#===========================================================
# コメント（英語）を取得。
#===========================================================
sub getCommentEnglish {
	my ($self) = @_;

	$self->get_attr($gmm::STR_TBL{$self->{gmm}{lang}}{CMT_ENG});
}

#===========================================================
# コメント（英語）を設定。
#===========================================================
sub setCommentEnglish {
	my ($self, $val) = @_;

	$self->set_attr($gmm::STR_TBL{$self->{gmm}{lang}}{CMT_ENG}, $val);
}

#===========================================================
# コメント（追加）を取得。
#===========================================================
sub getCommentAdd {
	my ($self) = @_;

	$self->get_attr($gmm::STR_TBL{$self->{gmm}{lang}}{CMT_ADD});
}

#===========================================================
# コメント（追加）を設定。
#===========================================================
sub setCommentAdd {
	my ($self, $val) = @_;

	$self->set_attr($gmm::STR_TBL{$self->{gmm}{lang}}{CMT_ADD}, $val);
}

#===========================================================
# ウェイト要不要を取得。
#===========================================================
sub getNeedWait {
	my ($self) = @_;

	return $self->get_attr('need_wait');
}

#===========================================================
# ウェイト要不要を取得。
#===========================================================
sub setNeedWait {
	my ($self, $val) = @_;

	return $self->set_attr('need_wait', $val);
}

#===========================================================
# 元ファイル名を取得。
#===========================================================
sub getOrgFile {
	my ($self) = @_;

	return $self->get_attr('org_file');
}

#===========================================================
# 元ファイル名を設定。
#===========================================================
sub setOrgFile {
	my ($self, $val) = @_;

	return $self->set_attr('org_file', $val);
}

#===========================================================
# フォント種別を取得。
#===========================================================
sub getFontName {
	my ($self, $lang) = @_;
	my $lang_node;

	if (!defined($lang)) {
		$lang = $self->{lang};
	}
	$lang_node = $self->get_lang_node($lang);
	if ($lang_node == 0) {
		return "";
	}
	return $lang_node->getAttribute('font');
}

#===========================================================
# ライン制限を取得。
#===========================================================
sub getLineLimit {
	my ($self, $lang) = @_;
	my $lang_node;

	if (!defined($lang)) {
		$lang = $self->{lang};
	}
	$lang_node = $self->get_lang_node($lang);
	if ($lang_node == 0) {
		return "";
	}
	return $lang_node->getAttribute('line');
}

#===========================================================
# 全言語のテキストを取得。
#===========================================================
sub getTextAllLang {
	my ($self) = @_;
	my $gmm = $self->{gmm};
	my $row = $gmm->{row}[$self->{id}];
	my (@lang, @res);

	@lang = $row->getElementsByTagName('language', 0);
	for my $i (@lang) {
		my ($lang, $text, $text_node);

		$lang = $i->getAttribute('name');
		($text_node) = $self->get_text_nodes($i);
		if ($text_node) {
			$text = $text_node->getData();
		} else {
			$text = "";
		}
		push @res, { lang => $lang, text => $text };
	}
	return \@res;
}

#-----------------------------------------------------------
# private
#-----------------------------------------------------------

#===========================================================
# テキストノードを取得。
#===========================================================
sub get_text_nodes {
	my ($self, $parent) = @_;
	my (@node, @text_node);

	@node = $parent->getChildNodes();
	for my $i (@node) {
		if ($i->getNodeType() == ::TEXT_NODE) {
			push @text_node, $i;
		}
	}
	return @text_node;
}

#===========================================================
# メッセージノードを取得、キャッシュする。
#===========================================================
sub get_msg_node {
	my ($self, $lang) = @_;
	my $gmm = $self->{gmm};
	my $cache = $self->{cache};

	if (!defined($cache->{text}{$lang})) {
		my $lang_node = $self->get_lang_node($lang);

		if ($lang_node == 0) {
			$cache->{text}{$lang} = 0;
		} else {
			($cache->{text}{$lang}) = $self->get_text_nodes($lang_node);
			if (!defined($cache->{text}{$lang})) {
				$cache->{text}{$lang} = $lang_node->addText("");
			}
		}
	}
	return $cache->{text}{$lang};
}

#===========================================================
# 言語ノードを取得、キャッシュする。
#===========================================================
sub get_lang_node {
	my ($self, $lang) = @_;
	my $gmm = $self->{gmm};
	my $cache = $self->{cache};

	if (!defined($cache->{lang}{$lang})) {
		my $row = $gmm->{row}[$self->{id}];
		my @lang;

		@lang = $row->getElementsByTagName('language', 0);
		for my $i (@lang) {
			if ($i->getAttribute('name') ne $gmm::STR_TBL{$gmm->{lang}}{$lang}) {
				next;
			}
			$cache->{lang}{$lang} = $i;
		}
		if (!defined($cache->{lang}{$lang})) {
			$cache->{lang}{$lang} = 0;
		}
	}
	return $cache->{lang}{$lang};
}

#===========================================================
# アトリビュートを取得、キャッシュする。
#===========================================================
sub get_attr {
	my ($self, $attr) = @_;
	my $text_node;

	$text_node = $self->get_attr_node($attr);
	if ($text_node == 0) {
		return "";
	} else {
		return $text_node->getData();
	}
}

#===========================================================
# アトリビュートを設定、キャッシュする。
#===========================================================
sub set_attr {
	my ($self, $attr, $val) = @_;
	my $text_node;

	$text_node = $self->get_attr_node($attr);
	if ($text_node == 0) {
		# 無いので作る。
		my $attr_node = $self->{gmm}{doc}->createElement('attribute');
		my $row = $self->{gmm}{row}[$self->{id}];

		$attr_node->setAttribute('name', $attr);
		$text_node = $self->{gmm}{doc}->createTextNode('');
		$attr_node->appendChild($text_node);
		$row->addText("\t");
		$row->appendChild($attr_node);
		$row->addText("\n\t");
		$self->{cache}{attr}{$attr} = $text_node;
	}
	$text_node->setData($val);
}

#===========================================================
# アトリビュートのテキストノードを取得する。
#===========================================================
sub get_attr_node {
	my ($self, $attr) = @_;
	my $gmm = $self->{gmm};
	my $cache = $self->{cache};

	if (!defined($cache->{attr}{$attr})) {
		my $row = $gmm->{row}[$self->{id}];
		my @attr;

		@attr = $row->getElementsByTagName('attribute', 0);
		for my $i (@attr) {
			if ($i->getAttribute('name') ne $attr) {
				next;
			}
			($cache->{attr}{$attr}) = $self->get_text_nodes($i);
		}
		if (!defined($cache->{attr}{$attr})) {
			$cache->{attr}{$attr} = 0;
		}
	}
	return $cache->{attr}{$attr};
}

1;
