#####################################################################
# gmm_layout.pl
# gmm にウインドウの情報を設定します。
#####################################################################
use strict;
use utf8;
binmode(STDOUT,"encoding(shiftjis)");
binmode(STDERR,"encoding(shiftjis)");

require 'pm/gmm/gmm.pm';
require 'pm/gmm/row.pm';
require 'pm/xls/layout.pm';
require 'pm/util/log.pm';
require 'pm/util/file.pm';

# オプション
my $OPT_TBL =
{
	JPN	=> [\&gmm::row::getJapaneseAttribute,\&gmm::row::setJapaneseAttribute],
	ENG => [\&gmm::row::getEnglishAttribute, \&gmm::row::setEnglishAttribute ],	
	FRA => [\&gmm::row::getFrenchAttribute,  \&gmm::row::setFrenchAttribute  ],
	NOE	=> [\&gmm::row::getGermanAttribute,  \&gmm::row::setGermanAttribute  ],	
	ITA => [\&gmm::row::getItalianAttribute, \&gmm::row::setItalianAttribute ],	
	ESP => [\&gmm::row::getSpanishAttribute, \&gmm::row::setSpanishAttribute],
	KOR => [\&gmm::row::getKoreanAttribute, \&gmm::row::setKoreanAttribute]
}; 

&main();

# main
#####################################################################
sub main()
{
	# 引数
	my ($xls_file,$org_dir,$out_dir,$get,$set) = check_arg();

	# ウインドウ情報の取得
	util::log::setFileName( $xls_file );	#ログ用
	my $layout = xls::layout->new( $xls_file );

	# ディレクトリ以下のファイルのリストを取得します。
	my @file_list = ();
	util::file::getFileList( \@file_list,$org_dir );

	# 各ファイルに対して処理を行う
	foreach my $org_file (@file_list)
	{
		# ファイルが gmm でない場合は次へ
		next unless ($org_file =~ /^\Q$org_dir\E(.+\.gmm)$/);
		my $file_name = $1;
		util::log::setFileName($file_name);# ログ用

		# 出力ファイル名の設定
		my $out_file = $out_dir.$file_name;
		
		# ウインドウ情報をに設定する。
		my $gmm = gmm->new($org_file);
		set($gmm,$layout,$get,$set);

		# 変換した gmm をファイルに保存
		$gmm->print($out_file);
	}
}
####################################################################
# ウインドウの情報を設定する。
####################################################################
sub set
{
	my ($gmm,$layout,$get,$set) = @_;

	while ( my ($id,$row) = each(%{$gmm->{'row'}}) )
	{
		util::log::setID($id);	# ログ用
		my $window = $row->getWindowContext();
		my ($old_f,$old_w,$old_l) = $row->$get();
		my $old_s = $row->getScroll();
		my ($f,$w,$l,$s) = $layout->getWindowInfo($window);

		unless (($f eq $old_f) && ($w eq $old_w) && ($l eq $old_l) && ($s eq $old_s))
		{
			$row->$set($f,$w,$l);
			$row->setScroll($s);
			my $msg = "font  : ${old_f}->${f}\n"
			         ."width : ${old_w}->${w}\n"
			         ."line  : ${old_l}->${l}\n"
			         ."scroll: ${old_s}->${s}";	         
			util::log::msg('UPDATE',$msg);
		}
	}
}
#####################################################################
# 引数のチェック
#####################################################################
sub check_arg()
{
	if (@ARGV == 4)
	{		
		return
		(
			shift(@ARGV),
			shift(@ARGV).'/',
			shift(@ARGV).'/',
			check_option(shift(@ARGV))
		);
	}
	else
	{
		&help();
		exit;
	}
}
#-------------------------------------------------------------------
# 言語指定のチェック
#-------------------------------------------------------------------
sub check_option
{
	my ($opt) = @_;

	if (defined($OPT_TBL->{$opt}))
	{
		return @{$OPT_TBL->{$opt}};
	}
	
	print STDERR "error! illegal option:$opt\n";
	&help();
	exit;			
}
#-------------------------------------------------------------------
# ヘルプ表示
#-------------------------------------------------------------------
sub help()
{
	print STDERR "\n gmm_layout.pl <layout.xls> <org dir> <out dir> <lang>\n";
	print STDERR '###################################################################'."\n";	
	print STDERR '# layout.xls を org dir 内のgmmに反映して out dir にはきだします。#'."\n";
	print STDERR '# <lang>: JPN/ENG/FRA/NOE/ITA/ESP/KOR                             #'."\n";
	print STDERR '###################################################################'."\n";
}
