#####################################################################
# gmm_merge.pl
# append gmm のデータを baes gmm にマージします。
#
# append dir 以下の gmm の値を base dir にマージする。 
# append dir と base dir の構成は全く同じである必要がある。
# 結果を out dir に出力する。
#####################################################################
use strict;
use utf8;

require 'pm/gmm/gmm.pm';
require 'pm/gmm/row.pm';
require 'pm/gmm/merge.pm';
require 'pm/util/log.pm';
require 'pm/util/file.pm';

# オプション
my $OPT_TBL =
{
	WIN_CNT	=>'WindowContext',	# ウインドウコンテキスト
	MSG_JPN	=>'JapaneseMessage',# 日本語メッセージ	
	CMT_JPN	=>'JapaneseComment',# 日本語コメント
	MSG_ENG =>'EnglishMessage',	# 英語メッセージ	
	CMT_ENG	=>'EnglishComment',	# 英語コメント
	MSG_FRA =>'FrenchMessage',	# フランス語メッセージ	
	CMT_FRA	=>'FrenchComment',	# フランス語コメント	
	MSG_NOE	=>'GermanMessage',	# ドイツ語メッセージ	
	CMT_NOE	=>'GermanComment',	# ドイツ語コメント
	MSG_ITA =>'ItalianMessage',	# イタリア語メッセージ	
	CMT_ITA	=>'ItalianComment',	# イタリア語コメント
	MSG_ESP =>'SpanishMessage',	# スペイン語メッセージ	
	CMT_ESP	=>'SpanishComment',	# スペイン語コメント
	MSG_KOR =>'KoreanMessage'	# 韓国語メッセージ	
};

&main();

# main
#####################################################################
sub main()
{
	# 引数
	my ($append_dir,$base_dir,$out_dir,$opt) = check_arg();

	# ディレクトリ以下のファイルのリストを取得します。
	my @file_list = ();
	util::file::getFileList( \@file_list,$append_dir );

	# 各ファイルに対して処理を行う
	foreach my $append_file ( @file_list )
	{
		# ファイルが gmm でない場合は次へ
		next unless ( $append_file =~ /^\Q$append_dir\E(.+\.gmm)$/ );
		my $file_name = $1;
		util::log::setFileName($file_name);# ログ用
		
		# ファイル名の設定
		my $base_file = $base_dir.$file_name;	# ベースファイル名の設定
		my $out_file  = $out_dir.$file_name;	# 出力ファイル名の設定

		# マージ
		my $append_gmm = gmm->new($append_file);
		my $base_gmm = gmm->new($base_file);
		merge($append_gmm,$base_gmm,$opt);

		# マージした gmm をファイルに保存
		$base_gmm->print($out_file);
	}
}
#-------------------------------------------------------------------
# マージオプションのチェック
#-------------------------------------------------------------------
sub merge
{
	my ($append_gmm,$base_gmm,$opt) = @_;

	my $append_row = $append_gmm->{'row'};
	my $base_row = $base_gmm->{'row'};
	
	while ( my ($id,$append) = each(%$append_row) )
	{
		util::log::setMsgId($id);	# ログ用

		unless ( defined($base_row->{$id}) )
		{
			# base に存在しない id の場合
			util::log::msg('idが存在しません。');
			next;
		}
		
		my $base = $base_row->{$id};
		foreach my $val (values %$opt)
		{
			gmm::merge::text($val,$append,$base);
		}	
	}
}
#-------------------------------------------------------------------
# 引数のチェック
#-------------------------------------------------------------------
sub check_arg
{	
	if (@ARGV >= 3)
	{
		return
		(
			$ARGV[0].'/',
			$ARGV[1].'/',
			$ARGV[2].'/',
			check_option($ARGV[3])
		);
	}
	else
	{
		&help();
		exit;
	}
}
#-------------------------------------------------------------------
# マージオプションのチェック
#-------------------------------------------------------------------
sub check_option
{
	my ($arg) = @_;
	
	my %opt;
	foreach my $val ( split(/\+/,$arg) )
	{
		if (exists($OPT_TBL->{$val}))
		{
			$opt{$val} = $OPT_TBL->{$val};
		}
		else
		{
			print STDERR "error! illegal option:$val\n";
			&help();
			exit;			
		}
	}
	return \%opt;
}
#-------------------------------------------------------------------
# ヘルプ表示
#-------------------------------------------------------------------
sub help
{
	print STDERR "gmm_merge.pl <append dir> <base dir> <out dir> opt1+opt2 ...  \n";
	print STDERR "option\n";
	print STDERR "\tWIN_CNT : Window Context\n";
	print STDERR "\tMSG_JPN : Japanese Message\n";	
	print STDERR "\tCMT_JPN : Japanese Comment\n";
	print STDERR "\tMSG_ENG : English Message\n";	
	print STDERR "\tCMT_ENG : English Comment\n";
	print STDERR "\tMSG_FRA : French Message\n";	
	print STDERR "\tCMT_FRA : French Comment\n";
	print STDERR "\tMSG_NOE : German Message\n";	
	print STDERR "\tCMT_NOE : German Comment\n";
	print STDERR "\tMSG_ITA : Itarian Message\n";	
	print STDERR "\tCMT_ITA : Itarian Comment\n";
	print STDERR "\tMSG_ESP : Spanish Message\n";	
	print STDERR "\tCMT_ESP : Spanish Comment\n";
	print STDERR "\tMSG_KOR : Korean Message\n";	

}

