/*---------------------------------------------------------------------------*
  Project:  DP WiFi Library
  File:     dpw_bt.c

  Copyright 2003-2006 Nintendo.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $NoKeywords: $
 *---------------------------------------------------------------------------*/

/*! @file
	@brief	DP WiFi Battle Tower Cu
	
	@author	kitayama(kitayama_shigetoshi@nintendo.co.jp)
	@author	Yamaguchi Ryo(yamaguchi_ryo@nintendo.co.jp)
	
*/

#ifdef _NITRO
#include <nitroWiFi/socl.h>
#endif

#include "include/libdpw/dpw_bt.h"
#include "include/libdpw/dpwi_session.h"
#include "include/libdpw/dpwi_assert.h"

/*-----------------------------------------------------------------------*
					^E萔錾
 *-----------------------------------------------------------------------*/

#ifdef DPW_SERVER_PUBLIC
#define BT_SERVER_URL		"http://gamestats2.gs.nintendowifi.net/pokemondpds/"
#else
#define BT_SERVER_URL		"http://sdkdev.gamespy.com/games/pokemondpds/web/"
#endif

#define BT_URL_GETROOMNUM	(const u8*)(BT_SERVER_URL "battletower/roomnum.asp")
#define BT_URL_DOWNLOADROOM	(const u8*)(BT_SERVER_URL "battletower/download.asp")
#define BT_URL_UPLOADPLAYER	(const u8*)(BT_SERVER_URL "battletower/upload.asp")
#define BT_URL_GETSERVERSTATE (const u8*)(BT_SERVER_URL "battletower/info.asp")
#define BT_URL_SETPROFILE (const u8*)(BT_SERVER_URL "common/setProfile.asp")
#define BT_URL_INIT			(BT_SERVER_URL "battletower/init.asp")
#define BT_URL_UPDATE 		(BT_SERVER_URL "admin/bt_update.asp")

#ifdef _NITRO
// \̂z̃TCYƂȂĂ邩`FbN
SDK_COMPILER_ASSERT(sizeof(Dpw_Bt_Leader) == 34);
SDK_COMPILER_ASSERT(sizeof(Dpw_Bt_Player) == 228);
SDK_COMPILER_ASSERT(sizeof(Dpw_Bt_Room) == sizeof(Dpw_Bt_Player) * 7 + sizeof(Dpw_Bt_Leader) * 30);
#endif

#define DB_HTTP_FLAG_NOT_COMPLETED	(-1)

#define BT_RESPONSE_SUCCESS			(1)
#define BT_RESPONSE_ILLIGAL_REQUEST	(2)
#define BT_RESPONSE_STOP_SERVICE	(3)
#define BT_RESPONSE_SERVER_FULL		(4)
#define BT_RESPONSE_SERVER_TIMEOUT	(5)

typedef enum {
	DPWi_BT_NOT_INIT,
	DPWi_BT_NORMAL,
	DPWi_BT_RROCESS_GETROOMNUM,
	DPWi_BT_CANCEL_GETROOMNUM,
	DPWi_BT_RROCESS_DOWNLOADROOM,
	DPWi_BT_CANCEL_DOWNLOADROOM,
	DPWi_BT_RROCESS_UPLOADPLAYER,
	DPWi_BT_CANCEL_UPLOADPLAYER,
	DPWi_BT_RROCESS_GETSERVERSTATE,
	DPWi_BT_CANCEL_GETSERVERSTATE,
	DPWi_BT_RROCESS_SETPROFILE,
	DPWi_BT_CANCEL_SETPROFILE,
	DPWi_BT_END_ASYNC
} DpwiBtState;

typedef struct {
	DpwiBtState state;		// Cȕ
	s32 last_result;		// Ōɍs񓯊̌
	s32 pid;				// gpPID
	u64 friend_key;			// thL[
	u8 send_buf[sizeof(Dpw_Bt_Player) + 11];	// Mobt@
	u8 recv_buf[2];			// Mobt@
	u8* user_recv_buf;		// [U[^ꂽMobt@
} DpwiBtCtrl;

/*-----------------------------------------------------------------------*
					֐vg^Cv錾
 *-----------------------------------------------------------------------*/

static BOOL Dpwi_Bt_CallSessionRequest(const u8* url, const void* data, int len, void* resbuf, int ressize );
static DpwBtError Dpwi_Bt_HandleCommonError(DpwiHttpError error);
static void Dpwi_Db_GhttpCopleteCallback(const char* buf, int len, DWCGHTTPResult result, void* param);

/*-----------------------------------------------------------------------*
					O[oϐ`
 *-----------------------------------------------------------------------*/

static DpwiBtCtrl dpw_bt;
static int db_ghttp_flag;

/*-----------------------------------------------------------------------*
					O[o֐`
 *-----------------------------------------------------------------------*/

/*!
	Dpw_Bt Cu܂BSĂDpw_Bt ̕t֐sOɃR[܂B
	
	GSvt@CID́A DWC_LoginAsync() ֐̃R[obNŎ擾łlŁAT[o[ňӂɃNCAg
	𔻕ʂ邽߂̒lłB
	
	̊֐R[ۂ́ADWC CũC^[lbgڑ֐Ńlbg[NɐڑĂsĂ
	BAOCsĂKv͂܂B
	
	@param[in] pid	GSvt@CID
	@param[in] friend_key	̃thL[BDWC_CreateFriendKey() Ŏ擾ł܂B
*/
void Dpw_Bt_Init(s32 pid, u64 friend_key) {

	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NOT_INIT, "dpw bt is already initialized.\n");

	dpw_bt.state = DPWi_BT_NORMAL;
	dpw_bt.last_result = 0;
	dpw_bt.pid = pid;
	dpw_bt.friend_key = friend_key;
}

/*!
	Q[t[ĂтAi߂܂B
	
	֐̂߁Å֐͏ɌĂяoƂł܂BԂł͉s܂B
*/
void Dpw_Bt_Main(void) {
	
	switch (dpw_bt.state) {
	case DPWi_BT_NOT_INIT:
		break;
	case DPWi_BT_NORMAL:
		break;
	case DPWi_BT_RROCESS_GETROOMNUM:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			dpw_bt.last_result = Dpwi_Bt_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			if (DpwiGetResponseSize() > 0) {
				dpw_bt.last_result = dpw_bt.recv_buf[0];
			} else {
				dpw_bt.last_result = DPW_BT_ERROR_SERVER_TIMEOUT;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_BT_RROCESS_DOWNLOADROOM:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			dpw_bt.last_result = Dpwi_Bt_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			if (DpwiGetResponseSize() == sizeof(Dpw_Bt_Room)) {
				dpw_bt.last_result = 0;
			} else {
				switch (dpw_bt.user_recv_buf[0]) {
				case BT_RESPONSE_ILLIGAL_REQUEST:
					dpw_bt.last_result = DPW_BT_ERROR_ILLIGAL_REQUEST;
					break;
				case BT_RESPONSE_SERVER_TIMEOUT:
					dpw_bt.last_result = DPW_BT_ERROR_SERVER_TIMEOUT;
					break;
				default:
					DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_bt.user_recv_buf[0]);
					dpw_bt.last_result = DPW_BT_ERROR_FATAL;
					break;
				}
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_BT_RROCESS_UPLOADPLAYER:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			dpw_bt.last_result = Dpwi_Bt_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			switch (dpw_bt.recv_buf[0]) {
			case BT_RESPONSE_SUCCESS:
				dpw_bt.last_result = 0;
				break;
			case BT_RESPONSE_ILLIGAL_REQUEST:
				dpw_bt.last_result = DPW_BT_ERROR_ILLIGAL_REQUEST;
				break;
			case BT_RESPONSE_SERVER_FULL:
				dpw_bt.last_result = DPW_BT_ERROR_SERVER_FULL;
				break;
			case BT_RESPONSE_SERVER_TIMEOUT:
				dpw_bt.last_result = DPW_BT_ERROR_SERVER_TIMEOUT;
				break;
			default:
				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_bt.recv_buf[0]);
				dpw_bt.last_result = DPW_BT_ERROR_FATAL;
				break;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_BT_RROCESS_GETSERVERSTATE:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			dpw_bt.last_result = Dpwi_Bt_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			switch (dpw_bt.recv_buf[0]) {
			case BT_RESPONSE_SUCCESS:
				dpw_bt.last_result = DPW_BT_STATUS_SERVER_OK;
				break;
			case BT_RESPONSE_STOP_SERVICE:
				dpw_bt.last_result = DPW_BT_STATUS_SERVER_STOP_SERVICE;
				break;
			case BT_RESPONSE_SERVER_FULL:
				dpw_bt.last_result = DPW_BT_STATUS_SERVER_FULL;
				break;
			case BT_RESPONSE_SERVER_TIMEOUT:
				dpw_bt.last_result = DPW_BT_ERROR_SERVER_TIMEOUT;
				break;
			default:
				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_bt.recv_buf[0]);
				dpw_bt.last_result = DPW_BT_ERROR_FATAL;
				break;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_BT_RROCESS_SETPROFILE:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			dpw_bt.last_result = Dpwi_Bt_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_bt.state = DPWi_BT_END_ASYNC;
			if (DpwiGetResponseSize() == sizeof(Dpw_Common_ProfileResult)) {
				dpw_bt.last_result = 0;
			} else {
    			switch (dpw_bt.recv_buf[0]) {
    			case BT_RESPONSE_SUCCESS:
    				dpw_bt.last_result = DPW_BT_STATUS_SERVER_OK;
    				break;
    			case BT_RESPONSE_STOP_SERVICE:
    				dpw_bt.last_result = DPW_BT_STATUS_SERVER_STOP_SERVICE;
    				break;
    			case BT_RESPONSE_SERVER_FULL:
    				dpw_bt.last_result = DPW_BT_STATUS_SERVER_FULL;
    				break;
    			case BT_RESPONSE_SERVER_TIMEOUT:
    				dpw_bt.last_result = DPW_BT_ERROR_SERVER_TIMEOUT;
    				break;
    			default:
    				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_bt.recv_buf[0]);
    				dpw_bt.last_result = DPW_BT_ERROR_FATAL;
    				break;
    			}
            }
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_BT_CANCEL_GETROOMNUM:
	case DPWi_BT_CANCEL_DOWNLOADROOM:
	case DPWi_BT_CANCEL_UPLOADPLAYER:
	case DPWi_BT_CANCEL_GETSERVERSTATE:
	case DPWi_BT_CANCEL_SETPROFILE:
		dpw_bt.state = DPWi_BT_END_ASYNC;
		dpw_bt.last_result = DPW_BT_ERROR_CANCEL;
		DpwiSessionShutdown();
		break;
	case DPWi_BT_END_ASYNC:
		break;
	}
}

/*!
	̊JȂǏIs܂B
	
	̊֐́A񓯊sĂƂɂ̓R[邱Ƃł܂BDpw_Bt_IsAsyncEnd()Ŕ񓯊
	̏ImFĂR[ĂB
*/
void Dpw_Bt_End(void) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NORMAL, "async process is running.\n");
	
	dpw_bt.state = DPWi_BT_NOT_INIT;
}

/*!
	Dpw_BtCu̔񓯊IĂ邩ۂԂ܂B
	
	֐ɁuAsyncvt֐R[ۂ́Å֐IɃR[āAImFĂB
	
	@retval TRUE	񓯊IĂ
	@retval FALSE	񓯊IĂȂ
*/
BOOL Dpw_Bt_IsAsyncEnd(void) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	
	switch (dpw_bt.state) {
	case DPWi_BT_NORMAL:
		return TRUE;
		break;
	case DPWi_BT_END_ASYNC:
		dpw_bt.state = DPWi_BT_NORMAL;
		return TRUE;
		break;
	}

	return FALSE;
}

/*!
	O̔񓯊̌ʂ擾܂B
	Bpw_Bt_IsAsyncEnd() ŏImFĂĂłB
	
	񓯊sꍇ́AK̊֐ŌʁiɃG[NĂȂjmFĂB@n
	G[̎ނ͔񓯊ɂĈقȂ܂Aȉ̃G[͋ʂłB
	
	@li ::DPW_BT_ERROR_SERVER_TIMEOUT
	@li ::DPW_BT_ERROR_CANCEL
	@li ::DPW_BT_ERROR_FATAL
	@li ::DPW_BT_ERROR_DISCONNECTED
	@li ::DPW_BT_ERROR_FAILURE
	
	@return	񓯊̌ʁB̒l̓G[ADpwBtError̒lԂ܂B
*/
s32 Dpw_Bt_GetAsyncResult(void) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	
	return dpw_bt.last_result;
}

/*!
	݃I[vĂ郋[̎擾Jn܂B
	
	擾oꍇ Dpw_Bt_GetAsyncResult() Ń[Ԃ܂B
	G[Nꍇ́Aʂ̃G[̂ꂩԂ܂B
	
	@param[in] rank	N
*/
void Dpw_Bt_GetRoomNumAsync(s32 rank) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NORMAL, "async process is already running.\n");
	DPW_MINMAX_TASSERT(rank, 1, DPW_BT_RANK_NUM);
	
	// ZbV
	DpwiSessionInitialize();

	dpw_bt.send_buf[0] = (u8)(rank - 1);	// T[o[0x[XȂ̂ŁA킹
	
	// ʐMJn
	if (Dpwi_Bt_CallSessionRequest(BT_URL_GETROOMNUM, dpw_bt.send_buf, 1, dpw_bt.recv_buf, 2)) {
		dpw_bt.state = DPWi_BT_RROCESS_GETROOMNUM;
	} else {
		dpw_bt.state = DPWi_BT_END_ASYNC;
		dpw_bt.last_result = DPW_BT_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	[f[^_E[h܂B
	
	ꍇA Dpw_Bt_GetAsyncResult() 0Ԃ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_BT_ERROR_ILLIGAL_REQUEST	F [ԍ̎w肪Ă܂B
	
	@param[in] rank	N
	@param[in] roomNo [ԍ
	@param[out] roomData _E[h[f[^L^obt@
*/
void Dpw_Bt_DownloadRoomAsync(s32 rank, s32 roomNo, Dpw_Bt_Room* roomData) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NORMAL, "async process is already running.\n");
	DPW_MINMAX_TASSERT(rank, 1, DPW_BT_RANK_NUM);
	DPW_MINMAX_TASSERT(roomNo, 1, DPW_BT_ROOM_NUM_MAX);
	DPW_NULL_TASSERT(roomData);
	
	dpw_bt.user_recv_buf = (u8*)roomData;
	
	// ZbV
	DpwiSessionInitialize();

	dpw_bt.send_buf[0] = (u8)(rank - 1);	// T[o[0x[XȂ̂ŁA킹
	dpw_bt.send_buf[1] = (u8)(roomNo - 1);	// T[o[0x[XȂ̂ŁA킹
	
	// ʐMJn
	if (Dpwi_Bt_CallSessionRequest(BT_URL_DOWNLOADROOM, dpw_bt.send_buf, 2, roomData, sizeof(Dpw_Bt_Room))) {
		dpw_bt.state = DPWi_BT_RROCESS_DOWNLOADROOM;
	} else {
		dpw_bt.state = DPWi_BT_END_ASYNC;
		dpw_bt.last_result = DPW_BT_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	vC[f[^Abv[h܂B
	
	ꍇA Dpw_Bt_GetAsyncResult() 0Ԃ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_BT_ERROR_SERVER_FULL		F T[o[̗eʃI[o[łB
	@li ::DPW_BT_ERROR_ILLIGAL_REQUEST	F [ԍ̎w肪Ă܂B
	
	@param[in] rank	N
	@param[in] roomNo	[ԍ
	@param[in] win	vC[̏
	@param[in] player	vC[f[^B֐ŃRs[܂̂ŁAÓIɕێKv͂܂B
*/
void Dpw_Bt_UploadPlayerAsync(s32 rank, s32 roomNo, s32 win, const Dpw_Bt_Player* player) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NORMAL, "async process is already running.\n");
	DPW_MINMAX_TASSERT(rank, 1, DPW_BT_RANK_NUM);
	DPW_MINMAX_TASSERT(roomNo, 1, DPW_BT_ROOM_NUM_MAX);
	DPW_MINMAX_TASSERT(win, 0, 7);
	DPW_NULL_TASSERT(player);
	
	// ZbV
	DpwiSessionInitialize();

	memcpy(&dpw_bt.send_buf[0], player, sizeof(Dpw_Bt_Player));
	dpw_bt.send_buf[sizeof(Dpw_Bt_Player)] = (u8)(rank - 1);	// T[o[0x[XȂ̂ŁA킹
	dpw_bt.send_buf[sizeof(Dpw_Bt_Player) + 1] = (u8)(roomNo - 1);	// T[o[0x[XȂ̂ŁA킹
	dpw_bt.send_buf[sizeof(Dpw_Bt_Player) + 2] = (u8)win;
	memcpy(&dpw_bt.send_buf[sizeof(Dpw_Bt_Player) + 3], &dpw_bt.friend_key, 8);
	
	// ʐMJn
	if (Dpwi_Bt_CallSessionRequest(BT_URL_UPLOADPLAYER, dpw_bt.send_buf, sizeof(Dpw_Bt_Player) + 11, dpw_bt.recv_buf, 2)) {
		dpw_bt.state = DPWi_BT_RROCESS_UPLOADPLAYER;
	} else {
		dpw_bt.state = DPWi_BT_END_ASYNC;
		dpw_bt.last_result = DPW_BT_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	Dpw_Bt CuŌݍsĂ񓯊LZ܂B
	
	LZꍇA Dpw_Bt_GetAsyncResult() ̕Ԓl DPW_BT_ERROR_CANCEL ƂȂ܂B
	
	ɒʐMIĂ邱Ƃ܂̂ŁALZ͏ɐƂ͌܂B
	LZłǂ́A Dpw_Bt_GetAsyncResult() ̕Ԓl DPW_BT_ERROR_CANCEL ƂȂĂ邩ŔfĂB
	
	Dpw_Bt_CancelAsync()̂񓯊łALZ邱Ƃ͂ł܂B
*/
void Dpw_Bt_CancelAsync(void) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	
	switch (dpw_bt.state) {
	case DPWi_BT_RROCESS_GETROOMNUM:
		DpwiSessionCancel();
		dpw_bt.state = DPWi_BT_CANCEL_GETROOMNUM;
		break;
	case DPWi_BT_RROCESS_DOWNLOADROOM:
		DpwiSessionCancel();
		dpw_bt.state = DPWi_BT_CANCEL_DOWNLOADROOM;
		break;
	case DPWi_BT_RROCESS_UPLOADPLAYER:
		DpwiSessionCancel();
		dpw_bt.state = DPWi_BT_CANCEL_UPLOADPLAYER;
		break;
	case DPWi_BT_RROCESS_GETSERVERSTATE:
		DpwiSessionCancel();
		dpw_bt.state = DPWi_BT_CANCEL_GETSERVERSTATE;
		break;
	case DPWi_BT_RROCESS_SETPROFILE:
		DpwiSessionCancel();
		dpw_bt.state = DPWi_BT_CANCEL_SETPROFILE;
		break;
	}
}

/*!
	T[o[Ԃ̒Jn܂B
	
	T[o[ƐɒʐMłꍇ́A Dpw_Bt_GetAsyncResult()  DpwBtServerStatus ̒lԂ܂B
	
	ɒʐMłȂꍇ́A ʂ̃G[̂ꂩԂ܂B
*/
void Dpw_Bt_GetServerStateAsync(void) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NORMAL, "async process is already running.\n");
	
	// ZbV
	DpwiSessionInitialize();

	// ʐMJn
	if (Dpwi_Bt_CallSessionRequest(BT_URL_GETSERVERSTATE, dpw_bt.send_buf, 0, dpw_bt.recv_buf, 2)) {
		dpw_bt.state = DPWi_BT_RROCESS_GETSERVERSTATE;
	} else {
		dpw_bt.state = DPWi_BT_END_ASYNC;
		dpw_bt.last_result = DPW_BT_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	lo^鏈Jn܂B
	
	ꍇ́A Dpw_Bt_GetAsyncResult() 0ԂAw肵resultp[^codeoDPW_PROFILE_RESULTCODE_SUCCESSԂ܂B
    
	ɒʐMɋNG[Nꍇ́ADpw_Bt_GetAsyncResult()BT̋ʂ̃G[Ԃ܂B
    lo^ɊւG[Nꍇ͎w肵resultp[^codeoDPW_PROFILE_RESULTCODE_SUCCESSȊO̒lZbg܂B
	
	@param[in] data	Abv[hf[^B֐ŃRs[܂̂ŁAÓIɕێKv͂܂B
                    MACAhX̓CuŊi[̂ŃZbgKv͂܂B
                    DP؍łłversion, language, countryCode, localCode, playerName, playerId, flaĝ݂ZbgĂBȊO0Ŗ߂ĂB
	@param[out] result T[õX|XBcodeoɓo^Ɋւ錋ʂZbgAmailAddrAuthResultɃ[AhXo^Ɋւ錋ʂZbg܂BAmailAddrAuthResultDP؍łł͕K0ɂȂ܂B
*/
void Dpw_Bt_SetProfileAsync(const Dpw_Common_Profile* data, Dpw_Common_ProfileResult* result) {
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NORMAL, "async process is already running.\n");
	DPW_NULL_TASSERT(data);
	
    DPW_TASSERTMSG(sizeof(Dpw_Common_Profile) <= sizeof(dpw_bt.send_buf), "Internal error: dpw send buf is too small.\n");
    
    // MacAhXZbg
	OS_GetMacAddress((u8*)data->macAddr);
    
	memcpy(dpw_bt.send_buf, data, sizeof(Dpw_Common_Profile));
	dpw_bt.user_recv_buf = (u8*)result;
	
	// ZbV
	DpwiSessionInitialize();

	// ʐMJn
	if (Dpwi_Bt_CallSessionRequest(BT_URL_SETPROFILE, dpw_bt.send_buf, sizeof(Dpw_Common_Profile), dpw_bt.user_recv_buf, sizeof(Dpw_Common_ProfileResult))) {
		dpw_bt.state = DPWi_BT_RROCESS_SETPROFILE;
	} else {
		dpw_bt.state = DPWi_BT_END_ASYNC;
		dpw_bt.last_result = DPW_BT_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	og^[̃T[oԂ܂B
	ŃubN܂B
	
	̊֐͔p~܂BDpw Admin ToolgpB
*/
BOOL Dpw_Bt_Db_InitServer(void) {
#if 0
	
	int ret;
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NORMAL, "async process is already running.\n");
	
	db_ghttp_flag = DB_HTTP_FLAG_NOT_COMPLETED;
	
	(void)DWC_InitGHTTP(NULL);
	
	ret = DWC_GetGHTTPData(BT_URL_INIT, Dpwi_Db_GhttpCopleteCallback, NULL);
	
	while (db_ghttp_flag == DB_HTTP_FLAG_NOT_COMPLETED) {
		
		if (!DWC_ProcessGHTTP()) {
			DWC_ClearError();
			return FALSE;
		}
		
		OS_Sleep(16);
	}
	
	DWC_ShutdownGHTTP();

	if (db_ghttp_flag == DWC_GHTTP_SUCCESS) {
		return TRUE;
	} else {
		DWC_ClearError();
		return FALSE;
	}
#else
    return FALSE;
#endif
}

/*!
	og^[̃T[ö̃Abvf[gs܂B
	ŃubN܂B
	
	̊֐͔p~܂BDpw Admin ToolgpB
*/
BOOL Dpw_Bt_Db_UpdateServer(void) {
#if 0
	
	int ret;
	
	DPW_TASSERTMSG(dpw_bt.state != DPWi_BT_NOT_INIT, "dpw bt is not initialized.\n");
	DPW_TASSERTMSG(dpw_bt.state == DPWi_BT_NORMAL, "async process is already running.\n");
	
	db_ghttp_flag = DB_HTTP_FLAG_NOT_COMPLETED;
	
	(void)DWC_InitGHTTP(NULL);
	
	ret = DWC_GetGHTTPData(BT_URL_UPDATE, Dpwi_Db_GhttpCopleteCallback, NULL);
	
	while (db_ghttp_flag == DB_HTTP_FLAG_NOT_COMPLETED) {
		
		if (!DWC_ProcessGHTTP()) {
			DWC_ClearError();
			return FALSE;
		}
		
		OS_Sleep(16);
	}
	
	DWC_ShutdownGHTTP();

	if (db_ghttp_flag == DWC_GHTTP_SUCCESS) {
		return TRUE;
	} else {
		DWC_ClearError();
		return FALSE;
	}
#else
    return FALSE;
#endif
}

/*-----------------------------------------------------------------------*
					[J֐`
 *-----------------------------------------------------------------------*/

static BOOL Dpwi_Bt_CallSessionRequest(const u8* url, const void* data, int len, void* resbuf, int ressize) {
	
	OS_TPrintf("[DPW BT] Connecting to %s.\n", url);
	switch (DpwiSessionRequest(url, dpw_bt.pid, data, len, (u8*)resbuf, ressize)) {
	case DPWI_COMMON_SESSION_SUCCESS:				// I
		return TRUE;
		break;
	case DPWI_COMMON_SESSION_ERROR_NOTINITIALIZED:	// 
		DPW_TASSERTMSG(FALSE, "common session not initialized.");
		break;
	case DPWI_COMMON_SESSION_ERROR_NOMEMORY:		// [s
		DPW_TASSERTMSG(FALSE, "common session memory shortage.");
		break;
	}
	
	return FALSE;
}

static DpwBtError Dpwi_Bt_HandleCommonError(DpwiHttpError error) {
	
	DpwBtError ret;
	
	OS_TPrintf("[DPW BT] ghttp error: %d\n", error);
	
	switch (error) {
	case DPWI_COMMON_SESSION_ERROR_IN_ERROR:			// G[ 
	case DPWI_COMMON_SESSION_ERROR_INVALID_POST:		// ȑM 
	case DPWI_COMMON_SESSION_ERROR_INVALID_FILE_NAME:	// ȃt@C 
	case DPWI_COMMON_SESSION_ERROR_INVALID_BUFFER_SIZE:	// ȃobt@TCY 
	case DPWI_COMMON_SESSION_ERROR_INVALID_URL:			// URL
	case DPWI_COMMON_SESSION_ERROR_UNSPECIFIED_ERROR:	// ̑̃G[
	case DPWI_COMMON_SESSION_ERROR_BUFFER_OVERFLOW:		// ꂽobt@邽߁At@C̎擾s 
	case DPWI_COMMON_SESSION_ERROR_PARSE_URL_FAILED:	// URL̓G[ 
//	case DPWI_COMMON_SESSION_ERROR_ENCRYPTION_ERROR:	// ÍG[ 
	case DPWI_COMMON_SESSION_ERROR_FILE_TOO_BIG:		// t@C傫邽߃_E[hs\ 
	case DPWI_COMMON_SESSION_ERROR_FILE_INCOMPLETE:		// _E[h̒f 
	case DPWI_COMMON_SESSION_ERROR_FILE_WRITE_FAILED:	// [Jt@Cւ̏݃G[ 
	case DPWI_COMMON_SESSION_ERROR_FILE_READ_FAILED:	// [Jt@C̓ǂݏoG[ 
	case DPWI_COMMON_SESSION_ERROR_BAD_RESPONSE:		// HTTPT[õX|X̉̓G[ 
	case DPWI_COMMON_SESSION_ERROR_BUFFER_OVER:			// COMMONw: Mobt@I[o[
		DPW_TASSERTMSG(FALSE, "library internal error. please contact author.");
		ret = DPW_BT_ERROR_FATAL;
		break;
	case DPWI_COMMON_SESSION_ERROR_INSUFFICIENT_MEMORY:	// s 
	case DPWI_COMMON_SESSION_ERROR_OUT_OF_MEMORY:		// 蓖Ďs 
	case DPWI_COMMON_SESSION_ERROR_MEMORY_ERROR: 		// 蓖Ďs 
		DPW_TASSERTMSG(FALSE, "common session memory shortage.");
		ret = DPW_BT_ERROR_FATAL;
		break;
	case DPWI_COMMON_SESSION_ERROR_HOST_LOOKUP_FAILED:	// zXgs 
		OS_TPrintf("[DPW BT] dns lookup failed.\n");
		ret = DPW_BT_ERROR_FAILURE;
		break;
	case DPWI_COMMON_SESSION_ERROR_SOCKET_FAILED:		// \Pbg̍쐬AAǂݏoAݎs 
	case DPWI_COMMON_SESSION_ERROR_CONNECT_FAILED:		// HTTPT[oւ̐ڑs 
		OS_TPrintf("[DPW BT] socket error.\n");
		ret = DPW_BT_ERROR_FAILURE;
		break;
	case DPWI_COMMON_SESSION_ERROR_UNAUTHORIZED:		// t@C擾 
	case DPWI_COMMON_SESSION_ERROR_FORBIDDEN:			// HTTPT[õt@CM 
	case DPWI_COMMON_SESSION_ERROR_FILE_NOT_FOUND:		// HTTPT[õt@Cs 
	case DPWI_COMMON_SESSION_ERROR_SERVER_ERROR:		// HTTPT[oG[
	case DPWI_COMMON_SESSION_ERROR_CHECKSUM:			// COMMONw: `FbNT̕sv
	case DPWI_COMMON_SESSION_ERROR_PID:					// COMMONw: PID̕sv
	case DPWI_COMMON_SESSION_ERROR_DATA_LENGTH: 		// COMMONw: f[^̒s
	case DPWI_COMMON_SESSION_ERROR_TOKEN_NOT_FOUND:		// COMMONw: g[NȂ
	case DPWI_COMMON_SESSION_ERROR_INCORRECT_HASH:		// COMMONw: nbVȂ
		OS_TPrintf("[DPW BT] server internal error.  please contact server administrator.\n");
		ret = DPW_BT_ERROR_SERVER_TIMEOUT;
		break;
	case DPWI_COMMON_SESSION_ERROR_REQUEST_REJECTED:	// HTTPT[õNGXg 
	case DPWI_COMMON_SESSION_ERROR_TOKEN_EXPIRED:		// COMMONw: g[N̗L؂
		OS_TPrintf("[DPW BT] server is now heavy.\n");
		ret = DPW_BT_ERROR_SERVER_TIMEOUT;
		break;
	default:
		DPW_TASSERTMSG(FALSE, "illigal ghttp error: %d", error);
		ret = DPW_BT_ERROR_FATAL;
		break;
	}
	
	// FATALG[łȂꍇ
	if (ret != DPW_BT_ERROR_FATAL) {
#ifdef _NITRO
		// NitroWiFĩC[Ŗ؂ĂȂ`FbN
		if (WCM_GetPhase() != WCM_PHASE_DCF) {
			OS_TPrintf("[DPW BT] disconnected from access point.\n");
			ret = DPW_BT_ERROR_DISCONNECTED;
		}
#endif
		// DWC̃G[ԂNA
		DWC_ClearError();
	}
	
	return ret;
}

#ifndef SDK_FINALROM
static void Dpwi_Db_GhttpCopleteCallback(const char* buf, int len, DWCGHTTPResult result, void* param) {
#pragma unused(buf, len, param)
	
	db_ghttp_flag = result;
}
#endif
