/*---------------------------------------------------------------------------*
  Project:  DP WiFi Library
  File:     dpw_tr.c

  Copyright 2003-2006 Nintendo.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $NoKeywords: $
 *---------------------------------------------------------------------------*/

/*! @file
	@brief	DP WiFi Trade Cu
	
	@author	kitayama(kitayama_shigetoshi@nintendo.co.jp)
	@author	Yamaguchi Ryo(yamaguchi_ryo@nintendo.co.jp)
	
*/

#ifdef _NITRO
#include <nitroWiFi/socl.h>
#endif

#include "include/libdpw/dpw_tr.h"
#include "include/libdpw/dpwi_session.h"
#include "include/libdpw/dpwi_assert.h"

/*-----------------------------------------------------------------------*
					^E萔錾
 *-----------------------------------------------------------------------*/

#ifdef DPW_SERVER_PUBLIC
#define TR_SERVER_URL		"http://gamestats2.gs.nintendowifi.net/pokemondpds/"
#else
#define TR_SERVER_URL		"http://sdkdev.gamespy.com/games/pokemondpds/web/"
#endif

#ifdef _NITRO
// \̂z̃TCYƂȂĂ邩`FbN
SDK_COMPILER_ASSERT(sizeof(Dpw_Tr_Data) == 292);
#endif

#define TR_URL_UPLOAD	((const unsigned char *)TR_SERVER_URL "worldexchange/post.asp")
#define TR_URL_UPLOADFINISH	((const unsigned char *)TR_SERVER_URL "worldexchange/post_finish.asp")
#define TR_URL_DOWNLOAD	((const unsigned char *)TR_SERVER_URL "worldexchange/get.asp")
#define TR_URL_GETUPLOADRESULT	((const unsigned char *)TR_SERVER_URL "worldexchange/result.asp")
#define TR_URL_DELETE	((const unsigned char *)TR_SERVER_URL "worldexchange/delete.asp")
#define TR_URL_RETURN	((const unsigned char *)TR_SERVER_URL "worldexchange/return.asp")
#define TR_URL_DOWNLOADMATCHDATA ((const unsigned char *)TR_SERVER_URL "worldexchange/search.asp")
#define TR_URL_TRADE	((const unsigned char *)TR_SERVER_URL "worldexchange/exchange.asp")
#define TR_URL_TRADEFINISH	((const unsigned char *)TR_SERVER_URL "worldexchange/exchange_finish.asp")
#define TR_URL_GETSERVERSTATE ((const unsigned char *)TR_SERVER_URL "worldexchange/info.asp")
#define TR_URL_SETPROFILE ((const unsigned char *)TR_SERVER_URL "common/setProfile.asp")
#define TR_URL_INIT	((const char *)TR_SERVER_URL "worldexchange/init.asp")
#define TR_URL_UPDATE ((const char *)TR_SERVER_URL "worldexchange/update.asp")
#define TR_URL_SHUFFLE ((const char *)TR_SERVER_URL "worldexchange/shuffle.asp")

#define DB_HTTP_FLAG_NOT_COMPLETED	(-1)

#define TR_RESPONSE_SUCCESS	(1)
#define TR_RESPONSE_ILLIGAL_REQUEST	(2)
#define TR_RESPONSE_TIMEOUT	(3)
#define TR_RESPONSE_NOT_EXCHANGED (4)
#define TR_RESPONSE_NO_DATA (5)
#define TR_RESPONSE_STOP_SERVICE (6)
#define TR_RESPONSE_SERVER_FULL (7)
#define TR_RESPONSE_NG_POKEMON_NAME (8)
#define TR_RESPONSE_NG_PARENT_NAME (9)
#define TR_RESPONSE_NG_MAIL_NAME (10)
#define TR_RESPONSE_NG_OWNER_NAME (11)
#define TR_RESPONSE_ILLEGAL_DATA (12)
#define TR_RESPONSE_CHEAT_DATA (13)
#define TR_RESPONSE_SERVER_TIMEOUT (14)

typedef enum {
	DPWi_TR_NOT_INIT,
	DPWi_TR_NORMAL,
	DPWi_TR_RROCESS_UPLOAD,
	DPWi_TR_CANCEL_UPLOAD,
	DPWi_TR_RROCESS_UPLOADFINISH,
	DPWi_TR_CANCEL_UPLOADFINISH,
	DPWi_TR_RROCESS_DOWNLOAD,
	DPWi_TR_CANCEL_DOWNLOAD,
	DPWi_TR_RROCESS_GETUPLOADRESULT,
	DPWi_TR_CANCEL_GETUPLOADRESULT,
	DPWi_TR_RROCESS_DELETE,
	DPWi_TR_CANCEL_DELETE,
	DPWi_TR_RROCESS_RETURN,
	DPWi_TR_CANCEL_RETURN,
	DPWi_TR_RROCESS_DOWNLOADMATCHDATA,
	DPWi_TR_CANCEL_DOWNLOADMATCHDATA,
	DPWi_TR_RROCESS_TRADE,
	DPWi_TR_CANCEL_TRADE,
	DPWi_TR_RROCESS_TRADEFINISH,
	DPWi_TR_CANCEL_TRADEFINISH,
	DPWi_TR_RROCESS_GETSERVERSTATE,
	DPWi_TR_CANCEL_GETSERVERSTATE,
	DPWi_TR_RROCESS_SETPROFILE,
	DPWi_TR_CANCEL_SETPROFILE,
	DPWi_TR_END_ASYNC
} DpwiTrState;

typedef struct {
	DpwiTrState state;
	s32 last_result;
	s32 pid;
	u64 friend_key;
	u8 send_buf[sizeof(Dpw_Tr_Data) + 4];
	u8 recv_buf[4];
	u8* user_recv_buf;
} DpwiTrCtrl;

/*-----------------------------------------------------------------------*
					֐vg^Cv錾
 *-----------------------------------------------------------------------*/

static BOOL Dpwi_Tr_CallSessionRequest(const u8* url, const void* data, int len, void* resbuf, int ressize );
static DpwTrError Dpwi_Tr_HandleCommonError(DpwiHttpError error);
static void Dpwi_Db_GhttpCopleteCallback(const char* buf, int len, DWCGHTTPResult result, void* param);

/*-----------------------------------------------------------------------*
					O[oϐ`
 *-----------------------------------------------------------------------*/

static DpwiTrCtrl dpw_tr;
static int db_ghttp_flag;

/*-----------------------------------------------------------------------*
					O[o֐`
 *-----------------------------------------------------------------------*/

/*!
	Dpw_Tr Cu܂BSĂDpw_Tr ̕t֐sOɃR[܂B
	
	GSvt@CID́A DWC_LoginAsync() ֐̃R[obNŎ擾łlŁAT[o[ňӂɃNCAg
	𔻕ʂ邽߂̒lłB
	
	̊֐R[ۂ́ADWC CũC^[lbgڑ֐Ńlbg[NɐڑĂsĂ
	BAOCsĂKv͂܂B
	
	@param[in] pid	GSvt@CID
	@param[in] friend_key	̃thL[BDWC_CreateFriendKey() Ŏ擾ł܂B
*/
void Dpw_Tr_Init(s32 pid, u64 friend_key) {
	
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NOT_INIT, "dpw tr is already initialized.\n");
	
	dpw_tr.state = DPWi_TR_NORMAL;
	dpw_tr.last_result = 0;
	dpw_tr.pid = pid;
	dpw_tr.friend_key = friend_key;
}

/*!
	Q[t[ĂтAi߂܂B
	
	֐̂߁Å֐͏ɌĂяoƂł܂BԂł͉s܂B
*/
void Dpw_Tr_Main(void) {
	
	switch (dpw_tr.state) {
	case DPWi_TR_NOT_INIT:
		break;
	case DPWi_TR_NORMAL:
		break;
	case DPWi_TR_RROCESS_UPLOAD:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			switch (dpw_tr.recv_buf[0]) {
			case TR_RESPONSE_SUCCESS:
				dpw_tr.last_result = 0;
				break;
			case TR_RESPONSE_ILLIGAL_REQUEST:
				dpw_tr.last_result = DPW_TR_ERROR_ILLIGAL_REQUEST;
				break;
			case TR_RESPONSE_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_DATA_TIMEOUT;
				break;
			case TR_RESPONSE_SERVER_FULL:
				dpw_tr.last_result = DPW_TR_ERROR_SERVER_FULL;
				break;
			case TR_RESPONSE_ILLEGAL_DATA:
				dpw_tr.last_result = DPW_TR_ERROR_ILLEGAL_DATA;
				break;
			case TR_RESPONSE_CHEAT_DATA:
				dpw_tr.last_result = DPW_TR_ERROR_CHEAT_DATA;
				break;
			case TR_RESPONSE_NG_POKEMON_NAME:
				dpw_tr.last_result = DPW_TR_ERROR_NG_POKEMON_NAME;
				break;
			case TR_RESPONSE_NG_PARENT_NAME:
				dpw_tr.last_result = DPW_TR_ERROR_NG_PARENT_NAME;
				break;
			case TR_RESPONSE_NG_MAIL_NAME:
				dpw_tr.last_result = DPW_TR_ERROR_NG_MAIL_NAME;
				break;
			case TR_RESPONSE_NG_OWNER_NAME:
				dpw_tr.last_result = DPW_TR_ERROR_NG_OWNER_NAME;
				break;
			case TR_RESPONSE_SERVER_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
				break;
			default:
				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.recv_buf[0]);
				dpw_tr.last_result = DPW_TR_ERROR_FATAL;
				break;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_UPLOADFINISH:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			switch (dpw_tr.recv_buf[0]) {
			case TR_RESPONSE_SUCCESS:
				dpw_tr.last_result = 0;
				break;
			case TR_RESPONSE_ILLIGAL_REQUEST:
				dpw_tr.last_result = DPW_TR_ERROR_ILLIGAL_REQUEST;
				break;
			case TR_RESPONSE_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_DATA_TIMEOUT;
				break;
			case TR_RESPONSE_NO_DATA:
				dpw_tr.last_result = DPW_TR_ERROR_NO_DATA;
				break;
			case TR_RESPONSE_SERVER_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
				break;
			default:
				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.recv_buf[0]);
				dpw_tr.last_result = DPW_TR_ERROR_FATAL;
				break;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_DOWNLOAD:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			if (DpwiGetResponseSize() == sizeof(Dpw_Tr_Data)) {
				dpw_tr.last_result = 0;
			} else {
				switch (dpw_tr.user_recv_buf[0]) {
				case TR_RESPONSE_NO_DATA:
					dpw_tr.last_result = DPW_TR_ERROR_NO_DATA;
					break;
				case TR_RESPONSE_TIMEOUT:
					dpw_tr.last_result = DPW_TR_ERROR_DATA_TIMEOUT;
					break;
				case TR_RESPONSE_SERVER_TIMEOUT:
					dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
					break;
				default:
					DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.user_recv_buf[0]);
					dpw_tr.last_result = DPW_TR_ERROR_FATAL;
					break;
				}
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_GETUPLOADRESULT:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			if (DpwiGetResponseSize() == sizeof(Dpw_Tr_Data)) {
				dpw_tr.last_result = 1;
			} else {
				switch (dpw_tr.user_recv_buf[0]) {
				case TR_RESPONSE_NO_DATA:
					dpw_tr.last_result = DPW_TR_ERROR_NO_DATA;
					break;
				case TR_RESPONSE_TIMEOUT:
					dpw_tr.last_result = DPW_TR_ERROR_DATA_TIMEOUT;
					break;
				case TR_RESPONSE_NOT_EXCHANGED:
					dpw_tr.last_result = 0;
					break;
				case TR_RESPONSE_SERVER_TIMEOUT:
					dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
					break;
				default:
					DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.user_recv_buf[0]);
					dpw_tr.last_result = DPW_TR_ERROR_FATAL;
					break;
				}
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_DELETE:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			switch (dpw_tr.recv_buf[0]) {
			case TR_RESPONSE_SUCCESS:
				dpw_tr.last_result = 0;
				break;
			case TR_RESPONSE_NO_DATA:
				dpw_tr.last_result = DPW_TR_ERROR_NO_DATA;
				break;
			case TR_RESPONSE_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_DATA_TIMEOUT;
				break;
			case TR_RESPONSE_SERVER_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
				break;
			default:
				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.recv_buf[0]);
				dpw_tr.last_result = DPW_TR_ERROR_FATAL;
				break;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_RETURN:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			switch (dpw_tr.recv_buf[0]) {
			case TR_RESPONSE_SUCCESS:
				dpw_tr.last_result = 0;
				break;
			case TR_RESPONSE_NO_DATA:
				dpw_tr.last_result = DPW_TR_ERROR_NO_DATA;
				break;
			case TR_RESPONSE_ILLIGAL_REQUEST:
				dpw_tr.last_result = DPW_TR_ERROR_ILLIGAL_REQUEST;
				break;
			case TR_RESPONSE_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_DATA_TIMEOUT;
				break;
			case TR_RESPONSE_SERVER_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
				break;
			default:
				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.recv_buf[0]);
				dpw_tr.last_result = DPW_TR_ERROR_FATAL;
				break;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_DOWNLOADMATCHDATA:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			{
				int response_size = DpwiGetResponseSize();
				
				if (response_size >= sizeof(Dpw_Tr_Data)) {
					// AĂTCYsizeof(Dpw_Tr_Data)ȏ̂Ƃ́Af[^AĂĂ
					dpw_tr.last_result = (s32)(DpwiGetResponseSize() / sizeof(Dpw_Tr_Data));
				} else if (response_size == 0) {
					// AĂTCY0̂Ƃ́AȂ
					dpw_tr.last_result = 0;
				} else {
					// ȊÔƂ̓T[o[̃X|XR[hɂ
					switch (dpw_tr.user_recv_buf[0]) {
					case TR_RESPONSE_SERVER_TIMEOUT:
						dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
						break;
					default:
						DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.user_recv_buf[0]);
						dpw_tr.last_result = DPW_TR_ERROR_FATAL;
						break;
					}
				}
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_TRADE:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			if (DpwiGetResponseSize() == sizeof(Dpw_Tr_Data)) {
				dpw_tr.last_result = 0;
			} else {
				switch (dpw_tr.user_recv_buf[0]) {
				case TR_RESPONSE_ILLIGAL_REQUEST:
					dpw_tr.last_result = DPW_TR_ERROR_ILLIGAL_REQUEST;
					break;
				case TR_RESPONSE_ILLEGAL_DATA:
					dpw_tr.last_result = DPW_TR_ERROR_ILLEGAL_DATA;
					break;
				case TR_RESPONSE_CHEAT_DATA:
					dpw_tr.last_result = DPW_TR_ERROR_CHEAT_DATA;
					break;
				case TR_RESPONSE_NG_POKEMON_NAME:
					dpw_tr.last_result = DPW_TR_ERROR_NG_POKEMON_NAME;
					break;
				case TR_RESPONSE_NG_PARENT_NAME:
					dpw_tr.last_result = DPW_TR_ERROR_NG_PARENT_NAME;
					break;
				case TR_RESPONSE_NG_MAIL_NAME:
					dpw_tr.last_result = DPW_TR_ERROR_NG_MAIL_NAME;
					break;
				case TR_RESPONSE_NG_OWNER_NAME:
					dpw_tr.last_result = DPW_TR_ERROR_NG_OWNER_NAME;
					break;
				case TR_RESPONSE_SERVER_TIMEOUT:
					dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
					break;
				default:
					DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.user_recv_buf[0]);
					dpw_tr.last_result = DPW_TR_ERROR_FATAL;
					break;
				}
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_TRADEFINISH:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			switch (dpw_tr.recv_buf[0]) {
			case TR_RESPONSE_SUCCESS:
				dpw_tr.last_result = 0;
				break;
			case TR_RESPONSE_SERVER_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
				break;
			case TR_RESPONSE_ILLIGAL_REQUEST:
				dpw_tr.last_result = DPW_TR_ERROR_ILLIGAL_REQUEST;
				break;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_GETSERVERSTATE:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			switch (dpw_tr.recv_buf[0]) {
			case TR_RESPONSE_SUCCESS:
				dpw_tr.last_result = DPW_TR_STATUS_SERVER_OK;
				break;
			case TR_RESPONSE_STOP_SERVICE:
				dpw_tr.last_result = DPW_TR_STATUS_SERVER_STOP_SERVICE;
				break;
			case TR_RESPONSE_SERVER_FULL:
				dpw_tr.last_result = DPW_TR_STATUS_SERVER_FULL;
				break;
			case TR_RESPONSE_SERVER_TIMEOUT:
				dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
				break;
			default:
				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.recv_buf[0]);
				dpw_tr.last_result = DPW_TR_ERROR_FATAL;
				break;
			}
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_RROCESS_SETPROFILE:
		switch( DpwiSessionThink() ){
		case DPWI_COMMON_SESSION_STATE_ERROR:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			dpw_tr.last_result = Dpwi_Tr_HandleCommonError(DpwiGetLastError());
			DpwiSessionShutdown();
			break;
		case DPWI_COMMON_SESSION_STATE_COMPLETED:
			dpw_tr.state = DPWi_TR_END_ASYNC;
			if (DpwiGetResponseSize() == sizeof(Dpw_Common_ProfileResult)) {
				dpw_tr.last_result = 0;
			} else {
    			switch (dpw_tr.recv_buf[0]) {
    			case TR_RESPONSE_SUCCESS:
    				dpw_tr.last_result = DPW_TR_STATUS_SERVER_OK;
    				break;
    			case TR_RESPONSE_STOP_SERVICE:
    				dpw_tr.last_result = DPW_TR_STATUS_SERVER_STOP_SERVICE;
    				break;
    			case TR_RESPONSE_SERVER_FULL:
    				dpw_tr.last_result = DPW_TR_STATUS_SERVER_FULL;
    				break;
    			case TR_RESPONSE_SERVER_TIMEOUT:
    				dpw_tr.last_result = DPW_TR_ERROR_SERVER_TIMEOUT;
    				break;
    			default:
    				DPW_TASSERTMSG(FALSE, "illigal response: %d", dpw_tr.recv_buf[0]);
    				dpw_tr.last_result = DPW_TR_ERROR_FATAL;
    				break;
    			}
            }
			DpwiSessionShutdown();
			break;
		}
		break;
	case DPWi_TR_CANCEL_UPLOAD:
	case DPWi_TR_CANCEL_UPLOADFINISH:
	case DPWi_TR_CANCEL_DOWNLOAD:
	case DPWi_TR_CANCEL_GETUPLOADRESULT:
	case DPWi_TR_CANCEL_DELETE:
	case DPWi_TR_CANCEL_RETURN:
	case DPWi_TR_CANCEL_DOWNLOADMATCHDATA:
	case DPWi_TR_CANCEL_TRADE:
	case DPWi_TR_CANCEL_TRADEFINISH:
	case DPWi_TR_CANCEL_GETSERVERSTATE:
	case DPWi_TR_CANCEL_SETPROFILE:
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_CANCEL;
		DpwiSessionShutdown();
		break;
	case DPWi_TR_END_ASYNC:
		break;
	}
}

/*!
	̊JȂǏIs܂B
	
	̊֐́A񓯊sĂƂɂ̓R[邱Ƃł܂B
	Dpw_Tr_IsAsyncEnd() Ŕ񓯊̏ImFĂR[ĂB
*/
void Dpw_Tr_End(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is running.\n");
	
	dpw_tr.state = DPWi_TR_NOT_INIT;
}

/*!
	Dpw_TrCu̔񓯊IĂ邩ۂԂ܂B
	
	֐ɁuAsyncvt֐R[ۂ́Å֐IɃR[āAImFĂB
	
	@retval TRUE	񓯊IĂ
	@retval FALSE	񓯊IĂȂ
*/
BOOL Dpw_Tr_IsAsyncEnd(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	
	switch (dpw_tr.state) {
	case DPWi_TR_NORMAL:
		return TRUE;
		break;
	case DPWi_TR_END_ASYNC:
		dpw_tr.state = DPWi_TR_NORMAL;
		return TRUE;
		break;
	default:
		break;
	}

	return FALSE;
}

/*!
	O̔񓯊̌ʂ擾܂B
	Bpw_Tr_IsAsyncEnd() ŏImFĂR[ĂB
	
	񓯊sꍇ́AK̊֐ŌʁiɃG[NĂȂjmFĂB
	G[̎ނ͔񓯊ɂĈقȂ܂Aȉ̃G[͋ʂłB
	
	@li ::DPW_TR_ERROR_SERVER_TIMEOUT
	@li ::DPW_TR_ERROR_CANCEL
	@li ::DPW_TR_ERROR_FATAL
	@li ::DPW_TR_ERROR_DISCONNECTED
	@li ::DPW_TR_ERROR_FAILURE
	
	@return	񓯊̌ʁB̒l̓G[A ::DpwTrError ̒lԂ܂B
*/
s32 Dpw_Tr_GetAsyncResult(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	
	return dpw_tr.last_result;
}

/*!
	|Pf[^a鏈Jn܂B
	
	ꍇ́A Dpw_Tr_GetAsyncResult() 0Ԃ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_TR_ERROR_SERVER_FULL		F T[o[̗eʃI[o[łB
	@li ::DPW_TR_ERROR_DATA_TIMEOUT		F ȑOaf[^^CAEgĂ܂B^}SԂĂB
	@li ::DPW_TR_ERROR_ILLIGAL_REQUEST	F T[o[ɊɃf[^܂B Dpw_Tr_DownloadAsync() ŊmFĂB
	@li ::DPW_TR_ERROR_ILLEGAL_DATA		F Abv[hꂽf[^Ă܂B
	@li ::DPW_TR_ERROR_CHEAT_DATA		F Abv[hꂽf[^słB
	@li ::DPW_TR_ERROR_NG_POKEMON_NAME	F Abv[hꂽ|P̖ONG[h܂ł܂B
	@li ::DPW_TR_ERROR_NG_PARENT_NAME	F Abv[hꂽ|P̐e̖ONG[h܂ł܂B
	@li ::DPW_TR_ERROR_NG_MAIL_NAME		F Abv[hꂽ[̖ONG[h܂ł܂B
	@li ::DPW_TR_ERROR_NG_OWNER_NAME	F Abv[hꂽlNG[h܂ł܂B
	
	@param[in] data	Abv[hf[^B֐ŃRs[܂̂ŁAÓIɕێKv͂܂B
	
	@sa Dpw_Tr_UploadFinishAsync()
*/
void Dpw_Tr_UploadAsync(const Dpw_Tr_Data* data) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	DPW_NULL_TASSERT(data);
	
	memcpy(&dpw_tr.send_buf[0], data, sizeof(Dpw_Tr_Data));
	
	// ZbV
	DpwiSessionInitialize();

	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_UPLOAD, dpw_tr.send_buf, sizeof(Dpw_Tr_Data), dpw_tr.recv_buf, 2)) {
		dpw_tr.state = DPWi_TR_RROCESS_UPLOAD;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	Dpw_Tr_UploadAsync()ŃAbv[hT[o[̃f[^Lɂ܂B
	
	ꍇA Dpw_Tr_GetAsyncResult() 0Ԃ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_TR_ERROR_NO_DATA			F T[o[Ƀf[^݂܂Bf[^𕜌ĂB
	@li ::DPW_TR_ERROR_DATA_TIMEOUT		F ȑOaf[^^CAEgĂ܂B^}SԂĂB
	@li ::DPW_TR_ERROR_ILLIGAL_REQUEST	F T[o[ɊɃf[^܂B Dpw_Tr_DownloadAsync() ŊmFĂB
	
	@sa Dpw_Tr_UploadAsync()
*/
void Dpw_Tr_UploadFinishAsync(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	
	// ̃thL[𑗂
	memcpy(&dpw_tr.send_buf[0], &dpw_tr.friend_key, 8);
	
	// ZbV
	DpwiSessionInitialize();

	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_UPLOADFINISH, dpw_tr.send_buf, 8, dpw_tr.recv_buf, 2)) {
		dpw_tr.state = DPWi_TR_RROCESS_UPLOADFINISH;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	Ōɗaf[^擾܂B
	
	ꍇA Dpw_Tr_GetAsyncResult() 0Ԃ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_TR_ERROR_NO_DATA			F T[o[Ƀf[^݂܂Bf[^𕜌ĂB
	@li ::DPW_TR_ERROR_DATA_TIMEOUT		F ȑOaf[^^CAEgĂ܂B^}SԂĂB
	
	T[o[ɂČꂽł擾ł܂B
	Dpw_Tr_DeleteAsync() s͎擾ł܂B
	
	@param[out] data	_E[hf[^L^obt@
*/
void Dpw_Tr_DownloadAsync(Dpw_Tr_Data* data) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	DPW_NULL_TASSERT(data);
	
	dpw_tr.user_recv_buf = (u8*)data;
	
	// ZbV
	DpwiSessionInitialize();
	
	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_DOWNLOAD, dpw_tr.send_buf, 0, data, sizeof(Dpw_Tr_Data))) {
		dpw_tr.state = DPWi_TR_RROCESS_DOWNLOAD;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	ȑOɗaf[^ꂽT[o[ɖ₢킹AĂ
	ꂽf[^̃_E[hs܂B
	
	sȂ0AȂ1 Dpw_Tr_GetAsyncResult() ŕԂ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_TR_ERROR_NO_DATA			F T[o[Ƀf[^݂܂Bf[^𕜌ĂB
	@li ::DPW_TR_ERROR_DATA_TIMEOUT		F ȑOaf[^^CAEgĂ܂B^}SԂĂB
	
	T[o[ɂČf[^́A Dpw_Tr_Data ^̈ȉ̃ołBȊO͕ύX܂B
	
	@li Dpw_Tr_PokemonData postData;
	@li u8 gender;
	@li u16 name[DPW_TR_NAME_SIZE];
	@li u16 trainerID;
	@li u8 countryCode;
	@li s16 localCode;
	@li u8 trainerType;
	@li u8 versionCode;
	@li u8 langCode;
	
	@param[out] data	_E[hf[^L^obt@
*/
void Dpw_Tr_GetUploadResultAsync(Dpw_Tr_Data* data) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	DPW_NULL_TASSERT(data);
	
	dpw_tr.user_recv_buf = (u8*)data;

	// ZbV
	DpwiSessionInitialize();
	
	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_GETUPLOADRESULT, dpw_tr.send_buf, 0, data, sizeof(Dpw_Tr_Data))) {
		dpw_tr.state = DPWi_TR_RROCESS_GETUPLOADRESULT;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	łɃAbv[hĂf[^̍폜s܂B
	̊֐́Aς݂̃f[^폜ꍇɂ̂ݎgpĂB
	
	ꍇA Dpw_Tr_GetAsyncResult() 0Ԃ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_TR_ERROR_NO_DATA			F T[o[Ƀf[^݂܂Bf[^𕜌ĂB
	@li ::DPW_TR_ERROR_DATA_TIMEOUT		F ȑOaf[^^CAEgĂ܂B^}SԂĂB
*/
void Dpw_Tr_DeleteAsync(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	
	// ZbV
	DpwiSessionInitialize();
	
	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_DELETE, dpw_tr.send_buf, 0, dpw_tr.recv_buf, 2)) {
		dpw_tr.state = DPWi_TR_RROCESS_DELETE;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	łɃAbv[hĂf[^̍폜s܂B
	̊֐́AT[o[̃f[^ĂȂꍇ̂ݍ폜ɐ܂B
	
	ꍇA Dpw_Tr_GetAsyncResult() 0Ԃ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_TR_ERROR_NO_DATA			F T[o[Ƀf[^݂܂Bf[^𕜌ĂB
	@li ::DPW_TR_ERROR_DATA_TIMEOUT		F ȑOaf[^^CAEgĂ܂B^}SԂĂB
	@li ::DPW_TR_ERROR_ILLIGAL_REQUEST	F f[^ς݂߁A폜Ɏs܂B Dpw_Tr_DownloadAsync() ŊmFĂB
*/
void Dpw_Tr_ReturnAsync(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	
	// ZbV
	DpwiSessionInitialize();
	
	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_RETURN, dpw_tr.send_buf, 0, dpw_tr.recv_buf, 2)) {
		dpw_tr.state = DPWi_TR_RROCESS_RETURN;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	Dpw_TrCuŌݍsĂ񓯊LZ܂B
	
	LZꍇA Dpw_Tr_GetAsyncResult() ̕Ԓl DPW_TR_ERROR_CANCEL ƂȂ܂B
	
	ɒʐMIĂ邱Ƃ܂̂ŁALZ͏ɐƂ͌܂B
	LZłǂ́A Dpw_Tr_GetAsyncResult() ̕Ԓl DPW_TR_ERROR_CANCEL ƂȂĂ邩ŔfĂB
	
	Dpw_Tr_CancelAsync() ̂񓯊s܂ALZ邱Ƃ͂ł܂B
*/
void Dpw_Tr_CancelAsync(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	
	switch (dpw_tr.state) {
	case DPWi_TR_RROCESS_UPLOAD:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_UPLOAD;
		break;
	case DPWi_TR_RROCESS_UPLOADFINISH:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_UPLOADFINISH;
		break;
	case DPWi_TR_RROCESS_DOWNLOAD:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_DOWNLOAD;
		break;
	case DPWi_TR_RROCESS_GETUPLOADRESULT:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_GETUPLOADRESULT;
		break;
	case DPWi_TR_RROCESS_DELETE:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_DELETE;
		break;
	case DPWi_TR_RROCESS_RETURN:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_RETURN;
		break;
	case DPWi_TR_RROCESS_DOWNLOADMATCHDATA:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_DOWNLOADMATCHDATA;
		break;
	case DPWi_TR_RROCESS_TRADE:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_TRADE;
		break;
	case DPWi_TR_RROCESS_TRADEFINISH:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_TRADEFINISH;
		break;
	case DPWi_TR_RROCESS_GETSERVERSTATE:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_GETSERVERSTATE;
		break;
	case DPWi_TR_RROCESS_SETPROFILE:
		DpwiSessionCancel();
		dpw_tr.state = DPWi_TR_CANCEL_SETPROFILE;
		break;
	default:
		break;
	}
}

/*!
	~|Pf[^ƍvf[^̃_E[hJn܂B
	
	ꍇ Dpw_Tr_GetAsyncResult() Ń_E[hꂽf[^(0-7)Ԃ܂B
	G[Nꍇ́Aʂ̃G[̂ꂩԂ܂B
	
	@param[in] searchData	|P̃f[^B֐ŃRs[܂̂ŁAÓIɕێKv͂܂B
	@param[in] maxNum	_E[h|Pf[^̍ő吔(7܂)
	@param[out] downloadData	_E[hf[^L^obt@BmaxNumŎw肵ȏ̃TCYpӂĂB
*/
void Dpw_Tr_DownloadMatchDataAsync(const Dpw_Tr_PokemonSearchData* searchData, s32 maxNum, Dpw_Tr_Data* downloadData) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	DPW_NULL_TASSERT(searchData);
	DPW_MINMAX_TASSERT(maxNum, 1, 7);
	DPW_NULL_TASSERT(downloadData);
	
	dpw_tr.user_recv_buf = (u8*)downloadData;
	
	// ZbV
	DpwiSessionInitialize();
	
	memcpy(&dpw_tr.send_buf[0], searchData, sizeof(Dpw_Tr_PokemonSearchData));
	dpw_tr.send_buf[sizeof(Dpw_Tr_PokemonSearchData)] = (u8)maxNum;
	
	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_DOWNLOADMATCHDATA, dpw_tr.send_buf,
								   sizeof(Dpw_Tr_PokemonSearchData) + 1, downloadData, (int)sizeof(Dpw_Tr_Data)*maxNum))
	{
		dpw_tr.state = DPWi_TR_RROCESS_DOWNLOADMATCHDATA;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	T[o[id ̃f[^ƌJn܂B
	
	uploadDatadownloadDataɂ͓|C^^邱Ƃł܂B
	
	ꍇA Dpw_Tr_GetAsyncResult() 0ԂAdownloadDataɌf[^܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_TR_ERROR_ILLIGAL_REQUEST	F g[hs܂Bg[hsȌԂɖ߂ĂB
	@li ::DPW_TR_ERROR_ILLEGAL_DATA		F Abv[hꂽf[^Ă܂B
	@li ::DPW_TR_ERROR_CHEAT_DATA		F Abv[hꂽf[^słB
	@li ::DPW_TR_ERROR_NG_POKEMON_NAME	F Abv[hꂽ|P̖ONG[h܂ł܂B
	@li ::DPW_TR_ERROR_NG_PARENT_NAME	F Abv[hꂽ|P̐e̖ONG[h܂ł܂B
	@li ::DPW_TR_ERROR_NG_MAIL_NAME		F Abv[hꂽ[̖ONG[h܂ł܂B
	@li ::DPW_TR_ERROR_NG_OWNER_NAME	F Abv[hꂽlNG[h܂ł܂B
	
	@param[in] id	ID
	@param[in] uploadData	Abv[hf[^B֐ŃRs[܂̂ŁAÓIɕێKv͂܂B
	@param[out] downloadData	_E[hf[^L^obt@
	
	@sa Dpw_Tr_TradeFinish()
*/
void Dpw_Tr_TradeAsync(s32 id, const Dpw_Tr_Data* uploadData, Dpw_Tr_Data* downloadData) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	DPW_NULL_TASSERT(uploadData);
	DPW_NULL_TASSERT(downloadData);
	
	dpw_tr.user_recv_buf = (u8*)downloadData;
	
	// ZbV
	DpwiSessionInitialize();
	
	memcpy(&dpw_tr.send_buf[0], uploadData, sizeof(Dpw_Tr_Data));
//	memcpy(&((Dpw_Tr_Data*)dpw_tr.send_buf)->friend_key, &dpw_tr.friend_key, 8);
	*(s32*)(&dpw_tr.send_buf[sizeof(Dpw_Tr_Data)]) = id;
	
	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_TRADE, dpw_tr.send_buf, sizeof(Dpw_Tr_Data)+4, downloadData, sizeof(Dpw_Tr_Data))) {
		dpw_tr.state = DPWi_TR_RROCESS_TRADE;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	Dpw_Tr_TradeAsync() ŃAbv[hT[o[̃f[^Lɂ܂B
	
	ꍇA Dpw_Tr_GetAsyncResult() 0Ԃ܂B
	G[Nꍇ́Aʂ̃G[͈ȉ̃G[Ԃ܂B
	
	@li ::DPW_TR_ERROR_ILLIGAL_REQUEST	F g[hs܂Bg[hsȌԂɖ߂ĂB
*/
void Dpw_Tr_TradeFinishAsync(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	
	// ̃thL[𑗂
	memcpy(&dpw_tr.send_buf[0], &dpw_tr.friend_key, 8);
	
	// ZbV
	DpwiSessionInitialize();

	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_TRADEFINISH, dpw_tr.send_buf, 8, dpw_tr.recv_buf, 2)) {
		dpw_tr.state = DPWi_TR_RROCESS_TRADEFINISH;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	T[o[Ԃ̒Jn܂B
	
	T[o[ƐɒʐMłꍇ́A Dpw_Tr_GetAsyncResult()  DpwTrServerStatus ̒lԂ܂B
	
	ɒʐMłȂꍇ́A ʂ̃G[̂ꂩԂ܂B
*/
void Dpw_Tr_GetServerStateAsync(void) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	
	// ZbV
	DpwiSessionInitialize();

	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_GETSERVERSTATE, dpw_tr.send_buf, 0, dpw_tr.recv_buf, 2)) {
		dpw_tr.state = DPWi_TR_RROCESS_GETSERVERSTATE;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	lo^鏈Jn܂B
	
	ꍇ́A Dpw_Tr_GetAsyncResult() 0ԂAw肵resultp[^codeoDPW_PROFILE_RESULTCODE_SUCCESSԂ܂B
    
	ɒʐMɋNG[Nꍇ́ADpw_Tr_GetAsyncResult()TR̋ʂ̃G[Ԃ܂B
    lo^ɊւG[Nꍇ͎w肵resultp[^codeoDPW_PROFILE_RESULTCODE_SUCCESSȊO̒lZbg܂B
	
	@param[in] data	Abv[hf[^B֐ŃRs[܂̂ŁAÓIɕێKv͂܂B
                    MACAhX̓CuŊi[̂ŃZbgKv͂܂B
                    DP؍łłversion, language, countryCode, localCode, playerName, playerId, flaĝ݂ZbgĂBȊO0Ŗ߂ĂB
	@param[out] result T[õX|XBcodeoɓo^Ɋւ錋ʂZbgAmailAddrAuthResultɃ[AhXo^Ɋւ錋ʂZbg܂BAmailAddrAuthResultDP؍łł͕K0ɂȂ܂B
*/
void Dpw_Tr_SetProfileAsync(const Dpw_Common_Profile* data, Dpw_Common_ProfileResult* result) {
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	DPW_NULL_TASSERT(data);
	
    DPW_TASSERTMSG(sizeof(Dpw_Common_Profile) <= sizeof(dpw_tr.send_buf), "Internal error: dpw send buf is too small.\n");
    
    // MacAhXZbg
	OS_GetMacAddress((u8*)data->macAddr);
    
	memcpy(dpw_tr.send_buf, data, sizeof(Dpw_Common_Profile));
	dpw_tr.user_recv_buf = (u8*)result;
	
	// ZbV
	DpwiSessionInitialize();

	// ʐMJn
	if (Dpwi_Tr_CallSessionRequest(TR_URL_SETPROFILE, dpw_tr.send_buf, sizeof(Dpw_Common_Profile), dpw_tr.user_recv_buf, sizeof(Dpw_Common_ProfileResult))) {
		dpw_tr.state = DPWi_TR_RROCESS_SETPROFILE;
	} else {
		dpw_tr.state = DPWi_TR_END_ASYNC;
		dpw_tr.last_result = DPW_TR_ERROR_FATAL;
		DpwiSessionShutdown();
	}
}

/*!
	EʐM̃T[oԂ܂B
	ŃubN܂B
	
	̊֐͔p~܂BDpw Admin ToolgpB
*/
BOOL Dpw_Tr_Db_InitServer(void) {
#if 0
	
	int ret;
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	
	db_ghttp_flag = DB_HTTP_FLAG_NOT_COMPLETED;
	
	(void)DWC_InitGHTTP(NULL);
	
	ret = DWC_GetGHTTPData(TR_URL_INIT, Dpwi_Db_GhttpCopleteCallback, NULL);
	
	while (db_ghttp_flag == DB_HTTP_FLAG_NOT_COMPLETED) {
		
		if (!DWC_ProcessGHTTP()) {
			DWC_ClearError();
			return FALSE;
		}
		
		OS_Sleep(16);
	}
	
	DWC_ShutdownGHTTP();
	
	if (db_ghttp_flag == DWC_GHTTP_SUCCESS) {
		return TRUE;
	} else {
		DWC_ClearError();
		return FALSE;
	}
#else
    return FALSE;
#endif
}

/*!
	EʐM̃T[ö̃Abvf[gs܂B
	ŃubN܂B
	
	̊֐͔p~܂BDpw Admin ToolgpB
*/
BOOL Dpw_Tr_Db_UpdateServer(void) {
#if 0
	
	int ret;
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	
	db_ghttp_flag = DB_HTTP_FLAG_NOT_COMPLETED;
	
	(void)DWC_InitGHTTP(NULL);
	
	ret = DWC_GetGHTTPData(TR_URL_UPDATE, Dpwi_Db_GhttpCopleteCallback, NULL);
	
	while (db_ghttp_flag == DB_HTTP_FLAG_NOT_COMPLETED) {
		
		if (!DWC_ProcessGHTTP()) {
			DWC_ClearError();
			return FALSE;
		}
		
		OS_Sleep(16);
	}
	
	DWC_ShutdownGHTTP();
	
	if (db_ghttp_flag == DWC_GHTTP_SUCCESS) {
		return TRUE;
	} else {
		DWC_ClearError();
		return FALSE;
	}
#else
    return FALSE;
#endif
}

/*!
	EʐḾAaĂ|P鏇ԂύX܂B
	̏́AiŃT[o[ł͈Ɉs\łB
	
	̊֐͔p~܂BDpw Admin ToolgpB
*/
BOOL Dpw_Tr_Db_ShuffleServerData(void) {
#if 0
	
	int ret;
	
	DPW_TASSERTMSG(dpw_tr.state != DPWi_TR_NOT_INIT, "dpw tr is not initialized.\n");
	DPW_TASSERTMSG(dpw_tr.state == DPWi_TR_NORMAL, "async process is already running.\n");
	
	db_ghttp_flag = DB_HTTP_FLAG_NOT_COMPLETED;
	
	(void)DWC_InitGHTTP(NULL);
	
	ret = DWC_GetGHTTPData(TR_URL_SHUFFLE, Dpwi_Db_GhttpCopleteCallback, NULL);
	
	while (db_ghttp_flag == DB_HTTP_FLAG_NOT_COMPLETED) {
		
		if (!DWC_ProcessGHTTP()) {
			DWC_ClearError();
			return FALSE;
		}
		
		OS_Sleep(16);
	}
	
	DWC_ShutdownGHTTP();
	
	if (db_ghttp_flag == DWC_GHTTP_SUCCESS) {
		return TRUE;
	} else {
		DWC_ClearError();
		return FALSE;
	}
#else
    return FALSE;
#endif
}

/*-----------------------------------------------------------------------*
					[J֐`
 *-----------------------------------------------------------------------*/

static BOOL Dpwi_Tr_CallSessionRequest(const u8* url, const void* data, int len, void* resbuf, int ressize) {
	
	OS_TPrintf("[DPW TR] Connecting to %s.\n", url);
	switch (DpwiSessionRequest(url, dpw_tr.pid, data, len, (u8*)resbuf, ressize)) {
	case DPWI_COMMON_SESSION_SUCCESS:				// I
		return TRUE;
		break;
	case DPWI_COMMON_SESSION_ERROR_NOTINITIALIZED:	// 
		DPW_TASSERTMSG(FALSE, "common session not initialized.");
		break;
	case DPWI_COMMON_SESSION_ERROR_NOMEMORY:		// [s
		DPW_TASSERTMSG(FALSE, "common session memory shortage.");
		break;
	}
	
	return FALSE;
}

static DpwTrError Dpwi_Tr_HandleCommonError(DpwiHttpError error) {
	
	DpwTrError ret;
	
	OS_TPrintf("[DPW TR] ghttp error: %d\n", error);
	
	switch (error) {
	case DPWI_COMMON_SESSION_ERROR_IN_ERROR:			// G[ 
	case DPWI_COMMON_SESSION_ERROR_INVALID_POST:		// ȑM 
	case DPWI_COMMON_SESSION_ERROR_INVALID_FILE_NAME:	// ȃt@C 
	case DPWI_COMMON_SESSION_ERROR_INVALID_BUFFER_SIZE:	// ȃobt@TCY 
	case DPWI_COMMON_SESSION_ERROR_INVALID_URL:			// URL
	case DPWI_COMMON_SESSION_ERROR_UNSPECIFIED_ERROR:	// ̑̃G[
	case DPWI_COMMON_SESSION_ERROR_BUFFER_OVERFLOW:		// ꂽobt@邽߁At@C̎擾s 
	case DPWI_COMMON_SESSION_ERROR_PARSE_URL_FAILED:	// URL̓G[ 
//	case DPWI_COMMON_SESSION_ERROR_ENCRYPTION_ERROR:	// ÍG[ 
	case DPWI_COMMON_SESSION_ERROR_FILE_TOO_BIG:		// t@C傫邽߃_E[hs\ 
	case DPWI_COMMON_SESSION_ERROR_FILE_INCOMPLETE:		// _E[h̒f 
	case DPWI_COMMON_SESSION_ERROR_FILE_WRITE_FAILED:	// [Jt@Cւ̏݃G[ 
	case DPWI_COMMON_SESSION_ERROR_FILE_READ_FAILED:	// [Jt@C̓ǂݏoG[ 
	case DPWI_COMMON_SESSION_ERROR_BAD_RESPONSE:		// HTTPT[õX|X̉̓G[ 
	case DPWI_COMMON_SESSION_ERROR_BUFFER_OVER:			// COMMONw: Mobt@I[o[
		DPW_TASSERTMSG(FALSE, "library internal error. please contact author.");
		ret = DPW_TR_ERROR_FATAL;
		break;
	case DPWI_COMMON_SESSION_ERROR_INSUFFICIENT_MEMORY:	// s 
	case DPWI_COMMON_SESSION_ERROR_OUT_OF_MEMORY:		// 蓖Ďs 
	case DPWI_COMMON_SESSION_ERROR_MEMORY_ERROR: 		// 蓖Ďs 
		DPW_TASSERTMSG(FALSE, "common session memory shortage.");
		ret = DPW_TR_ERROR_FATAL;
		break;
	case DPWI_COMMON_SESSION_ERROR_HOST_LOOKUP_FAILED:	// zXgs 
		OS_TPrintf("[DPW TR] dns lookup failed.\n");
		ret = DPW_TR_ERROR_FAILURE;
		break;
	case DPWI_COMMON_SESSION_ERROR_SOCKET_FAILED:		// \Pbg̍쐬AAǂݏoAݎs 
	case DPWI_COMMON_SESSION_ERROR_CONNECT_FAILED:		// HTTPT[oւ̐ڑs 
		OS_TPrintf("[DPW TR] socket error.\n");
		ret = DPW_TR_ERROR_FAILURE;
		break;
	case DPWI_COMMON_SESSION_ERROR_UNAUTHORIZED:		// t@C擾 
	case DPWI_COMMON_SESSION_ERROR_FORBIDDEN:			// HTTPT[õt@CM 
	case DPWI_COMMON_SESSION_ERROR_FILE_NOT_FOUND:		// HTTPT[õt@Cs 
	case DPWI_COMMON_SESSION_ERROR_SERVER_ERROR:		// HTTPT[oG[ 
	case DPWI_COMMON_SESSION_ERROR_CHECKSUM:			// COMMONw: `FbNT̕sv
	case DPWI_COMMON_SESSION_ERROR_PID:					// COMMONw: PID̕sv
	case DPWI_COMMON_SESSION_ERROR_DATA_LENGTH: 		// COMMONw: f[^̒s
	case DPWI_COMMON_SESSION_ERROR_TOKEN_NOT_FOUND:		// COMMONw: g[NȂ
	case DPWI_COMMON_SESSION_ERROR_INCORRECT_HASH:		// COMMONw: nbVȂ
		OS_TPrintf("[DPW TR] server internal error.  please contact server administrator.\n");
		ret = DPW_TR_ERROR_SERVER_TIMEOUT;
		break;
	case DPWI_COMMON_SESSION_ERROR_REQUEST_REJECTED:	// HTTPT[õNGXg 
	case DPWI_COMMON_SESSION_ERROR_TOKEN_EXPIRED:		// COMMONw: g[N̗L؂
		OS_TPrintf("[DPW TR] server is now heavy.\n");
		ret = DPW_TR_ERROR_SERVER_TIMEOUT;
		break;
	default:
		DPW_TASSERTMSG(FALSE, "illigal ghttp error: %d", error);
		ret = DPW_TR_ERROR_FATAL;
		break;
	}
	
	// FATALG[łȂꍇ
	if (ret != DPW_TR_ERROR_FATAL) {
#ifdef _NITRO
		// NitroWiFĩC[Ŗ؂ĂȂ`FbN
		if (WCM_GetPhase() != WCM_PHASE_DCF) {
			OS_TPrintf("[DPW TR] disconnected from access point.\n");
			ret = DPW_TR_ERROR_DISCONNECTED;
		}
#endif
		// DWC̃G[ԂNA
		DWC_ClearError();
	}
	
	return ret;
}

#ifndef SDK_FINALROM
static void Dpwi_Db_GhttpCopleteCallback(const char* buf, int len, DWCGHTTPResult result, void* param) {
#pragma unused(buf, len, param)
	
	db_ghttp_flag = result;
}
#endif

