#include "DividedHangul.hpp"
#include "FontBitmap.hpp"
#include "Bitmap2Color.hpp"

#include <assert.h>
#include <limits.h>

#include <iostream>
#include <fstream>
#include <vector>
using namespace std;

typedef struct {
	unsigned short letterNum;			//  2 bytes
	unsigned char  width;				//  1 byte
	unsigned char  partsSetNum;			//  1 byte
} HangulFontHeader;						//  4 bytes

typedef struct {
	unsigned char  x;					//  1 byte
	unsigned char  y;					//  1 byte
	unsigned char  width;				//  1 byte
	unsigned char  height;				//  1 byte
	unsigned short partsNum;			//  2 bytes
	unsigned char  bitDataSize;			//  1 byte
	unsigned char  partIdSize;			//  1 byte
	unsigned int   bitDataTableAddr;	//  4 bytes
	unsigned int   partIdTableAddr;		//  4 bytes
} PartsSetHeader;						// 16 bytes

static int getIndexOf(vector<Bitmap2Color *> *v, Bitmap2Color *bitmap);
static int countBitSize(unsigned int value);

static int getIndexOf(vector<Bitmap2Color *> *v, Bitmap2Color *bitmap) {
	for (int i = 0; i < v->size(); ++i) {
		if (bitmap->equals(v->at(i))) return i;
	}
	return -1;
}

// x \̂ɕKvȃrbg
static int countBitSize(unsigned int x) {
	if (x == 0) return 0;

	int n = 31;
	if (x <= 0x0000FFFF) { n -= 16; x <<= 16; }
	if (x <= 0x00FFFFFF) { n -=  8; x <<=  8; }
	if (x <= 0x0FFFFFFF) { n -=  4; x <<=  4; }
	if (x <= 0x3FFFFFFF) { n -=  2; x <<=  2; }
	n += (x >> 31);

	return n;
}

DividedHangul::DividedHangul() {
}

DividedHangul::~DividedHangul() {
	int num = _partsSetVector.size();

	for (int i = 0; i < num; ++i) {
		FontPartsSet *partsSet = _partsSetVector.at(i);

		delete[] partsSet->indexTable;
		for (int i = 0; i < partsSet->bitDataNum; ++i) {
			delete partsSet->bitDataTable[i];
		}
		delete[] partsSet->bitDataTable;
	}
}

void DividedHangul::addParts(FontBitmap *fontBitmap, int x, int y, int width, int height, int yofs) {
	FontPartsSet *partsSet = new FontPartsSet;

	partsSet->x = x;
	partsSet->y = y + yofs;
	partsSet->width  = width;
	partsSet->height = height;

	partsSet->indexNum = fontBitmap->getEntryNum();
	partsSet->indexTable = new int[partsSet->indexNum];

	vector<Bitmap2Color *> v;

	for (int i = 0; i < partsSet->indexNum; ++i) {
		Bitmap2Color *bitmap = new Bitmap2Color(partsSet->width, partsSet->height);

		for (int n = 0; n < height; ++n) {
			for (int m = 0; m < width; ++m) {
				bitmap->setPixel(m, n, (int)fontBitmap->isFontPixel(i, x+m, y+n));
			}
		}

		partsSet->indexTable[i] = getIndexOf(&v, bitmap);
		if (partsSet->indexTable[i] < 0) {
			v.push_back(bitmap);
			partsSet->indexTable[i] = v.size() - 1;
		} else {
			delete bitmap;
		}
	}

	partsSet->bitDataNum = v.size();
	partsSet->bitDataTable = new Bitmap2Color *[partsSet->bitDataNum];

	for (int i = 0; i < partsSet->bitDataNum; ++i) {
		partsSet->bitDataTable[i] = v.at(i);
	}

//	cout << partsSet->bitDataNum << " ptns. / " << partsSet->indexNum << " chars." << endl;

	_partsSetVector.push_back(partsSet);
}

int DividedHangul::_getPartsSetSize(FontPartsSet *partsSet) {
	int size = 0;
	
	size += sizeof(PartsSetHeader);		// 4 ̔{ۏ
	size += (partsSet->bitDataTable[0]->getSize() * partsSet->bitDataNum + 3) / 4 * 4;
	size += ((partsSet->bitDataNum < 0x100 ? 1 : 2) * partsSet->indexNum + 3) / 4 * 4;

	return size;
}

int DividedHangul::printCompressRate(void) {
	int areaNum = _partsSetVector.size();
	int indexNum = 2355;//_partsSetVector.at(0)->indexNum;
	int indexTableSizeTotal = 0;
	int dotTableSizeTotal = 0;
	int left = INT_MAX;
	int right = INT_MIN;
	int top = INT_MAX;
	int bottom = INT_MIN;

	cout << "x,y,width,height,partsnum,indexsize,indextablesize,dottablesize" << endl;

	for (int i = 0; i < areaNum; ++i) {
		FontPartsSet *partsSet = _partsSetVector.at(i);
		int indexSize = countBitSize(partsSet->bitDataNum);
		int indexTableSize = (indexNum * indexSize + 7) / 8;
		int dotTableSize = (partsSet->width * partsSet->height * partsSet->bitDataNum + 7) / 8;
		indexTableSizeTotal += indexTableSize;
		dotTableSizeTotal += dotTableSize;
		if (partsSet->x < left) left = partsSet->x;
		if (partsSet->y < top)  top  = partsSet->y;
		if (partsSet->x + partsSet->width  > right)  right  = partsSet->x + partsSet->width;
		if (partsSet->y + partsSet->height > bottom) bottom = partsSet->y + partsSet->height;
/*		cout << "area: (" << partsSet->x << "," << partsSet->y << "," << partsSet->width << "," << partsSet->height << ") ";
		cout << "partsnum: " << partsSet->bitDataNum << " [" << indexSize << "bits] ";
		cout << "indextable: " << indexTableSize << "Bytes ";
		cout << "dottable: " << dotTableSize << "Bytes";
*/		cout << partsSet->x << "," << partsSet->y << "," << partsSet->width << "," << partsSet->height << ",";
		cout << partsSet->bitDataNum << "," << indexSize << ",";
		cout << indexTableSize << "," << dotTableSize;
		cout << endl;
	}

	int originalSize = ((right - left) * (bottom - top) * indexNum + 7) / 8;
	double compressRate = ((indexTableSizeTotal + dotTableSizeTotal) * 10000 / originalSize + 5) / 10 / 10.0;

/*	cout << "total: " << indexTableSizeTotal + dotTableSizeTotal << "Bytes, ";
	cout << "original: " << originalSize << "Bytes, ";
	cout << "compress: " << compressRate << "%";
*/	cout << endl;
	cout << "total,original,compress" << endl;
	cout << indexTableSizeTotal + dotTableSizeTotal << "," << originalSize << "," << compressRate;
	cout << endl;

	return 0;
}

int DividedHangul::saveFile(const char *filename) {
	ofstream out;

	out.open(filename, ios::out|ios::binary);

	if (!out) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	HangulFontHeader hangulFontHeader;
	hangulFontHeader.letterNum   = _partsSetVector.at(0)->indexNum;
	hangulFontHeader.partsSetNum = _partsSetVector.size();

	{
		int left, right;

		left   = INT_MAX;
		right  = 0;
		
		for (int i = 0; i < hangulFontHeader.partsSetNum; ++i) {
			FontPartsSet *partsSet = _partsSetVector.at(i);
			if (left   > partsSet->x)                    left   = partsSet->x;
			if (right  < partsSet->x + partsSet->width)  right  = partsSet->x + partsSet->width;
		}

		hangulFontHeader.width  = right - left + 1;
	}

	unsigned int *partsSetAddrTable = new unsigned int[hangulFontHeader.partsSetNum];
	partsSetAddrTable[0] = sizeof(HangulFontHeader)									// hangulFontHeader
	                       + sizeof(unsigned int) * hangulFontHeader.partsSetNum;	// partsSetAddrTable
	for (int i = 1; i < hangulFontHeader.partsSetNum; ++i) {
		partsSetAddrTable[i] = partsSetAddrTable[i-1] + _getPartsSetSize(_partsSetVector.at(i-1));
	}

	out.write((char *)&hangulFontHeader, sizeof(HangulFontHeader));
	out.write((char *)partsSetAddrTable, sizeof(unsigned int) * hangulFontHeader.partsSetNum);

	for (int i = 0; i < hangulFontHeader.partsSetNum; ++i) {
		PartsSetHeader partsSetHeader;
		FontPartsSet *partsSet = _partsSetVector.at(i);

		// font set header
		partsSetHeader.x                = (unsigned char) partsSet->x;
		partsSetHeader.y                = (unsigned char) partsSet->y;
		partsSetHeader.width            = (unsigned char) partsSet->width;
		partsSetHeader.height           = (unsigned char) partsSet->height;
		partsSetHeader.partsNum         = (unsigned short)partsSet->bitDataNum;
		partsSetHeader.bitDataSize      = (unsigned char) partsSet->bitDataTable[0]->getSize();
		partsSetHeader.partIdSize       = (unsigned char) (partsSet->bitDataNum < 0x100 ? 1 : 2);
		partsSetHeader.bitDataTableAddr = (unsigned int)  (partsSetAddrTable[i] + sizeof(PartsSetHeader));
		partsSetHeader.partIdTableAddr  = (unsigned int)  (partsSetHeader.bitDataTableAddr + (partsSetHeader.bitDataSize * partsSetHeader.partsNum + 3) / 4 * 4);
		out.write((char *)&partsSetHeader, sizeof(PartsSetHeader));

		// bit data
		for (int j = 0; j < partsSetHeader.partsNum; ++j) {
			out.write((char *)partsSet->bitDataTable[j]->getBitData(), partsSetHeader.bitDataSize);
		}

		// for 4-byte alignment
		if (partsSetHeader.bitDataSize * partsSetHeader.partsNum % 4 != 0) {
			int paddingSize = 4 - (partsSetHeader.bitDataSize * partsSetHeader.partsNum % 4);
			unsigned char *padding = new unsigned char[paddingSize];
			out.write((char *)padding, paddingSize);
			delete[] padding;
		}

		// parts index data
		switch (partsSetHeader.partIdSize) {
			case 1:
				{
					int j;
					int indexTableSize = (partsSet->indexNum + 3) / 4 * 4;
					unsigned char *indexTable = new unsigned char[indexTableSize];

					for (j = 0; j < partsSet->indexNum; ++j) {
						indexTable[j] = (unsigned char)partsSet->indexTable[j];
					}
					for (; j < indexTableSize; ++j) {
						j = 0;
					}

					out.write((char *)indexTable, indexTableSize);

					delete[] indexTable;
				}
				break;
			case 2:
				{
					int j;
					int indexTableSize = (partsSet->indexNum + 1) / 2 * 2;
					unsigned short *indexTable = new unsigned short[indexTableSize];

					for (j = 0; j < partsSet->indexNum; ++j) {
						indexTable[j] = (unsigned short)partsSet->indexTable[j];
					}
					for (; j < indexTableSize; ++j) {
						j = 0;
					}

					out.write((char *)indexTable, indexTableSize * sizeof(unsigned short));

					delete[] indexTable;
				}
				break;
			default:
				assert(0);
		}
	}

	delete[] partsSetAddrTable;

	return 0;
}


typedef struct {
	unsigned short padding;		// unsigned long ̗vf4oCgEɑ
	char           bfType[2];
	unsigned long  bfSize;
	unsigned short bfReserved1;
	unsigned short bfReserved2;
	unsigned long  bfOffBits;
} BITMAPFILEHEADEREX;

typedef struct {
	unsigned long  biSize;
	long           biWidth;
	long           biHeight;
	unsigned short biPlanes;
	unsigned short biBitCount;
	unsigned long  biCompression;
	unsigned long  biSizeImage;
	long           biXPixPerMeter;
	long           biYPixPerMeter;
	unsigned long  biClrUsed;
	unsigned long  biClrImporant;
} BITMAPINFOHEADER;

typedef struct {
	unsigned char rgbBlue;
	unsigned char rgbGreen;
	unsigned char rgbRed;
	unsigned char rgbReserved;
} RGBQUAD;


int DividedHangul::_saveBitmap(FontPartsSet *partsSet, const char *filename) {
	int bitmapLineNum    = (partsSet->bitDataNum + BITMAP_CHARACTERNUMLINE - 1) / BITMAP_CHARACTERNUMLINE;
	int bitmapHeight     = BITMAP_BOXHEIGHT * bitmapLineNum;
	int bitmapImageSize  = BITMAP_WIDTH * bitmapHeight;
	int bitmapHeaderSize = sizeof(BITMAPFILEHEADEREX) - BITMAPFILEHEADEREX_PADDINGSIZE + sizeof(BITMAPINFOHEADER) + BITMAP_PALETTESIZE;
	int bitmapFileSize   = bitmapHeaderSize + bitmapImageSize;

	BITMAPFILEHEADEREX fileHeaderEx = {
		0,
		{'B', 'M'},
		bitmapFileSize,
		0,
		0,
		bitmapHeaderSize
	};

	BITMAPINFOHEADER   infoHeader = {
		BITMAPINFOHEADER_SIZE,
		BITMAP_WIDTH,
		bitmapHeight,
		BITMAP_PLANES,
		BITMAP_BITCOUNT,
		BITMAP_COMPRESSION,
		bitmapImageSize,
		BITMAP_XPIXPERMETER,
		BITMAP_YPIXPERMETER,
		BITMAP_CLRUSED,
		BITMAP_CLRIMPORTANT
	};

	RGBQUAD *paletteData = new RGBQUAD[BITMAP_CLRUSED];
	paletteData[0].rgbBlue = paletteData[0].rgbGreen = paletteData[0].rgbRed = 255;
	paletteData[1].rgbBlue = paletteData[1].rgbGreen = paletteData[1].rgbRed =   0;
	paletteData[2].rgbBlue = paletteData[2].rgbGreen = paletteData[2].rgbRed = 223;
	paletteData[3].rgbBlue = paletteData[3].rgbGreen = paletteData[3].rgbRed = 191;

	unsigned char *imageData = new unsigned char[bitmapImageSize];
	for (int i = 0; i < bitmapImageSize; ++i) imageData[i] = 3;

	for (int i = 0; i < partsSet->bitDataNum; ++i) {
		for (int partY = 0; partY < partsSet->height; ++partY) {
			for (int partX = 0; partX < partsSet->width; ++partX) {
				int bmpX = i % BITMAP_CHARACTERNUMLINE * BITMAP_BOXWIDTH  + partX + partsSet->x;
				int bmpY = i / BITMAP_CHARACTERNUMLINE * BITMAP_BOXHEIGHT + partY + partsSet->y;
				int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
				imageData[index] = partsSet->bitDataTable[i]->getPixel(partX, partY);
			}
		}
	}

	ofstream out;

	out.open(filename, ios::out|ios::binary);

	if (!out) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	out.write((char *)&fileHeaderEx + BITMAPFILEHEADEREX_PADDINGSIZE,
	          sizeof(BITMAPFILEHEADEREX) - BITMAPFILEHEADEREX_PADDINGSIZE);
	out.write((char *)&infoHeader, sizeof(BITMAPINFOHEADER));
	out.write((char *)paletteData, BITMAP_PALETTESIZE);
	out.write((char *)imageData,   bitmapImageSize);

	out.close();

	delete[] paletteData;
	delete[] imageData;

	return 0;
}
