#include "FontBitmap.hpp"
#include "Bitmap2Color.hpp"

#include <iostream>
#include <fstream>
using namespace std;

#include <assert.h>

typedef struct {
	unsigned short padding;		// unsigned long ̗vf4oCgEɑ
	char           bfType[2];
	unsigned long  bfSize;
	unsigned short bfReserved1;
	unsigned short bfReserved2;
	unsigned long  bfOffBits;
} BITMAPFILEHEADEREX;

typedef struct {
    unsigned long  biSize;
    long           biWidth;
    long           biHeight;
    unsigned short biPlanes;
    unsigned short biBitCount;
    unsigned long  biCompression;
    unsigned long  biSizeImage;
    long           biXPixPerMeter;
    long           biYPixPerMeter;
    unsigned long  biClrUsed;
    unsigned long  biClrImporant;
} BITMAPINFOHEADER;


FontBitmap::FontBitmap() {
	_entryNum = 0;

	_pixelDataSize = 0;
	_pixelData = NULL;

	_bitmapWidth = 0;
	_bitmapHeight = 0;
	_lineSize = 0;

	_colorGrid       = 0;
	_colorMargin     = 0;
	_colorFont       = 0;
	_colorBackground = 0;

	_blockWidth = 0;
	_blockHeight = 0;
	_cellWidth = 0;
	_cellHeight = 0;

	_charTop    = 0;
	_charBottom = 0;
	_charHeight = 0;

	_blockNumLine = 0;
}

FontBitmap::~FontBitmap() {
	if (_pixelData) delete _pixelData;
}

int FontBitmap::_getPixelIndex(int x, int y) {
	if (_bitmapHeight > 0) {
		return x + (_bitmapHeight - y - 1) * _lineSize;		// E
	} else {
		return x + (-y) * _lineSize;						// ォE
	}
}

int FontBitmap::_getPixel(int x, int y) {
	return _pixelData[_getPixelIndex(x, y)];
}

void FontBitmap::_searchColors() {
	_colorGrid   = _getPixel(0, 0);
	_colorMargin = _getPixel(1, 1);
	_colorFont   = _colorGrid;

	for (int y = 0; y < _bitmapHeight; ++y) {
		int color = _getPixel(0, y);
		if (color != _colorGrid) {
			_colorBackground = color;
			break;
		}
	}
}

int FontBitmap::_countBlockWidth() {
	int x = 1;
	int y = 1;

	while (_getPixel(x, y) != _colorGrid) x++;

	return x + 1;
}

int FontBitmap::_countBlockHeight() {
	int x = 1;
	int y = 1;

	while (_getPixel(x, y) != _colorGrid) y++;

	return y + 1;
}

// StHg̏[ԂiyW܂ށj
int FontBitmap::_getCharTop() {
	int top = _cellHeight;

	for (int i = 0; i < _entryNum; ++i) {
		for (int y = top; y >= 0; --y) {
			for (int x = 0; x < _cellWidth; ++x) {
				if (getCharPixel(i, x, y) == _colorFont) top = y;
			}
		}
	}

	return top;
}

// StHg̉[ԂiyW͊܂܂Ȃj
int FontBitmap::_getCharBottom() {
	int bottom = 0;

	for (int i = 0; i < _entryNum; ++i) {
		for (int y = bottom; y < _cellHeight; ++y) {
			for (int x = 0; x < _cellWidth; ++x) {
				if (getCharPixel(i, x, y) == _colorFont) bottom = y;
			}
		}
	}

	return bottom + 1;
}

int FontBitmap::loadFile(const char *filename) {
	ifstream in;
	BITMAPFILEHEADEREX fileHeaderEx;
	BITMAPINFOHEADER   infoHeader;

	in.open(filename, ios::in|ios::binary);

	if (!in) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	in.read((char *)&fileHeaderEx + BITMAPFILEHEADEREX_PADDINGSIZE,
	        sizeof(BITMAPFILEHEADEREX) - BITMAPFILEHEADEREX_PADDINGSIZE);

	if (fileHeaderEx.bfType[0] != 'B' || fileHeaderEx.bfType[1] != 'M') {
		cout << "File is not a bitmap." << endl;
		return 1;
	}

	in.read((char *)&infoHeader, sizeof(BITMAPINFOHEADER));

	if (infoHeader.biSize != BITMAPINFOHEADER_SIZE || infoHeader.biBitCount != 8) {
		cout << "This bitmap format is not supported." << endl;
		return 1;
	}

	_bitmapWidth  = infoHeader.biWidth;
	_bitmapHeight = infoHeader.biHeight;
	_lineSize     = (_bitmapWidth + 3) / 4 * 4;

	_pixelDataSize = fileHeaderEx.bfSize - fileHeaderEx.bfOffBits;
	_pixelData = new unsigned char[_pixelDataSize];
	in.seekg(fileHeaderEx.bfOffBits, ios::beg);
	in.read((char *)_pixelData, _pixelDataSize);

	if (in.gcount() != _pixelDataSize) {
		cout << "File is too small." << endl;
		return 1;
	}

	_searchColors();

	_blockWidth  = _countBlockWidth();
	_blockHeight = _countBlockHeight();
	_cellWidth  = _blockWidth  - 4;
	_cellHeight = _blockHeight - 6;

	_blockNumLine = _bitmapWidth / _blockWidth;
	_entryNum = _blockNumLine * _bitmapHeight / _blockHeight;

	_charTop    = _getCharTop();
	_charBottom = _getCharBottom();
	_charHeight = _charBottom - _charTop;

	in.close();

	return 0;
}

// Z̍[xWԂ
int FontBitmap::getCellX(int charCode) {
	return (charCode % _blockNumLine) * _blockWidth  + 2;
}

// Z̍[yWԂ
int FontBitmap::getCellY(int charCode) {
	return (charCode / _blockNumLine) * _blockHeight + 2;
}

// GA̍[xWԂ
int FontBitmap::getWidthLineX(int charCode) {
	return (charCode % _blockNumLine    ) * _blockWidth  + 2;
}

// GAyWԂ
int FontBitmap::getWidthLineY(int charCode) {
	return (charCode / _blockNumLine + 1) * _blockHeight - 3;
}

int FontBitmap::getCharPixel(int charCode, int x, int y) {
	x += getCellX(charCode);
	y += getCellY(charCode);

	return _getPixel(x, y);
}

bool FontBitmap::isFontPixel(int charCode, int x, int y) {
	return getCharPixel(charCode, x, y) == _colorFont;
}

void FontBitmap::getCharBitmap(int charCode, Bitmap2Color *bitmap2Color) {
	int widthLineY     = getWidthLineY(charCode);
	int widthLineXHead = getWidthLineX(charCode);
	while (_getPixel(widthLineXHead, widthLineY) == _colorBackground) widthLineXHead++;
	int widthLineXTail = widthLineXHead;
	while (_getPixel(widthLineXTail, widthLineY) == _colorFont)       widthLineXTail++;

	int charWidth = widthLineXTail - widthLineXHead;

	int cellYHead = getCellY(charCode);
	int charYHead = cellYHead + _charTop;
	int charYTail = cellYHead + _charBottom;

	bitmap2Color->setSize(charWidth, _charHeight);

	for (int bmpY = charYHead, charY = 0; bmpY < charYTail; ++bmpY, ++charY) {
		for (int bmpX = widthLineXHead, charX = 0; bmpX < widthLineXTail; ++bmpX, ++charX) {
			if (_getPixel(bmpX, bmpY) == _colorFont) bitmap2Color->setPixel(charX, charY, 1);
		}
	}
}
