#include "PokemonFont.hpp"
#include "FontBitmap.hpp"
#include "Bitmap2Color.hpp"

#include <iostream>
#include <fstream>
using namespace std;

PokemonFont::PokemonFont() {
	_header.bitDataOffs  = sizeof(FONTDATA_HEADER);
	_header.widthTblOffs = _header.bitDataOffs;
	_header.letterMax    = 0;
	_header.maxWidth     = PIXELDATA_WIDTH;
	_header.maxHeight    = PIXELDATA_HEIGHT;
	_header.letterCharX  = PIXELDATA_WIDTH  / TILE_WIDTH;
	_header.letterCharY  = PIXELDATA_HEIGHT / TILE_HEIGHT;
}

PokemonFont::PokemonFont(const char *filename) {
	ifstream in;
	in.open(filename, ios::in|ios::binary);

	if (!in) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return;
	}

	in.read((char *)&_header, sizeof(FONTDATA_HEADER));

	_setLetterMax(_header.letterMax);

	in.seekg(_header.bitDataOffs, ios::beg);
	for (int i = 0; i < _header.letterMax; ++i) {
		in.read((char *)_fontCharacterVector.at(i).pixelData, PIXELDATA_ENTRYSIZE * sizeof(unsigned short));
	}
	in.seekg(_header.widthTblOffs, ios::beg);
	for (int i = 0; i < _header.letterMax; ++i) {
		in.read((char *)&_fontCharacterVector.at(i).width, sizeof(unsigned char));
	}

	in.close();

	WidthTableEntry widthTableEntry;
	widthTableEntry.maxCode = _header.letterMax;
	widthTableEntry.width   = WIDTH_PROPOTIONAL;
	_widthTableVector.push_back(widthTableEntry);
}

PokemonFont::~PokemonFont() {
}

void PokemonFont::_setLetterMax(int num) {
	if (_fontCharacterVector.size() < num) {
		FontCharacter fontCharacter;
		for (int i = 0; i < PIXELDATA_ENTRYSIZE; ++i) {
			fontCharacter.pixelData[i] = 0;
		}
		fontCharacter.width = 0;
		fontCharacter.marking = false;
		for (int i = _fontCharacterVector.size(); i < num; ++i) {
			_fontCharacterVector.push_back(fontCharacter);
		}

		_header.widthTblOffs = _header.bitDataOffs + num * PIXELDATA_ENTRYSIZE * sizeof(unsigned short);
		_header.letterMax    = num;
	}
}

void PokemonFont::_setPixel(int charCode, int x, int y, int color) {
	unsigned short *pixelData = _fontCharacterVector.at(charCode).pixelData;
	int index = (x / TILE_WIDTH + (y / TILE_HEIGHT) * (PIXELDATA_WIDTH / TILE_WIDTH)) * PIXELDATA_TILESIZE + y % TILE_HEIGHT;
	int shift = (x % TILE_WIDTH) * 2;

	pixelData[index] &= ~(0xC000 >> shift);
	pixelData[index] |= color << (14 - shift);
}

int PokemonFont::_getPixel(int charCode, int x, int y) {
	unsigned short *pixelData = _fontCharacterVector.at(charCode).pixelData;
	int index = (x / TILE_WIDTH + (y / TILE_HEIGHT) * (PIXELDATA_WIDTH / TILE_WIDTH)) * PIXELDATA_TILESIZE + y % TILE_HEIGHT;
	int shift = (x % TILE_WIDTH) * 2;

	return (pixelData[index] >> (14 - shift)) & 0x0003;
}

void PokemonFont::_drawCharacter(Bitmap2Color *bitmap2Color, int charCode, int top, int left, int color) {
	int width  = bitmap2Color->getWidth();
	int height = bitmap2Color->getHeight();

	for (int bmpY = 0, pokeY = top; bmpY < height; ++bmpY, ++pokeY) {
		for (int bmpX = 0, pokeX = left; bmpX < width; ++bmpX, ++pokeX) {
			if (bitmap2Color->getPixel(bmpX, bmpY)) {
				_setPixel(charCode, pokeX, pokeY, color);
			}
		}
	}
}

void PokemonFont::_addShadowForBold(Bitmap2Color *bitmap2Color, int charCode, int top, int left) {
	top++;
	left++;

	int width  = bitmap2Color->getWidth();
	int height = bitmap2Color->getHeight();

	for (int bmpY = 0, pokeY = top; bmpY < height; ++bmpY, ++pokeY) {
		for (int bmpX = 1, pokeX = left + 1; bmpX < width - 1; ++bmpX, ++pokeX) {
			if (!bitmap2Color->getPixel(bmpX, bmpY)) {
				if (bitmap2Color->getPixel(bmpX - 1, bmpY) && bitmap2Color->getPixel(bmpX + 1, bmpY)) {
					if ((bmpY < 1 || !(bitmap2Color->getPixel(bmpX - 1, bmpY - 1) && !bitmap2Color->getPixel(bmpX + 1, bmpY - 1))) &&
						(bmpY >= height - 1 || !(bitmap2Color->getPixel(bmpX - 1, bmpY + 1) && !bitmap2Color->getPixel(bmpX + 1, bmpY + 1)))) {
						_setPixel(charCode, pokeX, pokeY, COLOR_DARKSHADOW);
					}
				}
			}
		}
	}
}

int PokemonFont::_getLetterInfo(int charCode, LETTER_HEADER *letterHeader) {
	int y = 0;

	while (y < PIXELDATA_HEIGHT && _getPixel(charCode, 0, y) == 0) ++y;

	if (y < PIXELDATA_HEIGHT) {
		int x = 0;
		letterHeader->y = y;
		while (x < PIXELDATA_WIDTH  && _getPixel(charCode, x, y) != 0) x++;
		while (y < PIXELDATA_HEIGHT && _getPixel(charCode, 0, y) != 0) y++;
		letterHeader->height = y - letterHeader->y;
		return x;
	} else {
		letterHeader->y      = 0;
		letterHeader->height = 0;
		return 0;
	}
}

void PokemonFont::_setData(Bitmap2Color *bitmap2Color, int charCode, int top, bool bold, int width) {
	if (width == WIDTH_PROPOTIONAL) {
		width = bitmap2Color->getWidth() + 1;

		if (width == 1) {
			_fontCharacterVector.at(charCode).width = 0;
			return;
		}
	}

	if (bold) {
		_drawCharacter(bitmap2Color, charCode, top,     0, COLOR_DARKSHADOW);	// e
		_drawCharacter(bitmap2Color, charCode, top,     1, COLOR_DARKSHADOW);	// eiEj
		_drawCharacter(bitmap2Color, charCode, top,     2, COLOR_DARKSHADOW);	// eiEEj
		_drawCharacter(bitmap2Color, charCode, top + 1, 0, COLOR_DARKSHADOW);	// eij
		_drawCharacter(bitmap2Color, charCode, top + 1, 1, COLOR_LIGHTSHADOW);	// {́iEj
		_drawCharacter(bitmap2Color, charCode, top + 1, 2, COLOR_FONT);			// {́iEEj
		_addShadowForBold(bitmap2Color, charCode, top, 0);
	} else {
		// wi
		int height = bitmap2Color->getHeight() + 1;
		for (int y = top; y < top + height; ++y) {
			for (int x = 0; x < width; ++x) {
				_setPixel(charCode, x, y, COLOR_BACKGROUND);
			}
		}

		_drawCharacter(bitmap2Color, charCode, top,     1, COLOR_SHADOW);			// eiEj
		_drawCharacter(bitmap2Color, charCode, top + 1, 0, COLOR_SHADOW);			// eij
		_drawCharacter(bitmap2Color, charCode, top + 1, 1, COLOR_SHADOW);			// eiEj
		_drawCharacter(bitmap2Color, charCode, top,     0, COLOR_FONT);			// {
	}

	_fontCharacterVector.at(charCode).width = width;
}

int PokemonFont::saveFile(const char *filename) {
	ofstream out;

	out.open(filename, ios::out|ios::binary);

	if (!out) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	out.write((char *)&_header, sizeof(_header));
	for (int i = 0; i < _header.letterMax; ++i) {
		out.write((char *)_fontCharacterVector.at(i).pixelData, PIXELDATA_ENTRYSIZE * sizeof(unsigned short));
	}

	int widthAreaNum = _widthTableVector.size();

	{
		int widthTableSize = sizeof(int) + sizeof(WidthTableEntry) * widthAreaNum;

		int prevMaxCode = 0;
		for (int i = 0; i < widthAreaNum; ++i) {
			WidthTableEntry *entry = &_widthTableVector.at(i);
			if (entry->width == WIDTH_PROPOTIONAL) {
				widthTableSize += entry->maxCode - prevMaxCode;
			}
			prevMaxCode = entry->maxCode;
		}

		out.write((char *)&widthTableSize, sizeof(int));
	}

	out.write((char *)&widthAreaNum, sizeof(int));

	int prevMaxCode = 0;
	int widthTableOffset = sizeof(int) + sizeof(WidthTableEntry) * widthAreaNum;
	for (int i = 0; i < widthAreaNum; ++i) {
		WidthTableEntry *entry = &_widthTableVector.at(i);
		if (entry->width == WIDTH_PROPOTIONAL) {
			entry->offset = widthTableOffset;
			widthTableOffset += (entry->maxCode - prevMaxCode) * sizeof(unsigned char);
		} else {
			entry->offset = 0;
		}
		out.write((char *)entry, sizeof(WidthTableEntry));
		prevMaxCode = entry->maxCode;
	}
	prevMaxCode = 0;
	for (int i = 0; i < widthAreaNum; ++i) {
		WidthTableEntry *entry = &_widthTableVector.at(i);
		if (entry->width == WIDTH_PROPOTIONAL) {
			for (int i = prevMaxCode; i < entry->maxCode; ++i) {
				out.write((char *)&_fontCharacterVector.at(i).width, sizeof(unsigned char));
			}
		}
		prevMaxCode = entry->maxCode;
	}

	out.close();

	return 0;
}

typedef struct {
	LETTER_HEADER header;
	int bitDataSize;
	unsigned char *bitData;
} CompressedLetter;

int PokemonFont::saveCompressedFile(const char *filename) {
	ofstream out;

	out.open(filename, ios::out|ios::binary);

	if (!out) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	CompressedLetter *letters = new CompressedLetter[_header.letterMax];

	for (int i = 0; i < _header.letterMax; ++i) {
		int width = _getLetterInfo(i, &letters[i].header);
/*		cout << i << "(" << (int)letters[i].header.x << ", " << (int)letters[i].header.y << ") - ("
		     << letters[i].header.x + letters[i].header.width << ", "
		     << letters[i].header.y + letters[i].header.height << ")" << endl;
*/
		letters[i].bitDataSize = (width * letters[i].header.height + 3) / 4;		// 2 bits / pixel
		letters[i].bitData = new unsigned char[letters[i].bitDataSize];
		for (int j = 0; j < letters[i].bitDataSize; ++j) letters[i].bitData[j] = 0;

		int pixelIndex = 0;
		for (int y = letters[i].header.y; y < letters[i].header.y + letters[i].header.height; y++) {
			for (int x = 0; x < width; x++) {
				letters[i].bitData[pixelIndex/4] |= _getPixel(i, x, y) << ((pixelIndex % 4) * 2);
				pixelIndex++;
			}
		}
	}

	unsigned short *letterDataAddrTable = new unsigned short[_header.letterMax];

	letterDataAddrTable[0] = sizeof(unsigned short)								// entryNum
	                         + _header.letterMax * sizeof(unsigned short);		// letterDataAddrTable
	for (int i = 1; i < _header.letterMax; ++i) {
		letterDataAddrTable[i] = letterDataAddrTable[i-1] + sizeof(LETTER_HEADER) + letters[i-1].bitDataSize;
	}

	out.write((char *)&_header.letterMax, sizeof(unsigned short));		// little endian 

	out.write((char *)letterDataAddrTable, _header.letterMax * sizeof(unsigned short));
	for (int i = 0; i < _header.letterMax; ++i) {
		out.write((char *)&letters[i].header, sizeof(LETTER_HEADER));
		out.write((char *)letters[i].bitData, letters[i].bitDataSize);
	}

	out.close();

	delete[] letterDataAddrTable;
	for (int i = 0; i < _header.letterMax; ++i) delete[] letters[i].bitData;
	delete[] letters;

	return 0;
}

// DP̃tHgoCiǂݍ
int PokemonFont::importFile(const char *filename, int offset) {
	ifstream in;
	FONTDATA_HEADER header;

	in.open(filename, ios::in|ios::binary);

	if (!in) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	if (_header.letterMax > offset) {
		return -1;
	} else if (_header.letterMax < offset) {
		WidthTableEntry widthTableEntry;
		widthTableEntry.maxCode = offset;
		widthTableEntry.width   = 0;
		_widthTableVector.push_back(widthTableEntry);
	}

	in.read((char *)&header, sizeof(FONTDATA_HEADER));
	if ((header.letterCharX != PIXELDATA_WIDTH  / TILE_WIDTH) ||
	    (header.letterCharY != PIXELDATA_HEIGHT / TILE_HEIGHT)) {
		cout << "This font data format is not supported." << endl;
		cout << "maxWidth:    " << (int)header.maxWidth << endl;
		cout << "maxHeight:   " << (int)header.maxHeight << endl;
		cout << "letterCharX: " << (int)header.letterCharX << endl;
		cout << "letterCharY: " << (int)header.letterCharY << endl;
		return 1;
	}

	_setLetterMax(offset + header.letterMax);

	in.seekg(header.bitDataOffs, ios::beg);
	for (int i = 0; i < header.letterMax; ++i) {
		in.read((char *)_fontCharacterVector.at(i+offset).pixelData, PIXELDATA_ENTRYSIZE * sizeof(unsigned short));
	}
	in.seekg(header.widthTblOffs, ios::beg);
	for (int i = 0; i < header.letterMax; ++i) {
		in.read((char *)&_fontCharacterVector.at(i+offset).width, sizeof(unsigned char));
	}

	in.close();

	WidthTableEntry widthTableEntry;
	widthTableEntry.maxCode = offset + header.letterMax;
	widthTableEntry.width   = WIDTH_PROPOTIONAL;
	_widthTableVector.push_back(widthTableEntry);

	return 0;
}

int PokemonFont::importBitmap(FontBitmap *fontBitmap, int offset, int top, bool bold, int width) {
	if (_header.letterMax > offset) {
		return -1;
	} else if (_header.letterMax < offset) {
		WidthTableEntry widthTableEntry;
		widthTableEntry.maxCode = offset;
		widthTableEntry.width   = 0;
		_widthTableVector.push_back(widthTableEntry);
	}

	int num = fontBitmap->getEntryNum();

	_setLetterMax(offset + num);

	for (int i = 0; i < num; ++i) {
		Bitmap2Color bitmap2Color;
		fontBitmap->getCharBitmap(i, &bitmap2Color);
		_setData(&bitmap2Color, offset + i, top, bold, width);
	}

	if (bold) fixFont("fix_button.txt", offset, num);

	WidthTableEntry widthTableEntry;
	widthTableEntry.maxCode = offset + num;
	widthTableEntry.width   = width;
	_widthTableVector.push_back(widthTableEntry);

	return 0;
}

int PokemonFont::fixFont(const char *filename, int offset, int num) {
	ifstream in;
	in.open(filename, ios::in);

	if (!in) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	int patternNum;
	in >> patternNum;

	vector<FixPattern *> fixPatternVector;

	for (int i = 0; i < patternNum; ++i) {
		FixPattern *fixPattern = new FixPattern;
		in >> fixPattern->width;
		in >> fixPattern->height;
		in >> fixPattern->marking;
		fixPattern->from = new int[fixPattern->width * fixPattern->height];
		fixPattern->to   = new int[fixPattern->width * fixPattern->height];
//		cout << fixPattern->width << ", " << fixPattern->height << endl;
		int j = 0;
		for (int y = 0; y < fixPattern->height; ++y) {
			for (int x = 0; x < fixPattern->width; ++x) {
				in >> fixPattern->from[j++];
//				cout << fixPattern->from[j-1] << ", ";
			}
//			cout << endl;
		}
		j = 0;
		for (int y = 0; y < fixPattern->height; ++y) {
			for (int x = 0; x < fixPattern->width; ++x) {
				in >> fixPattern->to[j++];
//				cout << fixPattern->to[j-1] << ", ";
			}
//			cout << endl;
		}
		fixPatternVector.push_back(fixPattern);
	}

	in.close();

	for (int i = 0; i < fixPatternVector.size(); ++i) {
		FixPattern *fixPattern = fixPatternVector.at(i);
		for (int j = 0; j < _header.letterMax; ++j) {
			for (int top = 0; top < PIXELDATA_HEIGHT - fixPattern->height; ++top) {
				for (int left = 0; left < PIXELDATA_WIDTH - fixPattern->width; ++left) {
					bool match = true;
					int k = 0;
					for (int y = 0; y < fixPattern->height; ++y) {
						for (int x = 0; x < fixPattern->width; ++x) {
							int from = fixPattern->from[k++];
							if (from >= 0 && _getPixel(j, left+x, top+y) != from) {
								match = false;
								break;
							}
						}
						if (!match) break;
					}
					if (match) {
						k = 0;
						for (int y = 0; y < fixPattern->height; ++y) {
							for (int x = 0; x < fixPattern->width; ++x) {
								int to = fixPattern->to[k++];
								if (to >= 0) _setPixel(j, left+x, top+y, to);
							}
						}
						if (fixPattern->marking) _fontCharacterVector.at(j).marking = true;
//						cout << "Fixed: (" << j << ", " << top << ", " << left << ")" << endl;
					}
				}
			}
		}
	}

	for (int i = 0; i < fixPatternVector.size(); ++i) {
		FixPattern *fixPattern = fixPatternVector.at(i);
		delete[] fixPattern->from;
		delete[] fixPattern->to;
		delete   fixPattern;
	}

	return 0;
}

typedef struct {
	unsigned short padding;		// unsigned long ̗vf4oCgEɑ
	char           bfType[2];
	unsigned long  bfSize;
	unsigned short bfReserved1;
	unsigned short bfReserved2;
	unsigned long  bfOffBits;
} BITMAPFILEHEADEREX;

typedef struct {
	unsigned long  biSize;
	long           biWidth;
	long           biHeight;
	unsigned short biPlanes;
	unsigned short biBitCount;
	unsigned long  biCompression;
	unsigned long  biSizeImage;
	long           biXPixPerMeter;
	long           biYPixPerMeter;
	unsigned long  biClrUsed;
	unsigned long  biClrImporant;
} BITMAPINFOHEADER;

typedef struct {
	unsigned char rgbBlue;
	unsigned char rgbGreen;
	unsigned char rgbRed;
	unsigned char rgbReserved;
} RGBQUAD;


int PokemonFont::saveBitmap(const char *filename) {
	int bitmapLineNum    = (_header.letterMax + BITMAP_CHARACTERNUMLINE - 1) / BITMAP_CHARACTERNUMLINE;
	int bitmapHeight     = BLOCK_HEIGHT * bitmapLineNum;
	int bitmapImageSize  = BITMAP_WIDTH * bitmapHeight;
	int bitmapHeaderSize = sizeof(BITMAPFILEHEADEREX) - BITMAPFILEHEADEREX_PADDINGSIZE + sizeof(BITMAPINFOHEADER) + BITMAP_PALETTESIZE;
	int bitmapFileSize   = bitmapHeaderSize + bitmapImageSize;

	BITMAPFILEHEADEREX fileHeaderEx = {
		0,
		{'B', 'M'},
		bitmapFileSize,
		0,
		0,
		bitmapHeaderSize
	};

	BITMAPINFOHEADER   infoHeader = {
		BITMAPINFOHEADER_SIZE,
		BITMAP_WIDTH,
		bitmapHeight,
		BITMAP_PLANES,
		BITMAP_BITCOUNT,
		BITMAP_COMPRESSION,
		bitmapImageSize,
		BITMAP_XPIXPERMETER,
		BITMAP_YPIXPERMETER,
		BITMAP_CLRUSED,
		BITMAP_CLRIMPORTANT
	};

	RGBQUAD *paletteData = new RGBQUAD[BITMAP_CLRUSED];
	for (int i = 0; i < BITMAP_CLRUSED; ++i) {
		paletteData[i].rgbReserved = paletteData[i].rgbBlue =
		paletteData[i].rgbGreen    = paletteData[i].rgbRed  = 0;
	}
	paletteData[0].rgbBlue = paletteData[0].rgbGreen = paletteData[0].rgbRed = 255;//255;
	paletteData[1].rgbBlue = paletteData[1].rgbGreen = paletteData[1].rgbRed = 0;//  0;
	paletteData[2].rgbBlue = paletteData[2].rgbGreen = paletteData[2].rgbRed = 31;//191;
	paletteData[3].rgbBlue = paletteData[3].rgbGreen = paletteData[3].rgbRed = 191;//127;

	paletteData[255].rgbRed  = 255;
	paletteData[255].rgbBlue = paletteData[255].rgbGreen = 0;

	unsigned char *imageData = new unsigned char[bitmapImageSize];

	for (int i = 0; i < bitmapImageSize; ++i) imageData[i] = 0;

	for (int charCode = 0; charCode < _header.letterMax; ++charCode) {
		int blockLeft = charCode % BITMAP_CHARACTERNUMLINE * BLOCK_WIDTH;
		int blockTop  = charCode / BITMAP_CHARACTERNUMLINE * BLOCK_HEIGHT;
		// Og
		{
			int frameColor = _fontCharacterVector.at(charCode).marking ? 255 : 1;
			for (int bmpX = blockLeft; bmpX < blockLeft + BLOCK_WIDTH; ++bmpX) {
				int bmpY = blockTop;
				int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
				imageData[index] = frameColor;
			}
			for (int bmpX = blockLeft; bmpX < blockLeft + BLOCK_WIDTH; ++bmpX) {
				int bmpY = blockTop + BLOCK_HEIGHT - 1;
				int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
				imageData[index] = frameColor;
			}
			for (int bmpY = blockTop + 1; bmpY < blockTop + BLOCK_HEIGHT - 1; ++bmpY) {
				int bmpX = blockLeft;
				int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
				imageData[index] = frameColor;
			}
			for (int bmpY = blockTop + 1; bmpY < blockTop + BLOCK_HEIGHT - 1; ++bmpY) {
				int bmpX = blockLeft + BLOCK_WIDTH - 1;
				int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
				imageData[index] = frameColor;
			}
		}
		// g
		for (int bmpX = blockLeft + 1; bmpX < blockLeft + BLOCK_WIDTH - 1; ++bmpX) {
			int bmpY = blockTop + 1;
			int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
			imageData[index] = 3;
		}
		for (int bmpX = blockLeft + 1; bmpX < blockLeft + BLOCK_WIDTH - 1; ++bmpX) {
			int bmpY = blockTop + BLOCK_HEIGHT - 2;
			int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
			imageData[index] = 3;
		}
		for (int bmpY = blockTop + 2; bmpY < blockTop + BLOCK_HEIGHT - 2; ++bmpY) {
			int bmpX = blockLeft + 1;
			int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
			imageData[index] = 3;
		}
		for (int bmpY = blockTop + 2; bmpY < blockTop + BLOCK_HEIGHT - 2; ++bmpY) {
			int bmpX = blockLeft + BLOCK_WIDTH - 2;
			int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
			imageData[index] = 3;
		}
		for (int bmpX = blockLeft + 2; bmpX < blockLeft + BLOCK_WIDTH - 2; ++bmpX) {
			int bmpY = blockTop + BLOCK_HEIGHT - 4;
			int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
			imageData[index] = 3;
		}
		// 
		for (int pokeY = 0; pokeY < PIXELDATA_HEIGHT; ++pokeY) {
			for (int pokeX = 0; pokeX < PIXELDATA_WIDTH; ++pokeX) {
				int bmpX = blockLeft + pokeX + 2;
				int bmpY = blockTop  + pokeY + 2;
				int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
				unsigned char color = _getPixel(charCode, pokeX, pokeY);
//				if (color == COLOR_BACKGROUND) color = 0;
				imageData[index] = color;
			}
		}
		// 
		for (int bmpX = blockLeft + 2; bmpX < blockLeft + 2 + _fontCharacterVector.at(charCode).width; ++bmpX) {
			int bmpY = blockTop + BLOCK_HEIGHT - 3;
			int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
			imageData[index] = 1;
		}
	}
	{
		int bmpX = 0;
		int bmpY = 2 + PIXELDATA_HEIGHT - 1;
		int index = bmpX + BITMAP_WIDTH * (bitmapHeight - 1 - bmpY);
		imageData[index] = 0;
	}

	ofstream out;

	out.open(filename, ios::out|ios::binary);

	if (!out) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	out.write((char *)&fileHeaderEx + BITMAPFILEHEADEREX_PADDINGSIZE,
	          sizeof(BITMAPFILEHEADEREX) - BITMAPFILEHEADEREX_PADDINGSIZE);
	out.write((char *)&infoHeader, sizeof(BITMAPINFOHEADER));
	out.write((char *)paletteData, BITMAP_PALETTESIZE);
	out.write((char *)imageData,   bitmapImageSize);

	out.close();

	delete[] imageData;
	delete[] paletteData;

	return 0;
}

/*
int PokemonFont::saveBitmap(const char *filename) {
	int bitmapWidth      = PIXELDATA_WIDTH * BITMAP_CHARACTERNUMLINE;
	int bitmapLineNum    = (_header.letterMax + BITMAP_CHARACTERNUMLINE - 1) / BITMAP_CHARACTERNUMLINE;
	int bitmapHeight     = PIXELDATA_HEIGHT * bitmapLineNum;
	int bitmapImageSize  = bitmapWidth * bitmapHeight;
	int bitmapHeaderSize = sizeof(BITMAPFILEHEADEREX) - BITMAPFILEHEADEREX_PADDINGSIZE + sizeof(BITMAPINFOHEADER) + BITMAP_PALETTESIZE;
	int bitmapFileSize   = bitmapHeaderSize + bitmapImageSize;

	BITMAPFILEHEADEREX fileHeaderEx = {
		0,
		{'B', 'M'},
		bitmapFileSize,
		0,
		0,
		bitmapHeaderSize
	};

	BITMAPINFOHEADER   infoHeader = {
		BITMAPINFOHEADER_SIZE,
		bitmapWidth,
		bitmapHeight,
		BITMAP_PLANES,
		BITMAP_BITCOUNT,
		BITMAP_COMPRESSION,
		bitmapImageSize,
		BITMAP_XPIXPERMETER,
		BITMAP_YPIXPERMETER,
		BITMAP_CLRUSED,
		BITMAP_CLRIMPORTANT
	};

	RGBQUAD *paletteData = new RGBQUAD[BITMAP_CLRUSED];
	for (int i = 0; i < BITMAP_CLRUSED; ++i) {
		paletteData[i].rgbReserved = paletteData[i].rgbBlue =
		paletteData[i].rgbGreen    = paletteData[i].rgbRed  = 0;
	}
	paletteData[ 0].rgbRed = 156; paletteData[ 0].rgbGreen = 197; paletteData[ 0].rgbBlue = 107;
	paletteData[ 1].rgbRed =  90; paletteData[ 1].rgbGreen =  90; paletteData[ 1].rgbBlue =  82;
	paletteData[ 2].rgbRed = 173; paletteData[ 2].rgbGreen = 189; paletteData[ 2].rgbBlue = 189;
	paletteData[15].rgbRed = 255; paletteData[15].rgbGreen = 255; paletteData[15].rgbBlue = 255;

	unsigned char *imageData = new unsigned char[bitmapImageSize];

	for (int i = 0; i < bitmapImageSize; ++i) imageData[i] = 0;

	for (int charCode = 0; charCode < _header.letterMax; ++charCode) {
		int blockLeft = charCode % BITMAP_CHARACTERNUMLINE * PIXELDATA_WIDTH;
		int blockTop  = charCode / BITMAP_CHARACTERNUMLINE * PIXELDATA_HEIGHT;
		// 
		for (int pokeY = 0; pokeY < PIXELDATA_HEIGHT; ++pokeY) {
			for (int pokeX = 0; pokeX < PIXELDATA_WIDTH; ++pokeX) {
				int bmpX = blockLeft + pokeX;
				int bmpY = blockTop  + pokeY;
				int index = bmpX + bitmapWidth * (bitmapHeight - 1 - bmpY);
				unsigned char color = _getPixel(charCode, pokeX, pokeY);
				switch (color) {
					case COLOR_NONE:		color =  0; break;
					case COLOR_FONT:		color =  1; break;
					case COLOR_SHADOW:		color =  2; break;
					case COLOR_BACKGROUND:	color = 15; break;
				}
				imageData[index] = color;
			}
		}
	}

	ofstream out;

	out.open(filename, ios::out|ios::binary);

	if (!out) {
		cout << "\"" << filename << "\" could not be opened." << endl;
		return 1;
	}

	out.write((char *)&fileHeaderEx + BITMAPFILEHEADEREX_PADDINGSIZE,
	          sizeof(BITMAPFILEHEADEREX) - BITMAPFILEHEADEREX_PADDINGSIZE);
	out.write((char *)&infoHeader, sizeof(BITMAPINFOHEADER));
	out.write((char *)paletteData, BITMAP_PALETTESIZE);
	out.write((char *)imageData,   bitmapImageSize);

	out.close();

	delete[] imageData;
	delete[] paletteData;

	return 0;
}
*/
