//============================================================================================
/**
 * @file	fnote_mem.c
 * @brief	`m[g@f[^Ǘ
 * @author	Hiroyuki Nakamura
 * @date	2006.03.06
 */
//============================================================================================
#include "common.h"
#include "system/savedata.h"
#include "system/buflen.h"
#include "system/pm_rtc.h"
#include "savedata/mystatus.h"
#include "msgdata/msg_place_name.h"
#include "battle/trno_def.h"
#include "field/field_place_name.h"
#include "../field/mapdefine.h"
#include "../field/zonedata.h"
#include "../field/sysflag.h"

#define	FNOTE_MEM_H_GLOBAL
#include "savedata/fnote_mem.h"


//============================================================================================
//	萔`
//============================================================================================

// Z[uf[^{
struct _FNOTE_DATA {
	FNOTE_WK_START	hed;			// Jnf[^
// 4
	u32	action[FNOTE_ACTION_MAX];	// s
// 4+16
	FNOTE_WK_POKE	pokemon;		// |P
// 4+16+4
	FNOTE_WK_TRAINER trainer;		// g[i[
// 4+16+4+4
	u8	sio[FNOTE_SIO_MAX][42];		// ʐM
// 4+16+4+4+84 = 112
};

// Ȍf[^
typedef struct {
	u32	name;		// OID
	u32	type;		// \^Cv
}EX_BUILD_DATA;

// We[u
typedef struct {
	u32	tr;			// g[i[ID
	u16	zone;		// ][ID
	u16	badge;		// obWtO
}GYM_TBL;


//============================================================================================
//	vg^Cv錾
//============================================================================================
static void FNOTE_StartDataSave( FNOTE_DATA * dat, FNOTE_WK_START * wk );
static void FNOTE_ActionDataSave( FNOTE_DATA * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_PokemonDataSave( FNOTE_DATA * dat, FNOTE_WK_POKE * wk );
static void FNOTE_TrainerDataSave( FNOTE_DATA * dat, FNOTE_WK_TRAINER * wk );
static void FNOTE_SioDataSave( FNOTE_DATA * dat, FNOTE_WK_SIO * wk );

static u32 * FNOTE_ActionSavePosGet( u32 * action );
static u8 * FNOTE_SioSavePosGet( FNOTE_DATA * dat );
static void FNOTE_ActionIDDataSave( u32 * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionTrainerDataSave( u32 * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionSitennouDataSave( u32 * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionGymLeaderDataSave( u32 * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionMapDataSave( u32 * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionItemDataSave( u32 * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionSkillDataSave( u32 * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_SioBattleDataSave( u8 * dat, FNOTE_WK_SIO * wk );
static void FNOTE_SioUnionGreetDataSave( u8 * dat, FNOTE_WK_SIO * wk );
static void FNOTE_SioPokeChgDataSave( u8 * dat, FNOTE_WK_SIO * wk );
static void FNOTE_SioIDDataSave( FNOTE_DATA * dat, FNOTE_WK_SIO * wk );
static void FNOTE_SioContestDataSave( u8 * dat, FNOTE_WK_SIO * wk );

static void FNOTE_StrCopy( const STRCODE * str, STRCODE * buf, u32 siz );

static void FNOTE_StartDataGet( FNOTE_DATA * dat, FNOTE_WK_START * wk );
static void FNOTE_ActionDataGet( FNOTE_DATA * dat, FNOTE_WK_ACTION * wk );
static void FNOTE_PokemonDataGet( FNOTE_DATA * dat, FNOTE_WK_POKE * wk );
static void FNOTE_TrainerDataGet( FNOTE_DATA * dat, FNOTE_WK_TRAINER * wk );
static void FNOTE_SioDataGet( FNOTE_DATA * dat, FNOTE_WK_SIO * wk );

static void FNOTE_ActionIDDataGet( u32 dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionTrainerDataGet( u32 dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionMapDataGet( u32 dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionItemDataGet( u32 dat, FNOTE_WK_ACTION * wk );
static void FNOTE_ActionSkillDataGet( u32 dat, FNOTE_WK_ACTION * wk );
static void FNOTE_SioBattleDataGet( u8 * dat, FNOTE_WK_SIO * wk );
static void FNOTE_SioUnionGreetDataGet( u8 * dat, FNOTE_WK_SIO * wk );
static void FNOTE_SioPokeChgDataGet( u8 * dat, FNOTE_WK_SIO * wk );
static void FNOTE_SioIDDataGet( u8 * dat, FNOTE_WK_SIO * wk );
static void FNOTE_SioContestDataGet( u8 * dat, FNOTE_WK_SIO * wk );

static u8 GymCheck( MYSTATUS * my, u32 zone_id );
static u8 TrainerCheck( u32 tr );


//============================================================================================
//	O[oϐ
//============================================================================================
// Ȍe[u
static const EX_BUILD_DATA BuildNameTable[] =
{
	{ MAPNAME_C1TV,				1 },	// er@RguL
	{ MAPNAME_C1POKETCH,		0 },	// |Pb`Jpj[
	{ MAPNAME_C1GTC,			1 },	// fsb
	{ MAPNAME_C1SCHOOL,			1 },	// g[i[YXN[
	{ MAPNAME_C3HAKUBUTUKAN,	0 },	// 񂱂͂Ԃ
	{ MAPNAME_T3FLOWER,			1 },	// t[@Vv
	{ MAPNAME_C4BICYCLE,		1 },	// TCN@Vbv
	{ MAPNAME_C5CONTEST,		1 },	// ReXg傤
	{ MAPNAME_C5POFFINN,		1 },	// |tBnEX
//	{ MAPNAME_D11HIROBA,		0 },	// ӂꂠЂ
	{ MAPNAME_C5BUILDING,		0 },	// Ԃ񂩂̂Ă
	{ MAPNAME_T4SODATEYA,		1 },	// |PĂ
	{ MAPNAME_C7DEPARTMENT,		0 },	// gofp[g
	{ MAPNAME_C7GAME,			1 },	// Q[R[i[
	{ MAPNAME_D26AZITO,			0 },	// MKAWg
	{ MAPNAME_C7TOSYOKAN,		0 },	// ~I@Ƃ傩
	{ MAPNAME_D20KISSAKI,		0 },	// LbTLł
	{ MAPNAME_C8TODAI,			0 },	// Vx́@Ƃ
	{ MAPNAME_C8ITIBA,			1 },	// iMT@
	{ MAPNAME_D31TOWER,			1 },	// og^[
	{ MAPNAME_R212YASIKI,		0 },	// |P₵
	{ MAPNAME_R213ASIATO,		1 },	// Ƃ͂̂
//	{ MAPNAME_D10POKEPARK,		1 },	// |Pp[N
	{ MAPNAME_R210KISSATEN,		0 },	// Ă
	{ MAPNAME_L02HOTEL,			0 },	// OhCN
	{ MAPNAME_L02RESTAURANT,	0 },	// Xg
	{ MAPNAME_D2HATUDEN,		0 },	// ɂ܂̂͂ł񂵂
	{ MAPNAME_D4TATARA,			0 },	// ^^Ă
	{ MAPNAME_D25YOUKAN,		0 }		// ̂悤
};

// We[u
static const GYM_TBL GymTbl[] =
{
	{ LEADER1_01, ZONE_ID_C03GYM0101, BADGE_ID_C03 },	// NKlA킭
	{ LEADER2_01, ZONE_ID_C04GYM0101, BADGE_ID_C04 },	// nN^CA
	{ LEADER3_01, ZONE_ID_C06GYM0101, BADGE_ID_C06 },	// mZA΂炢
	{ LEADER4_01, ZONE_ID_C07GYM0101, BADGE_ID_C07 },	// goAƂ
	{ LEADER5_01, ZONE_ID_C05GYM0101, BADGE_ID_C05 },	// XKAȂ݂̂
	{ LEADER6_01, ZONE_ID_C09GYM0101, BADGE_ID_C09 },	// LbTLAbNNC
	{ LEADER7_01, ZONE_ID_C02GYM0101, BADGE_ID_C02 },	// ~IA肫
	{ LEADER8_01, ZONE_ID_C08GYM0101, BADGE_ID_C08 }	// iMTÂڂ
};


//--------------------------------------------------------------------------------------------
/**
 * `m[gp[ÑTCY擾
 *
 * @param	none
 *
 * @return	[NTCY
 */
//--------------------------------------------------------------------------------------------
int FNOTE_GetWorkSize(void)
{
	return ( sizeof(FNOTE_DATA) * FNOTE_PAGE_MAX );
}

//--------------------------------------------------------------------------------------------
/**
 * `m[gp[N
 *
 * @param	wk		[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_Init( FNOTE_DATA * dat )
{
	memset( dat, 0, sizeof(FNOTE_DATA) );
}

//--------------------------------------------------------------------------------------------
/**
 * `m[gp[NiŚj
 *
 * @param	wk		[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
void FNOTE_InitAll( FNOTE_DATA * dat )
{
	memset( dat, 0, sizeof(FNOTE_DATA)*FNOTE_PAGE_MAX );
}

//--------------------------------------------------------------------------------------------
/**
 * Z[uf[^`m[g̃|C^擾
 *
 * @param	sv		Z[uf[^
 *
 * @return	`m[gf[^
 */
//--------------------------------------------------------------------------------------------
FNOTE_DATA * SaveData_GetFNote( SAVEDATA * sv )
{
	return SaveData_Get( sv, GMDATA_ID_FNOTE );
}

//--------------------------------------------------------------------------------------------
/**
 * Z[uy[W擾
 *
 * @param	dat		Z[uf[^
 * @param	flg		Z[utO ( TRUE =  )
 *
 * @retval	"flg = TRUE : Z[uy[W"
 * @retval	"flg = FALSE : NULL"
 */
//--------------------------------------------------------------------------------------------
FNOTE_DATA * FNOTE_SavePageGet( FNOTE_DATA * dat, BOOL flg )
{
	RTCDate	rtc;
	s32	i;

	// Z[u`FbN
	if( flg != TRUE ){ return NULL; }

	GF_RTC_GetDate( &rtc );

	if( dat[0].hed.month != 0 &&
		( dat[0].hed.year != rtc.year || dat[0].hed.month != rtc.month ||
		  dat[0].hed.day != rtc.day || dat[0].hed.week != rtc.week ) ){

		// f[^Vtg
		for( i=FNOTE_PAGE_MAX-1; i>=1; i-- ){
			dat[i] = dat[i-1];
		}
		FNOTE_Init( &dat[0] );
	}
	return &dat[0];


/*
	// tvf[^擾
	GF_RTC_GetDate( &rtc );
	for( i=0; i<FNOTE_PAGE_MAX; i++ ){
		if( dat[i].hed.year == rtc.year && dat[i].hed.month == rtc.month &&
			dat[i].hed.day == rtc.day && dat[i].hed.week == rtc.week ){

			return &dat[i];
		}
	}

	// f[^ꏊ擾
	for( i=0; i<FNOTE_PAGE_MAX; i++ ){
		if( dat[i].hed.month == 0 ){
			return &dat[i];
		}
	}

	// 󂫂ꍇ́AVtgČ̃f[^擾
	for( i=0; i<FNOTE_PAGE_MAX-1; i++ ){
		dat[i] = dat[i+1];
	}
	FNOTE_Init( &dat[i] );
	return &dat[i];
*/
}

//--------------------------------------------------------------------------------------------
/**
 * Q[Jn̖`m[g\`FbN
 *
 * @param	dat		Z[uf[^
 * @param	flg		Z[utO ( TRUE =  )
 *
 * @retval	"TRUE = \"
 * @retval	"FALSE = \Ȃ"
 */
//--------------------------------------------------------------------------------------------
BOOL FNOTE_GameStartOpenCheck( FNOTE_DATA * dat, BOOL flg )
{
	RTCDate	note;
	RTCDate	now;
	int	ofs;

	// Z[u`FbN
	if( flg != TRUE ){ return FALSE; }

	// ݂̎
	GF_RTC_GetDate( &now );
	// `m[g̍ŐV̎
	note.year  = dat[0].hed.year;
	note.month = dat[0].hed.month;
	note.day   = dat[0].hed.day;
	note.week  = dat[0].hed.week;

	ofs = GF_RTC_GetDaysOffset(&now) - GF_RTC_GetDaysOffset(&note);
// ----------------------------------------------------------------------------
// localize_spec_mark(JP_VER10) imatake 2006/12/01
// ڂm[g\Ȃꍇ̂C
#if	S2423_060817_FIX	// C
	// 12/31 <-> 1/1 ̏ꍇ́A2Nȏ
	if( (now.month==12 && now.day==31 && note.month==1 && note.day==1) ||
		(now.month==1 && now.day==1 && note.month==12 && note.day==31) ){
		s32	year = (s32)now.year - (s32)note.year;
		if( year >= 2 || year <= -2 ){
			return TRUE;
		}
	// 2ȏ̍
	}else if( ofs <= -2 || ofs >= 2 ){
		return TRUE;
	// ȊO͔NႦok
	}else{
		if( now.year != note.year ){
			return TRUE;
		}
	}
#else					// CO
	if( ofs == 0 ){
		if( now.year != note.year ){
			return TRUE;
		}
	}else if( ofs <= -2 ){
		return TRUE;
	}else if( ofs >= 2 ){
		return TRUE;
	}
#endif
// ----------------------------------------------------------------------------
	return FALSE;
}


//============================================================================================
//	f[^Z[u
//============================================================================================

//--------------------------------------------------------------------------------------------
/**
 * zZ[u
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[uf[^
 * @param	type	L^^Cv
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
void FNOTE_DataSave( FNOTE_DATA * dat, void * wk, u8 type )
{
	if( dat != NULL ){
		switch( type ){
		case FNOTE_TYPE_START:		// Jn
			FNOTE_StartDataSave( dat, wk );
			break;

		case FNOTE_TYPE_ACTION:		// s
			FNOTE_ActionDataSave( dat, wk );
			break;

		case FNOTE_TYPE_POKEMON:	// |P
			FNOTE_PokemonDataSave( dat, wk );
			break;

		case FNOTE_TYPE_TRAINER:	// g[i[
			FNOTE_TrainerDataSave( dat, wk );
			break;

		case FNOTE_TYPE_SIO:		// ʐM
			FNOTE_SioDataSave( dat, wk );
			break;
		}
	}
	sys_FreeMemoryEz( wk );
}

//--------------------------------------------------------------------------------------------
/**
 * Jnf[^Z[u
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[uf[^
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_StartDataSave( FNOTE_DATA * dat, FNOTE_WK_START * wk )
{
	if( dat->hed.year == wk->year && dat->hed.month == wk->month &&
		dat->hed.day == wk->day && dat->hed.week == wk->week ){
		return;
	}
	dat->hed = *wk;
}

//--------------------------------------------------------------------------------------------
/**
 * sf[^Z[u
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[uf[^
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionDataSave( FNOTE_DATA * dat, FNOTE_WK_ACTION * wk )
{
	switch( wk->id ){
	case FNOTE_ID_MYHOUSE:			// sF
	case FNOTE_ID_LABO:				// sF
	case FNOTE_ID_POKECENTER:		// sF|PZ^[
	case FNOTE_ID_FDSHOP_BUY:		// sFVbvE
	case FNOTE_ID_FDSHOP_BUY2:		// sFVbvEij
	case FNOTE_ID_FDSHOP_SALE:		// sFVbvE
	case FNOTE_ID_FDSHOP_SALE2:		// sFVbvEij
	case FNOTE_ID_FDSHOP_TRADE:		// sFVbvE
	case FNOTE_ID_GAMECORNER:		// sFQ[R[i[
	case FNOTE_ID_SAFARIZONE:		// sFTt@][
	case FNOTE_ID_UG_FOSSIL:		// sFnEΌ@
	case FNOTE_ID_UG_BASE:			// sFnE閧n
		FNOTE_ActionIDDataSave( dat->action, wk );
		break;

	case FNOTE_ID_GYMBEFORE:		// sFWi[_[|Oj
	case FNOTE_ID_GYMAFTER:			// sFWi[_[|j
		FNOTE_ActionGymLeaderDataSave( dat->action, wk );
		break;

	case FNOTE_ID_SITENNOU:			// sFlV
		FNOTE_ActionSitennouDataSave( dat->action, wk );
		break;

	case FNOTE_ID_CHAMPION:			// sF`sI
		FNOTE_ActionTrainerDataSave( dat->action, wk );
		break;

	case FNOTE_ID_CITY:				// sFX
	case FNOTE_ID_DUNGEON:			// sF_W
	case FNOTE_ID_BUILD:			// sF
		FNOTE_ActionMapDataSave( dat->action, wk );
		break;

	case FNOTE_ID_ITEMGET:			// sFACe擾
		FNOTE_ActionItemDataSave( dat->action, wk );
		break;

	case FNOTE_ID_IAIGIRI:			// sFZE
	case FNOTE_ID_SORAWOTOBU:		// sFZEƂ
	case FNOTE_ID_NAMINORI:			// sFZEȂ݂̂
	case FNOTE_ID_KAIRIKI:			// sFZE肫
	case FNOTE_ID_FLASH:			// sFZEtbV
	case FNOTE_ID_KIRIBARAI:		// sFZE΂炢
	case FNOTE_ID_IWAKUDAKI:		// sFZE킭
	case FNOTE_ID_TAKINOBORI:		// sFZÊڂ
	case FNOTE_ID_ROCKCLIMB:		// sFZEbNNC
	case FNOTE_ID_ANAWOHORU:		// sFZEȂق
	case FNOTE_ID_AMAIKAORI:		// sFZE܂
	case FNOTE_ID_TELEPORT:			// sFZEe|[g
	case FNOTE_ID_TAMAGOUMI:		// sFZE^}S
	case FNOTE_ID_MIRUKUNOMI:		// sFZE~N̂
		FNOTE_ActionSkillDataSave( dat->action, wk );
		break;
	}
}

//--------------------------------------------------------------------------------------------
/**
 * |Pf[^Z[u
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[uf[^
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_PokemonDataSave( FNOTE_DATA * dat, FNOTE_WK_POKE * wk )
{
	dat->pokemon = *wk;
}

//--------------------------------------------------------------------------------------------
/**
 * g[i[f[^Z[u
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[uf[^
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_TrainerDataSave( FNOTE_DATA * dat, FNOTE_WK_TRAINER * wk )
{
	dat->trainer = *wk;
}

//--------------------------------------------------------------------------------------------
/**
 * ʐMf[^Z[u
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[uf[^
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioDataSave( FNOTE_DATA * dat, FNOTE_WK_SIO * wk )
{
	u8 * sio = FNOTE_SioSavePosGet( dat );

	switch( wk->id ){
	case FNOTE_ID_SB_SINGLE:		// ʐMFʐMΐEVO
	case FNOTE_ID_SB_DOUBLE:		// ʐMFʐMΐE_u
	case FNOTE_ID_SB_MULTI:			// ʐMFʐMΐE}`
	case FNOTE_ID_SB_MIX1:			// ʐMFʐMΐE~bNX1vs1
	case FNOTE_ID_SB_MIX2:			// ʐMFʐMΐE~bNX2vs2
	case FNOTE_ID_U_BATTLE:			// ʐMFjI[Eΐ
		FNOTE_SioBattleDataSave( sio, wk );
		break;

	case FNOTE_ID_U_GREETING:		// ʐMFjI[EA
		FNOTE_SioUnionGreetDataSave( sio, wk );
		break;

	case FNOTE_ID_U_POKECHG:		// ʐMFjI[E
	case FNOTE_ID_U_MLTCHG:			// ʐMFjI[El
	case FNOTE_ID_GTC:				// ʐMFGTC
		FNOTE_SioPokeChgDataSave( sio, wk );
		break;

	case FNOTE_ID_U_PICTURE:		// ʐMFjI[EG
	case FNOTE_ID_U_TRANSCEIVER:	// ʐMFjI[EgV[o[
	case FNOTE_ID_RECORD:			// ʐMFR[hR[i[
	case FNOTE_ID_PORUTO:			// ʐMF|g쐬
	case FNOTE_ID_BTLTOWER:			// ʐMFog^[
		FNOTE_SioIDDataSave( dat, wk );
		break;

	case FNOTE_ID_CONTEST:			// ʐMFReXg
		FNOTE_SioContestDataSave( sio, wk );
		break;
	}
}

//--------------------------------------------------------------------------------------------
/**
 * sf[^̃Z[uʒu擾
 *
 * @param	action	Z[uf[^
 *
 * @return	Z[uʒu
 */
//--------------------------------------------------------------------------------------------
static u32 * FNOTE_ActionSavePosGet( u32 * action )
{
	u32	i;

	for( i=0; i<FNOTE_ACTION_MAX; i++ ){
		if( ( action[i] & 0x3f ) == 0 ){
			return &action[i];
		}
	}

	for( i=0; i<FNOTE_ACTION_MAX-1; i++ ){
		action[i]  = action[i+1];
	}
	action[i] = 0;
	return &action[i];
}

//--------------------------------------------------------------------------------------------
/**
 * ʐMf[^̃Z[uʒu擾
 *
 * @param	dat		Z[uf[^
 *
 * @return	Z[uʒu
 */
//--------------------------------------------------------------------------------------------
static u8 * FNOTE_SioSavePosGet( FNOTE_DATA * dat )
{
	u32	i;

	for( i=0; i<FNOTE_SIO_MAX; i++ ){
		if( dat->sio[i][0] == 0 ){
			return &dat->sio[i][0];
		}
	}

	for( i=0; i<FNOTE_SIO_MAX-1; i++ ){
		memcpy( &dat->sio[i][0], &dat->sio[i+1][0], 42 );
	}
	memset( &dat->sio[i][0], 0, 42 );
	return &dat->sio[i][0];
}


//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ sFID݂̂̂ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 *
 * @li	*dat = ( id & 0x3f )
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionIDDataSave( u32 * dat, FNOTE_WK_ACTION * wk )
{
	u32 * buf;
	u32	i;

	for( i=0; i<FNOTE_ACTION_MAX; i++ ){
		if( dat[i] == 0 ){
			break;
		}
	}
	if( i != 0 ){
		if( (dat[i-1]&0x3f) == wk->id ){ return; }
	}

	buf = FNOTE_ActionSavePosGet( dat );
	*buf = ( wk->id & 0x3f );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ sF`sI ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 *
 * @li	*dat = (map << 16) | ((trainer & 0x3ff) << 6) | (id & 0x3f)
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionTrainerDataSave( u32 * dat, FNOTE_WK_ACTION * wk )
{
	u32 * buf = FNOTE_ActionSavePosGet( dat );
	*buf = (wk->map << 16) | ((wk->trainer & 0x3ff) << 6) | (wk->id & 0x3f);
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ sFlV ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 *
 * @li	*dat = (map << 16) | ((trainer & 0x3ff) << 6) | (id & 0x3f)
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionSitennouDataSave( u32 * dat, FNOTE_WK_ACTION * wk )
{
	u32 * buf;
	u16	map;
	u16	i;

	buf = NULL;

	for( i=0; i<FNOTE_ACTION_MAX; i++ ){
		if( ( dat[i] & 0x3f ) == FNOTE_ID_SITENNOU ){
			dat[i] = 0;
			for( i=i; i<FNOTE_ACTION_MAX-1; i++ ){
				dat[i] = dat[i+1];
			}
			break;
		}
	}

	buf = FNOTE_ActionSavePosGet( dat );
	*buf = (wk->map << 16) | ((wk->trainer & 0x3ff) << 6) | (wk->id & 0x3f);
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ sFW[_[ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 *
 * @li	*dat = (map << 16) | ((trainer & 0x3ff) << 6) | (id & 0x3f)
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionGymLeaderDataSave( u32 * dat, FNOTE_WK_ACTION * wk )
{
	u32 * buf;
	u16	map;
	u16	i;

	buf = NULL;

	for( i=0; i<FNOTE_ACTION_MAX; i++ ){
		if( (dat[i]&0x3f) == FNOTE_ID_GYMBEFORE && ((dat[i]>>16)&0xffff) == wk->map ){
			dat[i] = 0;
			for( i=i; i<FNOTE_ACTION_MAX-1; i++ ){
				dat[i] = dat[i+1];
			}
			break;
		}
	}

	buf = FNOTE_ActionSavePosGet( dat );
	*buf = (wk->map << 16) | ((wk->trainer & 0x3ff) << 6) | (wk->id & 0x3f);
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ sFXA_WAȂ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 *
 * @li	*dat = (map << 16) | ( id & 0x3f )
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionMapDataSave( u32 * dat, FNOTE_WK_ACTION * wk )
{
	u32 * buf;
	u32	i;

	for( i=0; i<FNOTE_ACTION_MAX; i++ ){
		if( dat[i] == 0 ){ break; }
	}
	if( i != 0 ){
		if( (dat[i-1]&0x3f) == wk->id && (dat[i-1]>>16) == wk->map ){
			return;
		}
	}
	buf = FNOTE_ActionSavePosGet( dat );
	*buf = (wk->map << 16) | (wk->id & 0x3f);
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ sFACe擾 ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 *
 * @li	*dat = (item << 16 ) | ( id & 0x3f )
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionItemDataSave( u32 * dat, FNOTE_WK_ACTION * wk )
{
	u32 * buf;
	u32	i;

	for( i=0; i<FNOTE_ACTION_MAX; i++ ){
		if( dat[i] == 0 ){ break; }
	}
	if( i != 0 ){
		if( (dat[i-1]&0x3f) == wk->id && (dat[i-1]>>16) == wk->item ){
			return;
		}
	}
	buf = FNOTE_ActionSavePosGet( dat );
	*buf = (wk->item << 16) | (wk->id & 0x3f);
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ sFZ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 *
 * @li	*dat = ( id & 0x3f )
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionSkillDataSave( u32 * dat, FNOTE_WK_ACTION * wk )
{
	u32 * buf;
	u32	i;

	for( i=0; i<FNOTE_ACTION_MAX; i++ ){
		if( dat[i] == 0 ){ break; }
	}
	if( i != 0 ){
		if( (dat[i-1]&0x3f) == wk->id && (dat[i-1]>>16) == wk->map ){
			return;
		}
	}
	buf = FNOTE_ActionSavePosGet( dat );
	*buf = (wk->map << 16) | (wk->id & 0x3f);
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ ʐMFΐ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioBattleDataSave( u8 * dat, FNOTE_WK_SIO * wk )
{
	dat[0] = wk->id;
	dat[1] = (wk->result<<2) | (wk->sex1<<1) | (wk->sex2);
	FNOTE_StrCopy( wk->name1, (STRCODE *)&dat[2], 8 );
	FNOTE_StrCopy( wk->name2, (STRCODE *)&dat[18], 8 );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ ʐMFjI[EA ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioUnionGreetDataSave( u8 * dat, FNOTE_WK_SIO * wk )
{
	dat[0] = wk->id;
	dat[1] = wk->sex1;
	FNOTE_StrCopy( wk->name1, (STRCODE *)&dat[2], 8 );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ ʐMF ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioPokeChgDataSave( u8 * dat, FNOTE_WK_SIO * wk )
{
	dat[0] = wk->id;
	dat[1] = (wk->sex1 << 2) | (wk->poke_sex);
	FNOTE_StrCopy( wk->name1, (STRCODE *)&dat[2], 8 );
	FNOTE_StrCopy( wk->nickname, (STRCODE *)&dat[18], 12 );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ ʐMFog^[ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioIDDataSave( FNOTE_DATA * dat, FNOTE_WK_SIO * wk )
{
	u32	i;
	u8 * buf;

	for( i=0; i<FNOTE_SIO_MAX; i++ ){
		if( dat->sio[i][0] == 0 ){
			break;
		}
	}
	if( i != 0 ){
		if( dat->sio[i-1][0] == wk->id ){ return; }
	}
	buf = FNOTE_SioSavePosGet( dat );
	buf[0] = wk->id;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^Z[u [ ʐMFReXg ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		Z[u郏[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioContestDataSave( u8 * dat, FNOTE_WK_SIO * wk )
{
	dat[0] = wk->id;
	dat[1] = wk->result;
}


//============================================================================================
//	f[^쐬
//============================================================================================

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ tEJnꏊ ]
 *
 * @param	start	Jnꏊ
 * @param	heap	q[vID
 *
 * @return	쐬[N
 *
 * @li	tRTC擾
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_StartDataMake( u16 start, u32 heap )
{
	FNOTE_WK_START * wk;
	RTCDate	rtc;

	wk = sys_AllocMemory( heap, sizeof(FNOTE_WK_START) );

	GF_RTC_GetDate( &rtc );

	wk->year  = rtc.year;
	wk->month = rtc.month;
	wk->day   = rtc.day;
	wk->week  = rtc.week;
	wk->start = start;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * ځEsp[Nm
 *
 * @param	heap	q[vID
 *
 * @return	mۂ[N
 */
//--------------------------------------------------------------------------------------------
static FNOTE_WK_ACTION * FNOTE_ActionWorkAlloc( u32 heap )
{
	FNOTE_WK_ACTION * wk = sys_AllocMemory( heap, sizeof(FNOTE_WK_ACTION) );
	memset( wk, 0, sizeof(FNOTE_WK_ACTION) );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sF ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionHouseDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_MYHOUSE;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sF ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionLaboDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_LABO;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sF|PZ^[ ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_PokemonCenterDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_POKECENTER;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFVbvE ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionShopBuyDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_FDSHOP_BUY;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFVbvEij ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionShopBuy2DataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_FDSHOP_BUY2;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFVbvE ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionShopSaleDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_FDSHOP_SALE;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFVbvEij ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionShopSale2DataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_FDSHOP_SALE2;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFVbvE ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionShopTradeDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_FDSHOP_TRADE;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFWi[_[|Oj]
 *
 * @param	map		}bvID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionGymBeforeDataMake( u16 map, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id  = FNOTE_ID_GYMBEFORE;
	wk->map = map;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFWi[_[|j ]
 *
 * @param	map		}bvID
 * @param	trainer	g[i[ID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionGymAfterDataMake( u16 map, u16 trainer, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id      = FNOTE_ID_GYMAFTER;
	wk->map     = map;
	wk->trainer = trainer;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFlV ]
 *
 * @param	trainer	g[i[ID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionSitenouDataMake( u16 trainer, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id      = FNOTE_ID_SITENNOU;
	wk->trainer = trainer;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sF`sI ]
 *
 * @param	trainer	g[i[ID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionChampionDataMake( u16 trainer, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id      = FNOTE_ID_CHAMPION;
	wk->trainer = trainer;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFX ]
 *
 * @param	map		}bvID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionCityDataMake( u16 map, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id  = FNOTE_ID_CITY;
	wk->map = map;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sF_W ]
 *
 * @param	map		}bvID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionDungeonDataMake( u16 map, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id  = FNOTE_ID_DUNGEON;
	wk->map = map;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sF ]
 *
 * @param	map		}bvID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionBuildDataMake( u16 map, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id  = FNOTE_ID_BUILD;
	wk->map = map;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFQ[R[i[ ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionGameCornerDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id  = FNOTE_ID_GAMECORNER;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFTt@][ ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionSafariDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id  = FNOTE_ID_SAFARIZONE;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFACe擾 ]
 *
 * @param	item	ACeID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionItemDataMake( u16 item, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id   = FNOTE_ID_ITEMGET;
	wk->item = item;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFZ ]
 *
 * @param	waza	tB[hZԍ
 * @param	map		}bvID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionSkillDataMake( u8 waza, u16 map, u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id  = FNOTE_ID_IAIGIRI+waza;
	wk->map = map;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFnEΌ@ ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionUGFossilDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_UG_FOSSIL;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ sFnE閧n ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_ActionUGBaseDataMake( u32 heap )
{
	FNOTE_WK_ACTION * wk = FNOTE_ActionWorkAlloc( heap );
	wk->id = FNOTE_ID_UG_BASE;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * ځE|Pp[Nm
 *
 * @param	heap	q[vID
 *
 * @return	mۂ[N
 */
//--------------------------------------------------------------------------------------------
static FNOTE_WK_POKE * FNOTE_PokeWorkAlloc( u32 heap )
{
	FNOTE_WK_POKE * wk = sys_AllocMemory( heap, sizeof(FNOTE_WK_POKE) );
	memset( wk, 0, sizeof(FNOTE_WK_POKE) );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ |PFߊl ]
 *
 * @param	ptyme	vC
 * @param	mons	|PID
 * @param	sex		
 * @param	time	ԑ
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_PokemonGetDataMake( const PLAYTIME * ptime, u16 mons, u8 sex, u8 time, u32 heap )
{
	FNOTE_WK_POKE * wk;
	u8	min;
	
	wk = FNOTE_PokeWorkAlloc( heap );
	wk->id   = FNOTE_ID_POKEGET;
	wk->mons = mons;
	wk->sex  = sex;
	wk->time = time;

	min = PLAYTIME_GetMinute( ptime ) / 10;
	if( min == 0 || min == 2 || min == 4 ){
		wk->type = 0;
	}else if( min == 1 || min == 3 ){
		wk->type = 1;
	}else{
		wk->type = 2;
	}

	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ |PF| ]
 *
 * @param	ptyme	vC
 * @param	mons	|PID
 * @param	sex		
 * @param	time	ԑ
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_PokemonDownDataMake( const PLAYTIME * ptime, u16 mons, u8 sex, u8 time, u32 heap )
{
	FNOTE_WK_POKE * wk;
	u8	min;
	
	wk = FNOTE_PokeWorkAlloc( heap );
	wk->id   = FNOTE_ID_POKEDOWN;
	wk->mons = mons;
	wk->sex  = sex;
	wk->time = time;

	min = PLAYTIME_GetMinute( ptime ) / 10;
	if( min == 1 || min == 3 || min == 5 ){
		wk->type = 0;
	}else if( min == 2 || min == 4 ){
		wk->type = 1;
	}else{
		wk->type = 2;
	}

	return wk;
}


//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ g[i[ ]
 *
 * @param	map		}bvID
 * @param	trainer	g[i[ID
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_TrainerDataMake( u16 map, u16 trainer, u32 heap )
{
	FNOTE_WK_TRAINER * wk = sys_AllocMemory( heap, sizeof(FNOTE_WK_TRAINER) );
	wk->id   = 1;
	wk->map  = map;
	wk->trid = trainer;
	return wk;
}


//--------------------------------------------------------------------------------------------
/**
 * ځEʐMp[Nm
 *
 * @param	heap	q[vID
 *
 * @return	mۂ[N
 */
//--------------------------------------------------------------------------------------------
static FNOTE_WK_SIO * FNOTE_SioWorkAlloc( u32 heap )
{
	FNOTE_WK_SIO * wk = sys_AllocMemory( heap, sizeof(FNOTE_WK_SIO) );
	memset( wk, 0, sizeof(FNOTE_WK_SIO) );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * Rs[
 *
 * @param	str		Rs[镶
 * @param	buf		i[
 * @param	siz		
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_StrCopy( const STRCODE * str, STRCODE * buf, u32 siz )
{
	u32	i;

	for( i=0; i<siz; i++ ){
		buf[i] = EOM_;
	}
	for( i=0; i<siz; i++ ){
		if( str[i] == EOM_ ){ break; }
		buf[i] = str[i];
	}
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFʐMΐEVO ]
 *
 * @param	name	ΐ푊̖O
 * @param	sex		ΐ푊̐
 * @param	result	s
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioSingleBtlDataMake( STRCODE * name, u8 sex, u8 result, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id     = FNOTE_ID_SB_SINGLE;
	wk->result = result;
	wk->sex1   = sex;
	FNOTE_StrCopy( name, wk->name1, BUFLEN_PERSON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFʐMΐE_u ]
 *
 * @param	name	ΐ푊̖O
 * @param	sex		ΐ푊̐
 * @param	result	s
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioDoubleBtlDataMake( STRCODE * name, u8 sex, u8 result, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id     = FNOTE_ID_SB_DOUBLE;
	wk->result = result;
	wk->sex1   = sex;
	FNOTE_StrCopy( name, wk->name1, BUFLEN_PERSON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFʐMΐE}` ]
 *
 * @param	name1	ΐ푊P̖O
 * @param	name2	ΐ푊Q̖O
 * @param	sex1	ΐ푊P̐
 * @param	sex2	ΐ푊Q̐
 * @param	result	s
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioMultiBtlDataMake(
		 STRCODE * name1, STRCODE * name2, u8 sex1, u8 sex2, u8 result, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id     = FNOTE_ID_SB_MULTI;
	wk->result = result;
	wk->sex1   = sex1;
	wk->sex2   = sex2;
	FNOTE_StrCopy( name1, wk->name1, BUFLEN_PERSON_NAME );
	FNOTE_StrCopy( name2, wk->name2, BUFLEN_PERSON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFʐMΐE~bNX1vs1 ]
 *
 * @param	name	ΐ푊̖O
 * @param	sex		ΐ푊̐
 * @param	result	s
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioMixBtl1DataMake( STRCODE * name, u8 sex, u8 result, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id     = FNOTE_ID_SB_MIX1;
	wk->result = result;
	wk->sex1   = sex;
	FNOTE_StrCopy( name, wk->name1, BUFLEN_PERSON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFʐMΐE~bNX2vs2 ]
 *
 * @param	name1	ΐ푊P̖O
 * @param	name2	ΐ푊Q̖O
 * @param	sex1	ΐ푊P̐
 * @param	sex2	ΐ푊Q̐
 * @param	result	s
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioMixBtl2DataMake(
		 STRCODE * name1, STRCODE * name2, u8 sex1, u8 sex2, u8 result, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id     = FNOTE_ID_SB_MIX2;
	wk->result = result;
	wk->sex1   = sex1;
	wk->sex2   = sex2;
	FNOTE_StrCopy( name1, wk->name1, BUFLEN_PERSON_NAME );
	FNOTE_StrCopy( name2, wk->name2, BUFLEN_PERSON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFjI[EA ]
 *
 * @param	name	̖O
 * @param	sex		̐
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioUnionGreetDataMake( STRCODE * name, u8 sex, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id   = FNOTE_ID_U_GREETING;
	wk->sex1 = sex;
	FNOTE_StrCopy( name, wk->name1, BUFLEN_PERSON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFjI[E ]
 *
 * @param	name		̖O
 * @param	sex			̐
 * @param	nickname	ł|P̃jbNl[
 * @param	poke_sex	ł|P̐
 * @param	heap		q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioUnionChangeDataMake(
		STRCODE * name, u8 sex, STRCODE * nickname, u8 poke_sex, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id       = FNOTE_ID_U_POKECHG;
	wk->sex1     = sex;
	wk->poke_sex = poke_sex;
	FNOTE_StrCopy( name, wk->name1, BUFLEN_PERSON_NAME );
	FNOTE_StrCopy( nickname, wk->nickname, BUFLEN_POKEMON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFjI[EG ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioUnionPictureDataMake( u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id = FNOTE_ID_U_PICTURE;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFjI[El ]
 *
 * @param	nickname	ł|P̃jbNl[
 * @param	poke_sex	ł|P̐
 * @param	heap		q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioUnionMltChgDataMake( STRCODE * nickname, u8 poke_sex, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id       = FNOTE_ID_U_MLTCHG;
	wk->poke_sex = poke_sex;
	FNOTE_StrCopy( nickname, wk->nickname, BUFLEN_POKEMON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFjI[EgV[o[ ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioUnionTransceiverDataMake( u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id = FNOTE_ID_U_TRANSCEIVER;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFjI[Eΐ ]
 *
 * @param	name	ΐ푊̖O
 * @param	sex		ΐ푊̐
 * @param	result	s
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioUnionBattleDataMake( STRCODE * name, u8 sex, u8 result, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id     = FNOTE_ID_U_BATTLE;
	wk->result = result;
	wk->sex1   = sex;
	FNOTE_StrCopy( name, wk->name1, BUFLEN_PERSON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFR[hR[i[ ]
 *
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioRecordDataMake( u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id = FNOTE_ID_RECORD;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFReXg ]
 *
 * @param	result	
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioContestDataMake( u8 result, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id     = FNOTE_ID_CONTEST;
	wk->result = result;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMF|g쐬 ]
 *
 * @param	name	̖O
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioPorutoDataMake( u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id = FNOTE_ID_PORUTO;
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFGTC ]
 *
 * @param	name		̖O
 * @param	sex			̐
 * @param	nickname	ł|P̃jbNl[
 * @param	poke_sex	ł|P̐
 * @param	heap		q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioGTCDataMake( STRCODE * name, u8 sex, STRCODE * nickname, u8 poke_sex, u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id       = FNOTE_ID_GTC;
	wk->sex1     = sex;
	wk->poke_sex = poke_sex;
	FNOTE_StrCopy( name, wk->name1, BUFLEN_PERSON_NAME );
	FNOTE_StrCopy( nickname, wk->nickname, BUFLEN_POKEMON_NAME );
	return wk;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬 [ ʐMFog^[ ]
 *
 * @param	name	̖O
 * @param	heap	q[vID
 *
 * @return	쐬[N
 */
//--------------------------------------------------------------------------------------------
void * FNOTE_SioBattleTowerDataMake( u32 heap )
{
	FNOTE_WK_SIO * wk = FNOTE_SioWorkAlloc( heap );
	wk->id = FNOTE_ID_BTLTOWER;
	return wk;
}


//============================================================================================
//	f[^擾
//============================================================================================

//--------------------------------------------------------------------------------------------
/**
 * zf[^擾
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 * @param	type	L^^Cv
 * @param	page	擾y[W
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
void FNOTE_DataGet( FNOTE_DATA * dat, void * wk, u8 type, u8 page )
{
	switch( type ){
	case FNOTE_TYPE_START:		// Jn
		FNOTE_StartDataGet( &dat[page], wk );
		break;

	case FNOTE_TYPE_ACTION:		// s
		FNOTE_ActionDataGet( &dat[page], wk );
		break;

	case FNOTE_TYPE_POKEMON:	// |P
		FNOTE_PokemonDataGet( &dat[page], wk );
		break;

	case FNOTE_TYPE_TRAINER:	// g[i[
		FNOTE_TrainerDataGet( &dat[page], wk );
		break;

	case FNOTE_TYPE_SIO:		// ʐM
		FNOTE_SioDataGet( &dat[page], wk );
		break;
	}
}

//--------------------------------------------------------------------------------------------
/**
 * Jnf[^擾
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_StartDataGet( FNOTE_DATA * dat, FNOTE_WK_START * wk )
{
	*wk = dat->hed;
}

//--------------------------------------------------------------------------------------------
/**
 * sf[^擾
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionDataGet( FNOTE_DATA * dat, FNOTE_WK_ACTION * wk )
{
	u32	i;

	for( i=0; i<FNOTE_ACTION_MAX; i++ ){
		memset( &wk[i], 0, sizeof(FNOTE_WK_ACTION) );
		switch( dat->action[i] & 0x3f ){
		case FNOTE_ID_MYHOUSE:			// sF
		case FNOTE_ID_LABO:				// sF
		case FNOTE_ID_POKECENTER:		// sF|PZ^[
		case FNOTE_ID_FDSHOP_BUY:		// sFVbvE
		case FNOTE_ID_FDSHOP_BUY2:		// sFVbvEij
		case FNOTE_ID_FDSHOP_SALE:		// sFVbvE
		case FNOTE_ID_FDSHOP_SALE2:		// sFVbvEij
		case FNOTE_ID_FDSHOP_TRADE:		// sFVbvE
		case FNOTE_ID_GAMECORNER:		// sFQ[R[i[
		case FNOTE_ID_SAFARIZONE:		// sFTt@][
		case FNOTE_ID_UG_FOSSIL:		// sFnEΌ@
		case FNOTE_ID_UG_BASE:			// sFnE閧n
			FNOTE_ActionIDDataGet( dat->action[i], &wk[i] );
			break;

		case FNOTE_ID_GYMBEFORE:		// sFWi[_[|Oj
		case FNOTE_ID_GYMAFTER:			// sFWi[_[|j
		case FNOTE_ID_SITENNOU:			// sFlV
		case FNOTE_ID_CHAMPION:			// sF`sI
			FNOTE_ActionTrainerDataGet( dat->action[i], &wk[i] );
			break;

		case FNOTE_ID_CITY:				// sFX
		case FNOTE_ID_DUNGEON:			// sF_W
		case FNOTE_ID_BUILD:			// sF
			FNOTE_ActionMapDataGet( dat->action[i], &wk[i] );
			break;

		case FNOTE_ID_ITEMGET:			// sFACe擾
			FNOTE_ActionItemDataGet( dat->action[i], &wk[i] );
			break;

		case FNOTE_ID_IAIGIRI:			// sFZE
		case FNOTE_ID_SORAWOTOBU:		// sFZEƂ
		case FNOTE_ID_NAMINORI:			// sFZEȂ݂̂
		case FNOTE_ID_KAIRIKI:			// sFZE肫
		case FNOTE_ID_FLASH:			// sFZEtbV
		case FNOTE_ID_KIRIBARAI:		// sFZE΂炢
		case FNOTE_ID_IWAKUDAKI:		// sFZE킭
		case FNOTE_ID_TAKINOBORI:		// sFZÊڂ
		case FNOTE_ID_ROCKCLIMB:		// sFZEbNNC
		case FNOTE_ID_ANAWOHORU:		// sFZEȂق
		case FNOTE_ID_AMAIKAORI:		// sFZE܂
		case FNOTE_ID_TELEPORT:			// sFZEe|[g
		case FNOTE_ID_TAMAGOUMI:		// sFZE^}S
		case FNOTE_ID_MIRUKUNOMI:		// sFZE~N̂
			FNOTE_ActionSkillDataGet( dat->action[i], &wk[i] );
			break;
		}
	}
}

//--------------------------------------------------------------------------------------------
/**
 * |Pf[^擾
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_PokemonDataGet( FNOTE_DATA * dat, FNOTE_WK_POKE * wk )
{
	*wk = dat->pokemon;
}

//--------------------------------------------------------------------------------------------
/**
 * g[i[f[^擾
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_TrainerDataGet( FNOTE_DATA * dat, FNOTE_WK_TRAINER * wk )
{
	*wk = dat->trainer;
}

//--------------------------------------------------------------------------------------------
/**
 * ʐMf[^擾
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioDataGet( FNOTE_DATA * dat, FNOTE_WK_SIO * wk )
{
	u32	i;

	for( i=0; i<FNOTE_SIO_MAX; i++ ){
		memset( &wk[i], 0, sizeof(FNOTE_WK_SIO) );
		switch( dat->sio[i][0] ){
		case FNOTE_ID_SB_SINGLE:		// ʐMFʐMΐEVO
		case FNOTE_ID_SB_DOUBLE:		// ʐMFʐMΐE_u
		case FNOTE_ID_SB_MULTI:			// ʐMFʐMΐE}`
		case FNOTE_ID_SB_MIX1:			// ʐMFʐMΐE~bNX1vs1
		case FNOTE_ID_SB_MIX2:			// ʐMFʐMΐE~bNX2vs2
		case FNOTE_ID_U_BATTLE:			// ʐMFjI[Eΐ
			FNOTE_SioBattleDataGet( &dat->sio[i][0], &wk[i] );
			break;

		case FNOTE_ID_U_GREETING:		// ʐMFjI[EA
			FNOTE_SioUnionGreetDataGet( &dat->sio[i][0], &wk[i] );
			break;

		case FNOTE_ID_U_POKECHG:		// ʐMFjI[E
		case FNOTE_ID_U_MLTCHG:			// ʐMFjI[El
		case FNOTE_ID_GTC:				// ʐMFGTC
			FNOTE_SioPokeChgDataGet( &dat->sio[i][0], &wk[i] );
			break;

		case FNOTE_ID_U_PICTURE:		// ʐMFjI[EG
		case FNOTE_ID_U_TRANSCEIVER:	// ʐMFjI[EgV[o[
		case FNOTE_ID_RECORD:			// ʐMFR[hR[i[
		case FNOTE_ID_PORUTO:			// ʐMF|g쐬
		case FNOTE_ID_BTLTOWER:			// ʐMFog^[
			FNOTE_SioIDDataGet( &dat->sio[i][0], &wk[i] );
			break;

		case FNOTE_ID_CONTEST:			// ʐMFReXg
			FNOTE_SioContestDataGet( &dat->sio[i][0], &wk[i] );
			break;
		}
	}
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ sFID݂̂̂ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionIDDataGet( u32 dat, FNOTE_WK_ACTION * wk )
{
//	*dat = ( wk->id & 0x3f );
	wk->id = ( dat & 0x3f );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ sFW[_[AlVA`sI ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionTrainerDataGet( u32 dat, FNOTE_WK_ACTION * wk )
{
//	*dat = (wk->map << 16) | ((wk->trainer & 0x3ff) << 6) | (wk->id & 0x3f);
	wk->id      = ( dat & 0x3f );
	wk->trainer = ( dat >> 6 ) & 0x3ff;
	wk->map     = ( dat >> 16 ) & 0xffff;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ sFXA_WAȂ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionMapDataGet( u32 dat, FNOTE_WK_ACTION * wk )
{
//	*dat = (wk->map << 16) | (wk->id & 0x3f);
	wk->id  = ( dat & 0x3f );
	wk->map = ( dat >> 16 ) & 0xffff;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ sFACe擾 ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionItemDataGet( u32 dat, FNOTE_WK_ACTION * wk )
{
//	*dat = (wk->item << 16) | (wk->id & 0x3f);
	wk->id   = ( dat & 0x3f );
	wk->item = ( dat >> 16 ) & 0xffff;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ sFZ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_ActionSkillDataGet( u32 dat, FNOTE_WK_ACTION * wk )
{
//	*dat = (wk->map << 16) | (wk->id & 0x3f);
	wk->id  = ( dat & 0x3f );
	wk->map = ( dat >> 16 ) & 0xffff;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ ʐMFΐ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioBattleDataGet( u8 * dat, FNOTE_WK_SIO * wk )
{
/*
	dat[0] = wk->id;
	dat[1] = (wk->result<<2) | (wk->sex1<<1) | (wk->sex2);
	FNOTE_StrCopy( wk->name1, (STRCODE *)&dat[2], 8 );
	FNOTE_StrCopy( wk->name2, (STRCODE *)&dat[18], 8 );
*/
	wk->id       = dat[0];
	wk->result   = ( dat[1] >> 2 ) & 0x1f;
	wk->sex1     = ( dat[1] >> 1 ) & 1;
	wk->sex2     = dat[1] & 1;
	FNOTE_StrCopy( (STRCODE *)&dat[2], wk->name1, 8 );
	FNOTE_StrCopy( (STRCODE *)&dat[18], wk->name2, 8 );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ ʐMFjI[EA ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioUnionGreetDataGet( u8 * dat, FNOTE_WK_SIO * wk )
{
/*
	dat[0] = wk->id;
	dat[1] = wk->sex1;
	FNOTE_StrCopy( wk->name1, (STRCODE *)&dat[2], 8 );
*/
	wk->id   = dat[0];
	wk->sex1 = dat[1];
	FNOTE_StrCopy( (STRCODE *)&dat[2], wk->name1, 8 );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ ʐMF ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioPokeChgDataGet( u8 * dat, FNOTE_WK_SIO * wk )
{
/*
	dat[0] = wk->id;
	dat[1] = (wk->sex1 << 2) | (wk->poke_sex);
	FNOTE_StrCopy( wk->name1, (STRCODE *)&dat[2], 8 );
	FNOTE_StrCopy( wk->nickname, (STRCODE *)&dat[18], 12 );
*/
	wk->id       = dat[0];
	wk->sex1     = ( dat[1] >> 2 ) & 1;
	wk->poke_sex = dat[1] & 3;
	FNOTE_StrCopy( (STRCODE *)&dat[2], wk->name1, 8 );
	FNOTE_StrCopy( (STRCODE *)&dat[18], wk->nickname, 12 );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ ʐMFog^[ ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioIDDataGet( u8 * dat, FNOTE_WK_SIO * wk )
{
//	dat[0] = wk->id;
	wk->id = dat[0];
}

//--------------------------------------------------------------------------------------------
/**
 * f[^擾 [ ʐMFReXg ]
 *
 * @param	dat		Z[uf[^
 * @param	wk		擾ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void FNOTE_SioContestDataGet( u8 * dat, FNOTE_WK_SIO * wk )
{
/*
	dat[0] = wk->id;
	dat[1] = wk->result;
*/
	wk->id     = dat[0];
	wk->result = dat[1];
}


//--------------------------------------------------------------------------------------------
/**
 * f[^쐬/Z[u [ sFAAA_Wo ]
 *
 * @param	my		v[[f[^
 * @param	dat		`m[g̃Z[uf[^
 * @param	now		݂̃}bv
 * @param	before	Oɂ}bv
 * @param	heap	q[vID
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
void FNOTE_DataMakeSaveMapChg( MYSTATUS * my, FNOTE_DATA * dat, u32 now, u32 before, u32 heap )
{
	void * buf;
	u32	i;

	buf = NULL;

	// _W
	if( ZoneData_IsDungeon( before ) == TRUE ){
		// O
		if( ZoneData_IsOutDoor( now ) == TRUE ){
//			buf = FNOTE_ActionDungeonDataMake( (u16)before, heap );
			buf = FNOTE_ActionDungeonDataMake( (u16)ZoneData_GetPlaceNameID(before), heap );
		}
	// 
	}else if( ZoneData_IsRoom( before ) == TRUE ){
		// O
		if( ZoneData_IsOutDoor( now ) == TRUE ){
			// 
			if( before == ZONE_ID_T01R0201 ){
				buf = FNOTE_ActionHouseDataMake( heap );
			// 
			}else if( before == ZONE_ID_T02R0101 ){
				buf = FNOTE_ActionLaboDataMake( heap );
			// ̑
			}else{
				u32	name = ZoneData_GetPlaceNameID( before );
				for( i=0; i<NELEMS(BuildNameTable); i++ ){
					if( name != BuildNameTable[i].name ){ continue; }
//					buf = FNOTE_ActionBuildDataMake( (u16)before, heap );
					buf = FNOTE_ActionBuildDataMake( (u16)name, heap );
				}
			}
		}
	// O
	}else if( ZoneData_IsOutDoor( before ) == TRUE ){
		// 
		if( ZoneData_IsRoom( now ) == TRUE ){
			// W
			u8	gym = GymCheck( my, now );
			if( gym != 0xff ){
				buf = FNOTE_ActionGymBeforeDataMake( gym, heap );
			}
		}
	}

	if( buf != NULL ){
		FNOTE_DataSave( dat, buf, FNOTE_TYPE_ACTION );
	}
}

//--------------------------------------------------------------------------------------------
/**
 * Ȍ̕\^Cv擾
 *
 * @param	name	OID
 *
 * @return	\^Cv
 */
//--------------------------------------------------------------------------------------------
u32 FNOTE_ExBuildPutTypeGet( u32 name )
{
	u32	i;

	for( i=0; i<NELEMS(BuildNameTable); i++ ){
		if( name != BuildNameTable[i].name ){ continue; }
		return BuildNameTable[i].type;
	}
	return 0;
}

//--------------------------------------------------------------------------------------------
/**
 * W`FbN
 *
 * @param	my			v[[f[^
 * @param	zone_id		][ID
 *
 * @retval	"0xff != WID"
 * @retval	"0xff = WȊO"
 */
//--------------------------------------------------------------------------------------------
static u8 GymCheck( MYSTATUS * my, u32 zone_id )
{
	u8	i;

	for( i=0; i<NELEMS(GymTbl); i++ ){
		if( GymTbl[i].zone == zone_id ){
			if( MyStatus_GetBadgeFlag( my, GymTbl[i].badge ) == FALSE ){
				return i;
			}
			break;
		}
	}
	return 0xff;
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬/Z[u [ sFXɓ ]
 *
 * @param	dat		Z[uf[^
 * @param	map		݂̃}bv
 * @param	heap	q[vID
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
void FNOTE_DataMakeSaveCity( FNOTE_DATA * dat, u32 map, u32 heap )
{
	void * buf = FNOTE_ActionCityDataMake( (u16)map, heap );
	FNOTE_DataSave( dat, buf, FNOTE_TYPE_ACTION );
}

//--------------------------------------------------------------------------------------------
/**
 * f[^쐬/Z[u [ sFW[_[A`sIAlV & g[i[Fg[i[ ]
 *
 * @param	dat			`m[g̃Z[uf[^
 * @param	map			][ID
 * @param	trainer		g[i[ID
 * @param	heap		q[vID
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
void FNOTE_DataMakeSaveTrainer( FNOTE_DATA * dat, u16 map, u16 trainer, u32 heap )
{
	void * buf;
	u8	ret;
	
	ret = TrainerCheck( trainer );

	// W[_[
	if( ret < 8 ){
		buf = FNOTE_ActionGymAfterDataMake( ret, trainer, heap );
		FNOTE_DataSave( dat, buf, FNOTE_TYPE_ACTION );
	// lV
	}else if( ret == 8 ){
		buf = FNOTE_ActionSitenouDataMake( trainer, heap );
		FNOTE_DataSave( dat, buf, FNOTE_TYPE_ACTION );
	// `sI
	}else if( ret == 9 ){
		buf = FNOTE_ActionChampionDataMake( trainer, heap );
		FNOTE_DataSave( dat, buf, FNOTE_TYPE_ACTION );
	// ʂ̃g[i[
	}else{
		buf = FNOTE_TrainerDataMake( map, trainer, heap );
		FNOTE_DataSave( dat, buf, FNOTE_TYPE_TRAINER );
	}
}

//--------------------------------------------------------------------------------------------
/**
 * g[i[^Cv擾
 *
 * @param	tr		g[i[ID
 *
 * @retval	"8 > W[_[
 * @retval	"8 = lV"
 * @retval	"9 = `sI"
 * @retval	"10 = ʂ̃g[i["
 */
//--------------------------------------------------------------------------------------------
static u8 TrainerCheck( u32 tr )
{
	u8	i;

	// W[_[
	for( i=0; i<NELEMS(GymTbl); i++ ){
		if( GymTbl[i].tr == tr ){
			return i;
		}
	}
	// lV
	if( tr == BIGFOUR1_01 || tr == BIGFOUR2_01 || tr == BIGFOUR3_01 || tr == BIGFOUR4_01 ){
		return 8;
	}
	// `sI
	if( tr == CHAMPION_01 ){
		return 9;
	}
	return 10;
}

