﻿#!/usr/bin/perl

#===============================================================================
# 簡易会話用データコンバータ
#
# 簡易会話の単語として扱う各種gmmを参照し、
# カテゴリ別・インデックス別の単語ID配列データ等を作成する。
# 作成されたデータはＣプログラムから参照される。
#
# GameFreak taya
#
#===============================================================================

use utf8;
use encode;
use XML::Parser;
use FindBin;

use open OUT => ":encoding(shiftjis)";
binmode(STDERR,":encoding(shiftjis)");
binmode(STDOUT,":encoding(shiftjis)");

require 'gmmparse.pl';


#===============================================================
# 
#===============================================================
my $DUP_SYMBOL = "PMS_WORDID_DUP";
my $DUP_VALUE = "0xfffe";

my $END_SYMBOL = "PMS_WORDID_END";
my $END_VALUE = "0xffff";



#===============================================================
# 参照gmm（※ローカライズ作業でここを変更してはいけない）
#===============================================================

use constant SRC_MONSNAME	=> 0;
use constant SRC_WAZANAME	=> 1;
use constant SRC_BATTLE		=> 2;
use constant SRC_POKETYPE	=> 3;
use constant SRC_TOKUSEI	=> 4;
use constant SRC_VOICE		=> 5;
use constant SRC_LIFE		=> 6;
use constant SRC_HOBBY		=> 7;
use constant SRC_REPEAT		=> 8;


my @TargetGmm = (
	[ '../message/src/monsname.gmm',		'PMS_SRCID_MONSNAME',	'ポケモン名' ],
	[ '../message/src/wazaname.gmm',		'PMS_SRCID_WAZANAME',	'わざ名' ],
	[ '../message/src/pms/pmsw_battle.gmm',	'PMS_SRCID_BATTLE',	 	'バトル' ],
	[ '../message/src/typename.gmm',		'PMS_SRCID_POKETYPE',	'ポケモンタイプ名' ],
	[ '../message/src/tokusei.gmm',			'PMS_SRCID_TOKUSEI',	'とくせい' ],
	[ '../message/src/pms/pmsw_voice.gmm',	'PMS_SRCID_VOICE',	 	'こえ' ],
	[ '../message/src/pms/pmsw_life.gmm',	'PMS_SRCID_LIFE',	 	'せいかつ' ],
	[ '../message/src/pms/pmsw_hobby.gmm',	'PMS_SRCID_HOBBY',	 	'しゅみ' ],
	[ '../message/src/pms/pmsw_repeat.gmm',	'PMS_SRCID_REPEAT',	 	'くりかえし' ],

);

# 要素0番目を飛ばすgmmファイル
my @ZeroSkipGmm = ( SRC_MONSNAME, SRC_WAZANAME, SRC_TOKUSEI );


#===============================================================
# カテゴリパラメータ（※ローカライズ作業でここを変更してはいけない）
#===============================================================
use constant  CATEGORY_CONV_NORMAL	=> 0;
use constant  CATEGORY_CONV_DIV		=> 1;
use constant  CATEGORY_CONV_MIX		=> 2;
use constant  CATEGORY_CONV_MONSNO	=> 3;

my @CategoryParam = (
	[ 'ポケモン', 'pokemon1',	CATEGORY_CONV_DIV, SRC_MONSNAME,    0, 200  ],
	[ 'ポケモン２', 'pokemon2',	CATEGORY_CONV_DIV, SRC_MONSNAME,  201,  -1  ],
	[ 'わざ', 'skill1',			CATEGORY_CONV_DIV, SRC_WAZANAME,    0, 200  ],
	[ 'わざ２', 'skill2',		CATEGORY_CONV_DIV, SRC_WAZANAME,  201,  -1  ],
	[ 'バトル', 'battle',		CATEGORY_CONV_NORMAL, SRC_BATTLE  ],
	[ 'ステータス',	'status',	CATEGORY_CONV_MIX, SRC_TOKUSEI, SRC_POKETYPE  ],
	[ 'こえ',		'voice',	CATEGORY_CONV_NORMAL, SRC_VOICE  ],
	[ 'せいかつ',	'life',		CATEGORY_CONV_NORMAL, SRC_LIFE  ],
	[ 'しゅみ',		'hobby',	CATEGORY_CONV_NORMAL, SRC_HOBBY  ],
	[ 'くりかえし',	'repeat',	CATEGORY_CONV_NORMAL, SRC_REPEAT  ],
);


#===============================================================
# イニシャル順パラメータ（同じ配列内にある文字を同一項とみなす）
#===============================================================
my @InitialTable = (
	[ 'あ', 'ぁ', 'ア', 'ァ' ],
	[ 'い','ぃ','イ','ィ' ],
	[ 'う','ぅ','ウ','ゥ' ],
	[ 'え','ぇ','エ','ェ' ],
	[ 'お','ぉ','オ','ォ' ],
	[ 'か','が','カ','ガ' ],
	[ 'き','ぎ','キ','ギ' ],
	[ 'く','ぐ','ク','グ' ],
	[ 'け','げ','ケ','ゲ' ],
	[ 'こ','ご','コ','ゴ' ],
	[ 'さ','ざ','サ','ザ' ],
	[ 'し','じ','シ','ジ' ],
	[ 'す','ず','ス','ズ' ],
	[ 'せ','ぜ','セ','ゼ' ],
	[ 'そ','ぞ','ソ','ゾ' ],
	[ 'た','だ','タ','ダ' ],
	[ 'ち','ぢ','チ','ヂ' ],
	[ 'つ','づ','ツ','ヅ' ],
	[ 'て','で','テ','デ' ],
	[ 'と','ど','ト','ド' ],
	[ 'な','ナ' ],
	[ 'に','ニ' ],
	[ 'ぬ','ヌ' ],
	[ 'ね','ネ' ],
	[ 'の','ノ' ],
	[ 'は','ハ','ば','バ','ぱ','パ' ],
	[ 'ひ','ヒ','び','ビ','ぴ','ピ' ],
	[ 'ふ','フ','ぶ','ブ','ぷ','プ' ],
	[ 'へ','ヘ','べ','ベ','ぺ','ペ' ],
	[ 'ほ','ホ','ぼ','ボ','ぽ','ポ' ],
	[ 'ま','マ' ],
	[ 'み','ミ' ],
	[ 'む','ム' ],
	[ 'め','メ' ],
	[ 'も','モ' ],
	[ 'や','ゃ','ヤ','ャ' ],
	[ 'ゆ','ゅ','ユ','ュ' ],
	[ 'よ','ょ','ヨ','ョ' ],
	[ 'ら','ラ' ],
	[ 'り','リ' ],
	[ 'る','ル' ],
	[ 'れ','レ' ],
	[ 'ろ','ロ' ],
	[ 'わ','ワ','を','ヲ','ん','ン' ],
	[ 'a','A','ａ','Ａ' ],
	[ 'b','B','ｂ','Ｂ' ],
	[ 'c','C','ｃ','Ｃ' ],
	[ 'd','D','ｄ','Ｄ' ],
	[ 'e','E','ｅ','Ｅ' ],
	[ 'f','F','ｆ','Ｆ' ],
	[ 'g','G','ｇ','Ｇ' ],
	[ 'h','H','ｈ','Ｈ' ],
	[ 'i','I','ｉ','Ｉ' ],
	[ 'j','J','ｊ','Ｊ' ],
	[ 'k','K','ｋ','Ｋ' ],
	[ 'l','L','ｌ','Ｌ' ],
	[ 'm','M','ｍ','Ｍ' ],
	[ 'n','N','ｎ','Ｎ' ],
	[ 'o','O','ｏ','Ｏ' ],
	[ 'p','P','ｐ','Ｐ' ],
	[ 'q','Q','ｑ','Ｑ' ],
	[ 'r','R','ｒ','Ｒ' ],
	[ 's','S','ｓ','Ｓ' ],
	[ 't','T','ｔ','Ｔ' ],
	[ 'u','U','ｕ','Ｕ' ],
	[ 'v','V','ｖ','Ｖ' ],
	[ 'w','W','ｗ','Ｗ' ],
	[ 'x','X','ｘ','Ｘ' ],
	[ 'y','Y','ｙ','Ｙ' ],
	[ 'z','Z','ｚ','Ｚ' ],

);

# ↑このテーブルから、イニシャルごとのインデックスを作っておくためのハッシュ（高速化のため）
my %InitialIdxTable = ();

# ↑このテーブルに属さない文字のＩＤ
use constant INITIAL_ID_OTHER => -1;

#===============================================================
# このスクリプトの動作モード
#===============================================================
use constant  MODE_ERROR => 0;
use constant  MODE_WORD => 1;
use constant  MODE_INPUT => 2;
use constant  MODE_LIST => 3;


#===============================================================
# global
#===============================================================
my @SrcElems = ();
my @SrcString = ();
my $TotalElems  = 0;



#===============================================================
# 
#===============================================================
&main();
exit(0);

sub main {

	my $mode = check_mode(@ARGV);

	if( $mode == MODE_ERROR )
	{
		&usage();
		return 1;
	}

	my $OutputFile = $ARGV[1];


	if( $mode == MODE_LIST )
	{
		&output_list($OutputFile);
	}
	else
	{

		&Setup_InitialIdxTable();

		foreach my $ref (@TargetGmm)
		{
			my $filepath = $FindBin::Bin . '/' . $$ref[0];

			my @tmp = GmmParse::ToArray( $filepath, '日本語' );

			my $elems = @tmp;
			push @SrcElems, $elems;
			$TotalElems += $elems;
			@SrcString = (@SrcString, @tmp);
		}

		if( $mode == MODE_WORD )
		{
			output_word_data($OutputFile);
		}
		else
		{
			output_input_data($OutputFile);
		}

	}

}

sub check_mode {
	my @arg = @_;

	if( @arg == 2 )
	{
		if( $arg[0] eq 'w' ){ return MODE_WORD;  }
		if( $arg[0] eq 'i' ){ return MODE_INPUT; }
		if( $arg[0] eq 'l' ){ return MODE_LIST;  }
	}

	return MODE_ERROR;
}

sub usage {
	my ($pack,$filename,$line) = caller;
	print ">perl $filename <mode> <outputfile>\n";
	print "<mode> ...  w  単語データ作成\n";
	print "       ...  i  入力画面データ作成\n";
	print "       ...  l  データ作成用ソースリスト作成\n";
}
#===============================================================
# 高速化のため、イニシャルごとのインデックス値テーブルを作っておく
# あ・ぁ・ア・ァ=>1, い・ぃ・イ・ィ=>2, ... みたいなやつ。
#===============================================================
sub Setup_InitialIdxTable {
	my $idx = 0;

	foreach my $ref (@InitialTable)
	{
		foreach my $letter (@$ref)
		{
			$InitialIdxTable{$letter} = $idx;
		}
		$idx++;
	}
}

#===============================================================
# 
#===============================================================
sub output_word_data {
	my $filename = shift;

	if( open(FILE, ">$filename") )
	{
		my $i;
		my $tmp;

		print FILE &GetCommonCommentStr();

		print FILE "#ifdef __PMS_WORD_RES__\t// 単一のファイルからのみincludeを許可する\n\n";

		# ファイル毎の要素数
		print FILE &ToLongComment( "各gmmファイル毎の要素数" );
		print FILE "static const u16 PMS_SrcElems[] = {\n";
		$tmp = 0;
		for($i=0; $i<@SrcElems; $i++)
		{
			printf FILE "\t%4d,\t//%s\t%d ... %d\n", $SrcElems[$i], ${$TargetGmm[$i]}[2], $tmp, ($tmp+$SrcElems[$i]-1);
			$tmp += $SrcElems[$i];
		}
		print FILE "};\n\n";

		# ファイルのアーカイブIDテーブル
		print FILE &ToLongComment( "各gmmファイルのアーカイブ内ファイルID" );
		print FILE "static const u16 PMS_SrcFileID[] = {\n";
		foreach $tmp (@TargetGmm)
		{
			print FILE "\t";
			print FILE &MakeFileIDString($$tmp[0]);
			print FILE ",\n";
		}
		print FILE "};\n\n";

		$tmp = @TargetGmm;
		print FILE &ToLongComment( "関連定数" );
		print FILE "#define  PMS_SRCFILE_MAX  ($tmp)\n\n";

		print FILE "#undef __PMS_WORD_RES__\n";
		print FILE "#endif\n\n";

		close(FILE);
		print "->$filename\n";
	}
	else
	{
		print "$filename を出力できません\n";
		return 0;
	}

	return 1;

}

#===============================================================
# 
#===============================================================
sub output_input_data {
	my $filename = shift;

	if( open(FILE, ">$filename") )
	{
		my $i;
		my $tmp;
		my $cnt;

		print FILE &GetCommonCommentStr();

		print FILE "#ifdef __PMS_INPUT_RES__\n\n";

		print FILE "#define $END_SYMBOL\t($END_VALUE)\n";
		print FILE "#define $DUP_SYMBOL\t($DUP_VALUE)\n\n";

		print FILE "#define PMS_WORDNUM_MAX\t($TotalElems)\t// 総単語数\n\n";


		# カテゴリごとの単語IDテーブル
		print FILE &ToLongComment( "カテゴリごとの単語IDテーブル" );
		foreach $tmp (@CategoryParam) {
			my @datStrID = CreateCategoryIdxTable($tmp);
			my @datStr;
			GetSortTable(\@datStr, \@datStrID);
			print FILE "// $$tmp[0]\n";
			print FILE "static const u16 PMS_CategoryTable_$$tmp[1]\[\] = {\n";
			for($i=0; $i<@datStrID; $i++)
			{
				print FILE "\t$datStrID[$i]\t//$datStr[$i]\n";
			}
			print FILE "\t$END_SYMBOL\n";
			print FILE "};\n\n";

			$cnt = @datStrID;
			print FILE "#define  PMS_Category_$$tmp[1]_MaxNum\t($cnt)\n\n";
		}

		# 文字順単語IDテーブル
		$tmp = 0;
		print FILE &ToLongComment( "文字順単語IDテーブル" );
		my $ini_max = @InitialTable;
		for($i=0; $i <= $ini_max; $i++)
		{
			my @datStrID = CreateInitialIdxTable($i);
			if( @datStrID > 0 )
			{
				my @datStr;
				GetSortTable(\@datStr, \@datStrID);
				print FILE "// ${$InitialTable[$i]}[0]\n";
				print FILE "static const u16 PMS_InitialTable_$tmp\[\] = {\n";
				for(my $p = 0; $p < @datStrID; $p++)
				{
					print FILE "\t$datStrID[$p]\t//$datStr[$p]\n";
				}
				print FILE "\t$END_SYMBOL\n";
				print FILE "};\n\n";
				$tmp++;
			}
		}

		print FILE "// 文字順テーブルの先頭アドレスをテーブル化しておく\n";
		print FILE "static const u16* const PMS_InitialTable[] = {\n";
		for($i=0; $i<$tmp; $i++)
		{
			print FILE "\tPMS_InitialTable_$i,\n";
		}
		print FILE "};\n\n";


		print FILE "#undef __PMS_INPUT_RES__\n";
		print FILE "#endif  // __PMS_INPUT_RES__\n";

		print "->$filename\n";
		close(FILE);
	}
	else
	{
		print "$filename を出力できません\n";
		return 0;
	}

	return 1;

}


#===============================================================
# カテゴリに含まれる全単語を、IDテーブル化して返す
#===============================================================
sub CreateCategoryIdxTable {
	my $category = shift;
	my @ary = ();

	if( $$category[2] == CATEGORY_CONV_NORMAL )
	{
		@ary = CreateSrcIdxArray( $$category[3] );
	}
	elsif( $$category[2] == CATEGORY_CONV_MIX )
	{
		@ary = CreateSrcIdxArray( $$category[3] );
		@ary = (@ary, CreateSrcIdxArray( $$category[4] ));
	}
	elsif( $$category[2] == CATEGORY_CONV_DIV )
	{
		my @str;
		my @strID;

		@strID = CreateSrcIdxArray( $$category[3] );
		@strID = &GetSortTable( \@str, \@strID );

		my $s = $$category[4];
		my $e = $$category[5];
		if( $e == -1 ){ $e += @strID; }

		for( ; $s < $e; $s++)
		{
			push @ary , $strID[$s];
		}

	}

	return @ary;
}
#===============================================================
# 同一イニシャルの単語を、IDテーブル化して返す
#===============================================================
sub CreateInitialIdxTable {
	my $ini = shift;
	my $letter;
	my @allStr;
	my @ary = ();

	@allStrID = CreateAllStrIdTable();

	if($ini >= @InitialTable)
	{
		print "hooyhoy!\n";
		$ini = INITIAL_ID_OTHER;
	}

	foreach $str_id (@allStrID )
	{
		$letter = substr( $SrcString[$str_id], 0, 1 );
		if( &GetInitialID($letter) == $ini )
		{
			push @ary, $str_id;
		}
	}

	my $num = @ary;
	return @ary;
}
#===============================================================
# １文字を受け取り、イニシャルＩＤを返す
#===============================================================
sub GetInitialID {
	my $letter = shift;

	if( exists( $InitialIdxTable{$letter} ) )
	{
		return $InitialIdxTable{$letter};
	}

	return INITIAL_ID_OTHER;
}

#===============================================================
# 使用する全ての単語IDをテーブルにして返す
# ※ 0～@#SrcString をそのまま使うと、
#	 ポケモン名等の先頭ダミー文字列が含まれてしまうため、ここで吸収している
#===============================================================
sub CreateAllStrIdTable {
	my $i;
	my @ary = ();

	for( $i = 0; $i<@TargetGmm; $i++ )
	{
		@ary = (@ary, CreateSrcIdxArray( $i ));
	}

	return @ary;
}


#===============================================================
# gmm １本文を、単語IDテーブル化して返す
# input : gmm ID
#===============================================================
sub CreateSrcIdxArray {
	my $src_no = shift;

	my $p = 0;

	for(my $i=0; $i<$src_no; $i++)
	{
		$p += $SrcElems[$i];
	}

	my $end = $p + $SrcElems[$src_no];

	my @ary = ();

	# ポケモン名などは先頭にダミー要素が入っているので、ここでカットしとく
	foreach my $tmp (@ZeroSkipGmm) {
		if( $tmp == $src_no ){
			$p++;
			last;
		}
	}

	for( ; $p < $end; $p++)
	{
		push @ary, $p;
	}

	return @ary;
}

#===============================================================
# 
#===============================================================
sub GetSortTable {
	my $strRef = shift;
	my $strIdRef = shift;
	my @idAry = @$strIdRef;

	my @strAry = ();
	my @sortAry = ();
	my %Idx;
	my $id;
	my $i;
	my $src_str;
	my $str;

	for($i=0; $i < @idAry; $i++)
	{
		$id = $idAry[$i];
		$src_str = $SrcString[$id];
		$str = $SrcString[$id];

		# 日本語では、平仮名・片仮名を同一視。海外版では大文字・小文字を同一視。
		$str =~ tr/ぁ-ん/ァ-ン/;
		$str =~ tr/a-z/A-Z/;

		$sortAry[$i] = [ $str, $src_str, $id ];


		if( exists ($Idx{$src_str}) )
		{
			$Idx{$src_str} .= sprintf( "%4d,", $id );
		}
		else
		{
			$Idx{$src_str} = sprintf( "%4d,", $id );
		}
	}

	@sortAry = sort sort_func @sortAry;

	my @tmp;
	my $dup;
	@idAry = ();
	for($i=0; $i<@sortAry; $i++)
	{
		$src_str = ${$sortAry[$i]}[1];
		if( $Idx{$src_str} ne "" )
		{
			@tmp = split(/,/, $Idx{$src_str});
			$dup = @tmp;
			if( $dup > 1 )
			{
				push @idAry, "$DUP_SYMBOL,$dup," . $Idx{$src_str};
			}
			else
			{
				push @idAry, $Idx{$src_str};
			}
			$Idx{$src_str} = "";
			push @$strRef, $src_str;
		}
	}

	@$strIdRef = @idAry;
	return @idAry;
}
sub sort_func {

	my $ret = $$a[0] cmp $$b[0];

	if( $ret == 0 )
	{
		$ret = $$a[1] cmp $$b[1];
	}


	return $ret;#$a <=> $b;
}

#===============================================================
# 
#===============================================================
sub ToLongComment {
	my $str = shift;
	my $line = "//----------------------------------------------------------------\n";
	my $ret = $line;
	$ret .= "/**\n";
	$ret .= " * $str\n";
	$ret .= "*/\n";
	$ret .= $line;
	$ret .= "\n";

	return $ret;
}

#===============================================================
# 
#===============================================================
sub GetCommonCommentStr {
	my @str = (
		"// このファイルはコンバータで自動生成されています\n",
		"// 手作業での編集は行わないようにお願いします\n",
		"// Game Freak   taya\n\n\n",
		);

	my $ret = "";

	foreach my $tmp (@str)
	{
		$ret .= $tmp;
	}

	return $ret;
}

#===============================================================
# 
#===============================================================
sub MakeFileIDString {
	my $path = shift;

	my @tmp = split(/\//, $path);

	my $fn = $tmp[$#tmp];

	$fn =~ s/\.gmm//;
	$fn = "NARC_msg_" . $fn . "_dat";

	return $fn;
}

#===============================================================
# 
#===============================================================
sub output_list {
	my $file = shift;

	if( open(FILE, ">$file") )
	{
		print FILE "SOURCE = \\\n";

		foreach my $ref (@TargetGmm)
		{
			print FILE "\t$$ref[0]  \\\n";
		}

		close(FILE);

		print "->$file\n";
	}
	else
	{
		print "$file が出力できない\n";
	}
}

