//============================================================================================
/**
 * @file	p_status.c
 * @brief	|PXe[^X
 * @author	Hiroyuki Nakamura
 * @date	2004.12.21
 */
//============================================================================================
#include "common.h"
#include "gflib/touchpanel.h"
#include "system/procsys.h"
#include "system/arc_tool.h"
#include "system/arc_util.h"
#include "system/clact_tool.h"
#include "system/brightness.h"
#include "system/fontproc.h"
#include "system/msgdata.h"
#include "system/wordset.h"
#include "system/numfont.h"
#include "system/softsprite.h"
#include "system/buflen.h"
#include "system/wipe.h"
#include "system/window.h"
#include "battle/battle_common.h"
#include "poketool/monsno.h"
#include "poketool/pokeicon.h"
#include "poketool/pokeparty.h"
#include "poketool/waza_tool.h"
#include "msgdata/msg.naix"
#include "msgdata/msg_pokestatus.h"
#include "itemtool/item.h"
#include "application/app_tool.h"

#define	P_STATUS_H_GLOBAL
#include "application/p_status.h"
#include "pst_sys.h"
#include "pst_obj.h"
#include "pst_bmp.h"
#include "pst_3d.h"
#include "pst_sub.h"
#include "pst_snd_def.h"
#include "pst_gra.naix"


//============================================================================================
//	萔`
//============================================================================================
enum {
	SEQ_IN = 0,			// tF[h҂
	SEQ_MAIN,			// CRg[
	SEQ_WAZAIN,			// ZC҂
	SEQ_WAZAOUT,		// ZAEg҂
	SEQ_WAZA_SELECT,	// ZI
	SEQ_WAZA_CHANGE,	// ZւI
	SEQ_WAZA_OBOE,		// Zo/Y
	SEQ_WAZA_OBOE_ERROR,	// ZoG[
	SEQ_BUTTON_ANM,		// ʂ̃{^Aj҂

	SEQ_COND_INIT,		// L[҂
	SEQ_COND_MSG,		// bZ[W\
	SEQ_COND_END,		// RfBVύXI

	SEQ_OUT,			// ItF[h҂
};

#define	TMP_MSGBUF_SIZ	( 128 )		// e|bZ[Wobt@TCY


//============================================================================================
//	vg^Cv錾
//============================================================================================
static PROC_RESULT PokeStatusProc_Init( PROC * proc, int * seq );
static PROC_RESULT PokeStatusProc_Main( PROC * proc, int * seq );
static PROC_RESULT PokeStatusProc_End( PROC * proc, int * seq );

static int PST_SeqIn( PST_WORK * wk );
static int PST_SeqMain( PST_WORK * wk );
static int PST_SeqWazaIn( PST_WORK * wk );
static int PST_SeqWazaOut( PST_WORK * wk );
static int PST_SeqWazaSelect( PST_WORK * wk );
static int PST_SeqWazaChange( PST_WORK * wk );
static int PST_SeqWazaAdd( PST_WORK * wk );
static int PST_SeqWazaAddError( PST_WORK * wk );
static int PST_SeqButtnAnm( PST_WORK * wk );
static int PST_SeqCondUpInit( PST_WORK * wk );
static int PST_SeqCondUpMsgPut( PST_WORK * wk );
static int PST_SeqCondUpEnd( PST_WORK * wk );
static u8 PST_SeqOut( PST_WORK * wk );

static void PokeStatusVramBankSet(void);
static void PokeStatusBgSet( GF_BGL_INI * ini );
static void PokeStatusBgExit( GF_BGL_INI * ini );
static void PokeStatusBgGraphicSet( PST_WORK * wk );
static void PokeStatusVBlank( void * work );
static void MsgBufCreate( PST_WORK * wk );
static void MsgBufRelease( PST_WORK * wk );
static void PokeStatusSoftSpriteSet( PST_WORK * wk );
static void PokeDataSet( PST_WORK * wk );
static void PokeDataSet_PokePasoParam(PST_WORK * wk,POKEMON_PASO_PARAM * ppp,PST_POKEDATA * dat);
static void PokeDataSet_PokemonParam( PST_WORK * wk, POKEMON_PARAM * pp, PST_POKEDATA * dat );
static void PokeDataSet_PokeParty( PST_WORK * wk );
static void InitPageSet( PST_WORK * wk );

static void PageChangeWrite( PST_WORK * wk, u8 next );
static void PageChange( PST_WORK * wk, s8 mv );
static u8 SubButtonPageChg( PST_WORK * wk );

static void PageBgScreenChg( PST_WORK * wk );
static void HPGaugeSet( PST_WORK * wk );
static void ExpGaugeSet( PST_WORK * wk );

static void PokeChange( PST_WORK * wk, s8 mv );
static s8 PokeSearch( PST_WORK * wk, s8 mv );
static s8 PokeSearch_PokemonParam( PST_WORK * wk, s8 mv );
static s8 PokeSearch_PokeParty( PST_WORK * wk, s8 mv );
static s8 PokeSearch_PokemonPasoParam( PST_WORK * wk, s8 mv );

static u8 WazaInfoScrollIn( PST_WORK * wk );
static u8 WazaInfoScrollOut( PST_WORK * wk );
static u8 WazaCursorMove( PST_WORK * wk, s8 mv );
static void WazaInfoChange( PST_WORK * wk );
static void WazaChangeMain( PST_WORK * wk );
static void WazaInfoPut( PST_WORK * wk );


//============================================================================================
//	O[oϐ
//============================================================================================
// vZX`f[^
const PROC_DATA PokeStatusProcData = {
	PokeStatusProc_Init,
	PokeStatusProc_Main,
	PokeStatusProc_End,
	NO_OVERLAY_ID,
};


//============================================================================================
//	vZX֐
//============================================================================================

//--------------------------------------------------------------------------------------------
/**
 * vZX֐F
 *
 * @param	proc	vZXf[^
 * @param	seq		V[PX
 *
 * @return	
 */
//--------------------------------------------------------------------------------------------
static PROC_RESULT PokeStatusProc_Init( PROC * proc, int * seq )
{
	PST_WORK * wk;

	sys_VBlankFuncChange( NULL, NULL );	// VBlankZbg
	sys_HBlankIntrStop();	//HBlank荞ݒ~

	GF_Disp_GX_VisibleControlInit();
	GF_Disp_GXS_VisibleControlInit();
	GX_SetVisiblePlane( 0 );
	GXS_SetVisiblePlane( 0 );

	sys_KeyRepeatSpeedSet( 4, 8 );
	sys_CreateHeap( HEAPID_BASE_APP, HEAPID_POKESTATUS, 0x30000 );

	wk = PROC_AllocWork( proc, sizeof(PST_WORK), HEAPID_POKESTATUS );
	memset( wk, 0, sizeof(PST_WORK) );
	wk->dat = PROC_GetParentWork( proc );
	wk->bgl = GF_BGL_BglIniAlloc( HEAPID_POKESTATUS );

	APP_WipeStart( APP_WIPE_IN, HEAPID_POKESTATUS, &wk->wipe_flg );

	FontProc_LoadBitData( FONT_SYSTEM, HEAPID_POKESTATUS );

	/*************************/
//	wk->dat->mode = PST_MODE_NORMAL;
//	wk->dat->page_flg = 0xff;

//	wk->dat->mode = PST_MODE_PORUTO;
//	wk->dat->mode = PST_MODE_CONDITION;
//	wk->dat->page_flg = 0x48;

//	wk->dat->mode = PST_MODE_WAZAADD;
//	wk->dat->page_flg = 0x57;
	/*************************/

	InitTPSystem();						// ^b`plVXe
	InitTPNoBuff(4);

	PokeStatusVramBankSet();			// VRAM蓖
	PokeStatusBgSet( wk->bgl );			// BG
	PokeStatusBgGraphicSet( wk );		// BGOtBbNZbg

	PokeStatus_3DInit( wk );			// 3D

	MsgBufCreate( wk );					// bZ[W֘A쐬

	PokeDataSet( wk );					// |Pf[^擾

	PokeStatusCellActInit( wk );		// ZAN^[
	PokeStatusCellActSet( wk );			// ZAN^[Zbg
	PokeStatus_SoftSpriteSet( wk );		// \tgEFAXvCgZbg
	PokeStatusBallAnmChg( wk );			// ߊl{[؂ւ
	PokeStatus_StIconChg( wk );			// ԈُACR؂ւ

	PokeStatus_DefaultBmpAdd( wk );
	InitPageSet( wk );					// y[W쐬

	PokeStatus_SubButtonMake( wk );		// ʂ̃{^Zbg

	PokeStatus_CameraInit( wk );		// JZbg

	sys_VBlankFuncChange( PokeStatusVBlank, wk );	// VBlankZbg

	GF_Disp_DispOn();

	return PROC_RES_FINISH;
}

//--------------------------------------------------------------------------------------------
/**
 * vZX֐FC
 *
 * @param	proc	vZXf[^
 * @param	seq		V[PX
 *
 * @return	
 */
//--------------------------------------------------------------------------------------------
static PROC_RESULT PokeStatusProc_Main( PROC * proc, int * seq )
{
	PST_WORK * wk  = PROC_GetWork( proc );

	switch( *seq ){
	case SEQ_IN:			// tF[h҂
		*seq = PST_SeqIn( wk );
		break;

	case SEQ_MAIN:			// CRg[
		*seq = PST_SeqMain( wk );
		break;

	case SEQ_WAZAIN:		// ZC҂
		*seq = PST_SeqWazaIn( wk );
		break;

	case SEQ_WAZAOUT:		// ZAEg҂
		*seq = PST_SeqWazaOut( wk );
		break;

	case SEQ_WAZA_SELECT:	// ZI
		*seq = PST_SeqWazaSelect( wk );
		break;

	case SEQ_WAZA_CHANGE:	// ZւI
		*seq = PST_SeqWazaChange( wk );
		break;

	case SEQ_WAZA_OBOE:		// Zo/Y
		*seq = PST_SeqWazaAdd( wk );
		break;

	case SEQ_WAZA_OBOE_ERROR:	// ZoG[
		*seq = PST_SeqWazaAddError( wk );
		break;

	case SEQ_BUTTON_ANM:	// ʂ̃{^Aj҂
		*seq = PST_SeqButtnAnm( wk );
		break;

	case SEQ_COND_INIT:		// L[҂
		*seq = PST_SeqCondUpInit( wk );
		break;

	case SEQ_COND_MSG:		// bZ[W\
		*seq = PST_SeqCondUpMsgPut( wk );
		break;

	case SEQ_COND_END:		// RfBVύXI
		*seq = PST_SeqCondUpEnd( wk );
		break;

	case SEQ_OUT:			// ItF[h҂
		if( PST_SeqOut( wk ) == TRUE ){
			return PROC_RES_FINISH;
		}
		break;
	}

	PokeStatus_CellActAnm( wk );
	PokeStatus_KezuyaAnm( wk );

	CATS_Draw( wk->crp );

	PokeStatus_3DMain( wk );

	return PROC_RES_CONTINUE;
}

//--------------------------------------------------------------------------------------------
/**
 * vZX֐FI
 *
 * @param	proc	vZXf[^
 * @param	seq		V[PX
 *
 * @return	
 */
//--------------------------------------------------------------------------------------------
static PROC_RESULT PokeStatusProc_End( PROC * proc, int * seq )
{
	PST_WORK * wk  = PROC_GetWork( proc );

	sys_VBlankFuncChange( NULL, NULL );	// VBlankZbg

	PokeStatus_3DRelease( wk );			// 3D폜
	PokeStatusCellActRerease( wk );		// ZAN^[폜
//	PokeStatusBmpExit( wk );			// BMPEBhEJ
	PokeStatus_BmpFreeAll( wk );
	PokeStatusBgExit( wk->bgl );		// BGL폜
	StopTP();							// ^b`plI
	DellVramTransferManager();

	MsgBufRelease( wk );

	FontProc_UnloadBitData( FONT_SYSTEM );

/*
	TCB_Delete( wk->vtask );
*/

	PROC_FreeWork( proc );				// [NJ

	sys_DeleteHeap( HEAPID_POKESTATUS );

	return PROC_RES_FINISH;
}


//--------------------------------------------------------------------------------------------
/**
 * CV[PXFtF[h҂
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static int PST_SeqIn( PST_WORK * wk )
{
	if( wk->wipe_flg == TRUE ){
		if( wk->dat->mode == PST_MODE_WAZAADD ){
			return SEQ_WAZA_OBOE;
		}else if( wk->dat->mode == PST_MODE_CONDITION ){
			return SEQ_COND_INIT;
		}else{
			return SEQ_MAIN;
		}
	}
	return SEQ_IN;
}

//--------------------------------------------------------------------------------------------
/**
 * CV[PXFCRg[
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static int PST_SeqMain( PST_WORK * wk )
{
	if( sys.trg & PAD_KEY_LEFT ){
		PageChange( wk, -1 );
		return SEQ_MAIN;
	}
	if( sys.trg & PAD_KEY_RIGHT ){
		PageChange( wk, 1 );
		return SEQ_MAIN;
	}

	if( sys.trg & PAD_KEY_UP ){
		PokeChange( wk, -1 );
		return SEQ_MAIN;
	}
	if( sys.trg & PAD_KEY_DOWN ){
		PokeChange( wk, 1 );
		return SEQ_MAIN;
	}

	if( sys.trg & PAD_BUTTON_CANCEL ){
		Snd_SePlay( PST_SE_CANCEL );
		wk->dat->ret_mode = PST_RET_CANCEL;
		APP_WipeStart( APP_WIPE_OUT, HEAPID_POKESTATUS, &wk->wipe_flg );
		return SEQ_OUT;
	}

	if( sys.trg & PAD_BUTTON_DECIDE ){
		if( wk->dat->mode == PST_MODE_PORUTO && wk->page == PST_PAGE_CONDITION ){
			Snd_SePlay( PST_SE_DECIDE );
			wk->dat->ret_mode = PST_RET_DECIDE;
			APP_WipeStart( APP_WIPE_OUT, HEAPID_POKESTATUS, &wk->wipe_flg );
			return SEQ_OUT;
		}
		if( wk->page == PST_PAGE_B_SKILL || wk->page == PST_PAGE_C_SKILL ){
			Snd_SePlay( PST_SE_DECIDE );
			wk->sub_seq = 0;
			return SEQ_WAZAIN;
		}else if( wk->page == PST_PAGE_RET ){
			Snd_SePlay( PST_SE_CANCEL );
			wk->dat->ret_mode = PST_RET_CANCEL;
			APP_WipeStart( APP_WIPE_OUT, HEAPID_POKESTATUS, &wk->wipe_flg );
			return SEQ_OUT;
		}
	}

	if( SubButtonPageChg( wk ) == TRUE ){
		wk->sub_seq = SEQ_MAIN;
		return SEQ_BUTTON_ANM;
	}

	return SEQ_MAIN;
}

//--------------------------------------------------------------------------------------------
/**
 * CV[PXFZC҂
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static int PST_SeqWazaIn( PST_WORK * wk )
{
	if( WazaInfoScrollIn( wk ) == TRUE ){
		return SEQ_WAZA_SELECT;
	}
	return SEQ_WAZAIN;
}

//--------------------------------------------------------------------------------------------
/**
 * CV[PXFZAEg҂
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static int PST_SeqWazaOut( PST_WORK * wk )
{
	if( WazaInfoScrollOut( wk ) == TRUE ){
		return SEQ_MAIN;
	}
	return SEQ_WAZAOUT;
}

//--------------------------------------------------------------------------------------------
/**
 * CV[PXFZI
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static int PST_SeqWazaSelect( PST_WORK * wk )
{
	if( sys.trg & PAD_KEY_UP ){
		if( WazaCursorMove( wk, -1 ) == TRUE ){
			Snd_SePlay( PST_SE_CURSOR_MOVE );
			WazaInfoChange( wk );
		}
		return SEQ_WAZA_SELECT;
	}

	if( sys.trg & PAD_KEY_DOWN ){
		if( WazaCursorMove( wk, 1 ) == TRUE ){
			Snd_SePlay( PST_SE_CURSOR_MOVE );
			WazaInfoChange( wk );
		}
		return SEQ_WAZA_SELECT;
	}

	if( sys.trg & PAD_BUTTON_DECIDE ){
		if( wk->cur_pos == 4 ){
			Snd_SePlay( PST_SE_CANCEL );
			wk->sub_seq = 0;
			return SEQ_WAZAOUT;
		}else{
			Snd_SePlay( PST_SE_DECIDE );
			PokeStatus_SubWazaCursorSet( wk );
			wk->sub_cur = wk->cur_pos;
			return SEQ_WAZA_CHANGE;
		}
	}

	if( sys.trg & PAD_BUTTON_CANCEL ){
		Snd_SePlay( PST_SE_CANCEL );
		wk->sub_seq = 0;
		return SEQ_WAZAOUT;
	}

	return SEQ_WAZA_SELECT;
}

//--------------------------------------------------------------------------------------------
/**
 * CV[PXFZւI
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static int PST_SeqWazaChange( PST_WORK * wk )
{
	if( sys.trg & PAD_KEY_UP ){
		if( WazaCursorMove( wk, -1 ) == TRUE ){
			Snd_SePlay( PST_SE_CURSOR_MOVE );
			WazaInfoChange( wk );
		}
		return SEQ_WAZA_CHANGE;
	}

	if( sys.trg & PAD_KEY_DOWN ){
		if( WazaCursorMove( wk, 1 ) == TRUE ){
			Snd_SePlay( PST_SE_CURSOR_MOVE );
			WazaInfoChange( wk );
		}
		return SEQ_WAZA_CHANGE;
	}

	if( sys.trg & PAD_BUTTON_DECIDE ){
		CLACT_SetDrawFlag( wk->awp[PST_OBJ_WAZACURSOR2], 0 );
		if( wk->cur_pos != 4 && wk->cur_pos != wk->sub_cur ){
			Snd_SePlay( PST_SE_DECIDE );
			WazaChangeMain( wk );
			PokeStatus_WazaTypePosChange( wk, wk->cur_pos, wk->sub_cur );
//			PokeStatus_SkillChgRewrite( wk );
			WazaInfoChange( wk );
		}else{
			Snd_SePlay( PST_SE_CANCEL );
		}
		return SEQ_WAZA_SELECT;
	}

	if( sys.trg & PAD_BUTTON_CANCEL ){
		Snd_SePlay( PST_SE_CANCEL );
		CLACT_SetDrawFlag( wk->awp[PST_OBJ_WAZACURSOR2], 0 );
		return SEQ_WAZA_SELECT;
	}
	return SEQ_WAZA_CHANGE;
}

//--------------------------------------------------------------------------------------------
/**
 * CV[PXFZo/Y
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static int PST_SeqWazaAdd( PST_WORK * wk )
{
	if( sys.trg & PAD_KEY_LEFT ){
//		PageChange( wk, -1 );
		return SEQ_WAZA_OBOE;
	}
	if( sys.trg & PAD_KEY_RIGHT ){
//		PageChange( wk, 1 );
		return SEQ_WAZA_OBOE;
	}

	if( sys.trg & PAD_KEY_UP ){
		if( WazaCursorMove( wk, -1 ) == TRUE ){
			Snd_SePlay( PST_SE_CURSOR_MOVE );
			WazaInfoChange( wk );
		}
		return SEQ_WAZA_OBOE;
	}

	if( sys.trg & PAD_KEY_DOWN ){
		if( WazaCursorMove( wk, 1 ) == TRUE ){
			Snd_SePlay( PST_SE_CURSOR_MOVE );
			WazaInfoChange( wk );
		}
		return SEQ_WAZA_OBOE;
	}

	if( sys.trg & PAD_BUTTON_DECIDE ){
		Snd_SePlay( PST_SE_DECIDE );
		if( wk->cur_pos != 4 ){
			if( HidenWazaCheck( wk->pp.waza[wk->cur_pos] ) == TRUE ){
				CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_KIND], 0 );
				PokeStatus_WazaOboeErrorMsgPut( wk );
				return SEQ_WAZA_OBOE_ERROR;
			}
		}
		wk->dat->ret_sel = wk->cur_pos;
		APP_WipeStart( APP_WIPE_OUT, HEAPID_POKESTATUS, &wk->wipe_flg );
		return SEQ_OUT;
	}

	if( sys.trg & PAD_BUTTON_CANCEL ){
		Snd_SePlay( PST_SE_CANCEL );
		wk->dat->ret_sel = 4;
		APP_WipeStart( APP_WIPE_OUT, HEAPID_POKESTATUS, &wk->wipe_flg );
		return SEQ_OUT;
	}

/*
	if( SubButtonPageChg( wk ) == TRUE ){
		wk->sub_seq = SEQ_WAZA_OBOE;
		return SEQ_BUTTON_ANM;
	}
*/

	return SEQ_WAZA_OBOE;
}

static int PST_SeqWazaAddError( PST_WORK * wk )
{
	if( sys.trg & (PAD_BUTTON_DECIDE|PAD_BUTTON_CANCEL) ){
		WazaInfoPut( wk );
		return SEQ_WAZA_OBOE;
	}
	return SEQ_WAZA_OBOE_ERROR;
}


//--------------------------------------------------------------------------------------------
/**
 * CV[PXFʂ̃{^Aj҂
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static int PST_SeqButtnAnm( PST_WORK * wk )
{
	if( PokeStatus_SubButtonAnm( wk ) == TRUE ){
		return wk->sub_seq;
	}

	if( wk->btn_seq == 1 ){
		PageChangeWrite( wk, wk->btn_pos );
	}

	return SEQ_BUTTON_ANM;
}



static int PST_SeqCondUpInit( PST_WORK * wk )
{
	if( sys.trg & (PAD_BUTTON_DECIDE|PAD_BUTTON_CANCEL) ){

		wk->pp.style     = 0xff;
		wk->pp.beautiful = 0xff;
		wk->pp.cute      = 0xff;
		wk->pp.clever    = 0xff;
		wk->pp.strong    = 0xff;

		wk->sub_seq = 0x1f;

		PokeStatus_ConditionParamInit( wk );

		// bEBhEZbg
		TalkFontPaletteLoad( PALTYPE_MAIN_BG, PST_PAL_TALKFONT * 32, HEAPID_POKESTATUS );
		TalkWinGraphicSet(
			wk->bgl, GF_BGL_FRAME1_M, PST_CGX_TALKWIN, PST_PAL_TALKWIN, 0, HEAPID_POKESTATUS );

		PokeStatus_CondUpObjPut( wk );

		return SEQ_COND_MSG;
	}
	return SEQ_COND_INIT;
}

static int PST_SeqCondUpMsgPut( PST_WORK * wk )
{
	u8	i;

	if( sys.trg & (PAD_BUTTON_DECIDE|PAD_BUTTON_CANCEL) ){
		if( wk->sub_seq & 0x80 ){
			PokeStatus_CondUpMsgPut( wk, 0xff );
			return SEQ_COND_END;
		}
		for( i=0; i<5; i++ ){
			if( ( wk->sub_seq & (1<<i) ) ){
				PokeStatus_CondUpMsgPut( wk, i );
				wk->sub_seq ^= (1<<i);
				if( wk->sub_seq == 0 ){
					return SEQ_COND_END;
				}
				break;
			}
		}
	}
	return SEQ_COND_MSG;
}

static int PST_SeqCondUpEnd( PST_WORK * wk )
{
	if( sys.trg & (PAD_BUTTON_DECIDE|PAD_BUTTON_CANCEL) ){
//		Snd_SePlay( PST_SE_CANCEL );
		APP_WipeStart( APP_WIPE_OUT, HEAPID_POKESTATUS, &wk->wipe_flg );
		return SEQ_OUT;
	}
	return SEQ_COND_END;
}




//--------------------------------------------------------------------------------------------
/**
 * CV[PXFItF[h҂
 *
 * @param	wk		[N
 *
 * @return	ڍsV[PX
 */
//--------------------------------------------------------------------------------------------
static u8 PST_SeqOut( PST_WORK * wk )
{
	if( wk->wipe_flg == TRUE ){
		return TRUE;
	}
	return FALSE;
}



//--------------------------------------------------------------------------------------------
/**
 * VBlank֐
 *
 * @param	none
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
/*
static void PokeListVBlank( TCB_PTR tcb, void * work )
{
	PLIST_WORK * wk = work;

	if( wk->vblank_func != NULL ){
		wk->vblank_func( work );
	}

	CATS_Draw( wk->icon_crp );
	CATS_Draw( wk->crp );

	// ZAN^[
	DoVramTransferManager();	// Vram]}l[W[s
	CATS_RenderOamTrans();
	
	OS_SetIrqCheckFlag( OS_IE_V_BLANK );
}
*/

static void PokeStatusVBlank( void * work )
{
	PST_WORK * wk = work;

	GF_BGL_VBlankFunc( wk->bgl );
	SoftSpriteTextureTrans( wk->p3d.ssm );

	// ZAN^[
	DoVramTransferManager();	// Vram]}l[W[s
	CATS_RenderOamTrans();
	
	OS_SetIrqCheckFlag( OS_IE_V_BLANK );
}



//--------------------------------------------------------------------------------------------
/**
 * VRAMݒ
 *
 * @param	none
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void PokeStatusVramBankSet(void)
{
	GF_BGL_DISPVRAM tbl = {
		GX_VRAM_BG_128_A,				// C2DGWBG
		GX_VRAM_BGEXTPLTT_NONE,			// C2DGWBGgpbg

		GX_VRAM_SUB_BG_128_C,			// Tu2DGWBG
		GX_VRAM_SUB_BGEXTPLTT_NONE,		// Tu2DGWBGgpbg

		GX_VRAM_OBJ_64_E,				// C2DGWOBJ
		GX_VRAM_OBJEXTPLTT_NONE,		// C2DGWOBJgpbg

		GX_VRAM_SUB_OBJ_16_I,			// Tu2DGWOBJ
		GX_VRAM_SUB_OBJEXTPLTT_NONE,	// Tu2DGWOBJgpbg

		GX_VRAM_TEX_0_B,				// eNX`C[WXbg
		GX_VRAM_TEXPLTT_01_FG			// eNX`pbgXbg
	};
	GF_Disp_SetBank( &tbl );
}

//--------------------------------------------------------------------------------------------
/**
 * BGݒ
 *
 * @param	ini		BGLf[^
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void PokeStatusBgSet( GF_BGL_INI * ini )
{
	{	// BG SYSTEM
		GF_BGL_SYS_HEADER BGsys_data = {
			GX_DISPMODE_GRAPHICS, GX_BGMODE_0, GX_BGMODE_0, GX_BG0_AS_3D,
		};
		GF_BGL_InitBG( &BGsys_data );
	}

/*
	BG0 = 3D
	BG1 = font
	BG2 = SubBG
	BG3 = BG
*/

	// 
	{	// FONT (BMP)
		GF_BGL_BGCNT_HEADER TextBgCntDat = {
			0, 0, 0x800, 0, GF_BGL_SCRSIZ_256x256, GX_BG_COLORMODE_16,
			GX_BG_SCRBASE_0xf800, GX_BG_CHARBASE_0x10000, GX_BG_EXTPLTT_01,
			0, 0, 0, FALSE
		};
		GF_BGL_BGControlSet( ini, GF_BGL_FRAME1_M, &TextBgCntDat, GF_BGL_MODE_TEXT );
		GF_BGL_ScrClear( ini, GF_BGL_FRAME1_M );
	}

	{	// SUB PAGE (CHAR)
		GF_BGL_BGCNT_HEADER TextBgCntDat = {
			0, 0, 0x1000, 0, GF_BGL_SCRSIZ_512x256, GX_BG_COLORMODE_16,
			GX_BG_SCRBASE_0xe800, GX_BG_CHARBASE_0x00000, GX_BG_EXTPLTT_01,
			1, 0, 0, FALSE
		};
		GF_BGL_BGControlSet( ini, GF_BGL_FRAME2_M, &TextBgCntDat, GF_BGL_MODE_TEXT );
		GF_BGL_ScrClear( ini, GF_BGL_FRAME2_M );
		GF_BGL_ScrollReq( ini, GF_BGL_FRAME2_M, GF_BGL_SCROLL_X_SET, 136 );
	}

	{	// BG (CHAR)
		GF_BGL_BGCNT_HEADER TextBgCntDat = {
			0, 0, 0x800, 0, GF_BGL_SCRSIZ_256x256, GX_BG_COLORMODE_16,
			GX_BG_SCRBASE_0xe000, GX_BG_CHARBASE_0x00000, GX_BG_EXTPLTT_01,
			3, 0, 0, FALSE
		};
		GF_BGL_BGControlSet( ini, GF_BGL_FRAME3_M, &TextBgCntDat, GF_BGL_MODE_TEXT );
	}

	// 
	{	// BUTTON (CHAR)
		GF_BGL_BGCNT_HEADER TextBgCntDat = {
			0, 0, 0x800, 0, GF_BGL_SCRSIZ_256x256, GX_BG_COLORMODE_16,
			GX_BG_SCRBASE_0xf800, GX_BG_CHARBASE_0x10000, GX_BG_EXTPLTT_01,
			0, 0, 0, FALSE
		};
		GF_BGL_BGControlSet( ini, GF_BGL_FRAME0_S, &TextBgCntDat, GF_BGL_MODE_TEXT );
		GF_BGL_ScrClear( ini, GF_BGL_FRAME0_S );
	}
	{	// BG (CHAR)
		GF_BGL_BGCNT_HEADER TextBgCntDat = {
			0, 0, 0x800, 0, GF_BGL_SCRSIZ_256x256, GX_BG_COLORMODE_16,
			GX_BG_SCRBASE_0xf000, GX_BG_CHARBASE_0x00000, GX_BG_EXTPLTT_01,
			1, 0, 0, FALSE
		};
		GF_BGL_BGControlSet( ini, GF_BGL_FRAME1_S, &TextBgCntDat, GF_BGL_MODE_TEXT );
	}

	GF_BGL_ClearCharSet( GF_BGL_FRAME1_M, 32, 0, HEAPID_POKESTATUS );
	GF_BGL_ClearCharSet( GF_BGL_FRAME0_S, 32, 0, HEAPID_POKESTATUS );
}


//--------------------------------------------------------------------------------------------
/**
 * BG
 *
 * @param	ini		BGLf[^
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void PokeStatusBgExit( GF_BGL_INI * ini )
{
	GF_Disp_GX_VisibleControlInit();
	GF_Disp_GXS_VisibleControlInit();

	GF_BGL_BGControlExit( ini, GF_BGL_FRAME1_S );
	GF_BGL_BGControlExit( ini, GF_BGL_FRAME0_S );
	GF_BGL_BGControlExit( ini, GF_BGL_FRAME3_M );
	GF_BGL_BGControlExit( ini, GF_BGL_FRAME2_M );
	GF_BGL_BGControlExit( ini, GF_BGL_FRAME1_M );

	sys_FreeMemory( HEAPID_POKESTATUS, ini );
}

//--------------------------------------------------------------------------------------------
/**
 * OtBbNf[^Zbg
 *
 * @param	wk		|PXgʂ̃[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void PokeStatusBgGraphicSet( PST_WORK * wk )
{
	ArcUtil_BgCharSet(
		ARC_PSTATUS_GRA, NARC_pst_gra_p_st_bg_NCGR,
		wk->bgl, GF_BGL_FRAME3_M, 0, 0, 0, HEAPID_POKESTATUS );
	ArcUtil_ScrnSet(
		ARC_PSTATUS_GRA, NARC_pst_gra_p_st_s0_NSCR,
		wk->bgl, GF_BGL_FRAME3_M, 0, 0, 0, HEAPID_POKESTATUS );
	ArcUtil_PalSet(
		ARC_PSTATUS_GRA, NARC_pst_gra_p_st_bg_NCLR,
		PALTYPE_MAIN_BG, 0, 0, HEAPID_POKESTATUS );

	ArcUtil_ScrnSet(
		ARC_PSTATUS_GRA, NARC_pst_gra_p_st_s4b_NSCR,
		wk->bgl, GF_BGL_FRAME2_M, 0, 0, 0, HEAPID_POKESTATUS );

	ArcUtil_BgCharSet(
		ARC_PSTATUS_GRA, NARC_pst_gra_p_st_but_NCGR,
		wk->bgl, GF_BGL_FRAME0_S, 0, 0, 0, HEAPID_POKESTATUS );
	ArcUtil_BgCharSet(
		ARC_PSTATUS_GRA, NARC_pst_gra_p_st_sub_bg_NCGR,
		wk->bgl, GF_BGL_FRAME1_S, 0, 0, 0, HEAPID_POKESTATUS );
	ArcUtil_ScrnSet(
		ARC_PSTATUS_GRA, NARC_pst_gra_p_st_sub_NSCR,
		wk->bgl, GF_BGL_FRAME1_S, 0, 0, 0, HEAPID_POKESTATUS );
	ArcUtil_PalSet(
		ARC_PSTATUS_GRA, NARC_pst_gra_p_st_sub_NCLR,
		PALTYPE_SUB_BG, 0, 0, HEAPID_POKESTATUS );
}

//--------------------------------------------------------------------------------------------
/**
 * bZ[W֘A쐬
 *
 * @param	wk		[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void MsgBufCreate( PST_WORK * wk )
{
	// bZ[W}l[W쐬
	wk->msg_man = MSGMAN_Create(
					MSGMAN_TYPE_NORMAL, ARC_MSG, NARC_msg_pokestatus_dat, HEAPID_POKESTATUS );
	// 8x8tHg쐬
	wk->num_font = NUMFONT_Create(
					FBMP_COL_BLACK, FBMP_COL_BLK_SDW, FBMP_COL_NULL, HEAPID_POKESTATUS );

	wk->wset = WORDSET_Create( HEAPID_POKESTATUS );

	wk->pp.monsname = STRBUF_Create( BUFLEN_POKEMON_NAME, HEAPID_POKESTATUS );
	wk->pp.nickname = STRBUF_Create( BUFLEN_POKEMON_NAME, HEAPID_POKESTATUS );
	wk->pp.oya = STRBUF_Create( BUFLEN_PERSON_NAME, HEAPID_POKESTATUS );

	wk->msg_buf = STRBUF_Create( TMP_MSGBUF_SIZ, HEAPID_POKESTATUS );

	wk->wman = MSGMAN_Create(
					MSGMAN_TYPE_NORMAL, ARC_MSG, NARC_msg_wazaname_dat, HEAPID_POKESTATUS );
}

static void MsgBufRelease( PST_WORK * wk )
{
	MSGMAN_Delete( wk->wman );
	MSGMAN_Delete( wk->msg_man );
	NUMFONT_Delete( wk->num_font );
	WORDSET_Delete( wk->wset );
	STRBUF_Delete( wk->pp.monsname );
	STRBUF_Delete( wk->pp.nickname );
	STRBUF_Delete( wk->pp.oya );
	STRBUF_Delete( wk->msg_buf );
}

//--------------------------------------------------------------------------------------------
/**
 * |Pf[^擾
 *
 * @param	wk		[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void PokeDataSet( PST_WORK * wk )
{
	void * pp = PokeStatusPokeParamGet( wk );

	if( wk->dat->ppt == PST_PP_TYPE_POKEPASO ){
		PokeDataSet_PokePasoParam( wk, pp, &wk->pp );
	}else{
		PokeDataSet_PokemonParam( wk, pp, &wk->pp );
	}
}

//--------------------------------------------------------------------------------------------
/**
 * |Pf[^擾 ( POKEMON_PASO_PARAM )
 *
 * @param	ppp		POKEMON_PASO_PARAM
 * @param	dat		f[^i[ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void PokeDataSet_PokePasoParam(
				PST_WORK * wk, POKEMON_PASO_PARAM * ppp, PST_POKEDATA * dat )
{
	POKEMON_PARAM * pp = PokemonParam_AllocWork( HEAPID_POKESTATUS );

	PokeReplace( ppp, pp );
	PokeDataSet_PokemonParam( wk, pp, dat );
	sys_FreeMemoryEz( pp );
}

//--------------------------------------------------------------------------------------------
/**
 * |Pf[^擾 ( POKEMON_PARAM )
 *
 * @param	pp		POKEMON_PARAM
 * @param	dat		f[^i[ꏊ
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void PokeDataSet_PokemonParam(
				PST_WORK * wk, POKEMON_PARAM * pp, PST_POKEDATA * dat )
{
	POKEMON_PASO_PARAM * ppp;
	u16	i;
	u8	pp_count;

	dat->mons = (u16)PokeParaGet( pp, ID_PARA_monsno, NULL );

	ppp = PPPPointerGet(pp);

	MSGMAN_GetString( wk->msg_man, mes_status_02_05, wk->msg_buf );
	WORDSET_RegisterPokeMonsName( wk->wset, 0, ppp );
	WORDSET_ExpandStr( wk->wset, wk->pp.monsname, wk->msg_buf );

	MSGMAN_GetString( wk->msg_man, mes_status_01_01, wk->msg_buf );
	WORDSET_RegisterPokeNickName( wk->wset, 0, ppp );
	WORDSET_ExpandStr( wk->wset, wk->pp.nickname, wk->msg_buf );

	MSGMAN_GetString( wk->msg_man, mes_status_02_08, wk->msg_buf );
	WORDSET_RegisterPokeOyaName( wk->wset, 0, ppp );
	WORDSET_ExpandStr( wk->wset, wk->pp.oya, wk->msg_buf );

	dat->item = (u16)PokeParaGet( pp, ID_PARA_item, NULL );
	dat->lv   = (u8)PokeParaGet( pp, ID_PARA_level, NULL );
	dat->sex  = PokeSexGet( pp );
	dat->ball = (u8)PokeParaGet( pp, ID_PARA_get_ball, NULL );

	dat->type1 = PokePersonalParaGet( dat->mons, ID_PER_type1 );
	dat->type2 = PokePersonalParaGet( dat->mons, ID_PER_type2 );

	dat->id = PokeParaGet( pp, ID_PARA_id_no, NULL );
	dat->now_exp  = PokeParaGet( pp, ID_PARA_exp, NULL );
//	dat->next_exp = PokeParaNextLevelExpGet( pp );

	dat->now_lv_exp  = PokeLevelExpGet( dat->mons, dat->lv );
	dat->next_lv_exp = PokeLevelExpGet( dat->mons, dat->lv+1 );

	dat->hp  = (u16)PokeParaGet( pp, ID_PARA_hp, NULL );
	dat->mhp = (u16)PokeParaGet( pp, ID_PARA_hpmax, NULL );
	dat->atc = (u16)PokeParaGet( pp, ID_PARA_pow, NULL );
	dat->def = (u16)PokeParaGet( pp, ID_PARA_def, NULL );
	dat->spa = (u16)PokeParaGet( pp, ID_PARA_spepow, NULL );
	dat->spd = (u16)PokeParaGet( pp, ID_PARA_spedef, NULL );
	dat->agi = (u16)PokeParaGet( pp, ID_PARA_agi, NULL );

	dat->tokusei = (u8)PokeParaGet( pp, ID_PARA_speabino, NULL );
	dat->seikaku = PokeSeikakuGet( pp );

	for( i=0; i<4; i++ ){
		dat->waza[i] = (u16)PokeParaGet( pp, ID_PARA_waza1+i, NULL );
		dat->wpp[i]  = (u8)PokeParaGet( pp, ID_PARA_pp1+i, NULL );
		pp_count = (u8)PokeParaGet( pp, ID_PARA_pp_count1+i, NULL );
		dat->mpp[i]  = WT_PPMaxGet( dat->waza[i], pp_count );
	}

	dat->style     = (u8)PokeParaGet( pp, ID_PARA_style, NULL );		// 悳
	dat->beautiful = (u8)PokeParaGet( pp, ID_PARA_beautiful, NULL );	// 
	dat->cute      = (u8)PokeParaGet( pp, ID_PARA_cute, NULL );			// 킢
	dat->clever    = (u8)PokeParaGet( pp, ID_PARA_clever, NULL );		// 
	dat->strong    = (u8)PokeParaGet( pp, ID_PARA_strong, NULL );		// ܂
	dat->fur       = (u8)PokeParaGet( pp, ID_PARA_fur, NULL );			// щ

	dat->mark = PokeParaGet( pp, ID_PARA_mark, NULL );

	dat->st = PokeParaGet( pp, ID_PARA_condition, NULL );

	dat->tamago = PokeParaGet( pp, ID_PARA_tamago_flag, NULL );
}

//--------------------------------------------------------------------------------------------
/**
 * y[WZbg
 *
 * @param	wk		[N
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
static void InitPageSet( PST_WORK * wk )
{
	switch( wk->dat->mode ){
	case PST_MODE_NORMAL:		// ʏ
		wk->page = PST_PAGE_INFO;
		break;
	case PST_MODE_WAZAADD:		// Zo/ZY
		wk->page = PST_PAGE_B_SKILL;
		break;
	case PST_MODE_PORUTO:		// |ggp
	case PST_MODE_CONDITION:	// RfBVύXf
		wk->page = PST_PAGE_CONDITION;
		break;
	}

	PokeStatusTabPosChg( wk );			// ^uOBJ؂ւ
	PokeStatus_PageTypeIconDrow( wk );	// ACRZbg
	PokeStatus_TabArrowPut( wk );		// ^uʒuύX
	PokeStatus_MarkChg( wk );

	PokeStatus_KezuyaInit( wk );

	PokeStatus_PageBmpAdd( wk );
	PokeStatus_PageBmpPut( wk );

	PageBgScreenChg( wk );				// BG؂ւ
	PokeStatus_ConditionParamInit( wk );

	if( wk->dat->mode == PST_MODE_WAZAADD ){
		WazaInfoPut( wk );
	}
}



//--------------------------------------------------------------------------------------------
/**
 * y[W쐬
 *
 * @param	dat		Xe[^Xʃf[^
 * @param	tbl		y[We[u
 *
 * @return	none
 */
//--------------------------------------------------------------------------------------------
void PokeStatus_PageSet( PSTATUS_DATA * dat, const u8 * tbl )
{
	u8	i;

	dat->page_flg = 0;
	for( i=0; i<PST_PAGE_MAX; i++ ){
		if( tbl[i] == PST_PAGE_MAX ){ break; }
		dat->page_flg |= (1<<tbl[i]);
	}
}


u8 PokeStatus_PageCheck( PST_WORK * wk, u32 pos )
{
	return ( wk->dat->page_flg & (1<<pos) );
}


static void PageChangeWrite( PST_WORK * wk, u8 next )
{
	if( wk->page == next ){return; }

	PokeStatus_PageBmpFree( wk );
	wk->page = next;

	PokeStatusTabPosChg( wk );
	PokeStatus_PageTypeIconDrow( wk );
	PokeStatus_KezuyaInit( wk );
	PokeStatus_PageBmpAdd( wk );
	PokeStatus_PageBmpPut( wk );

	PageBgScreenChg( wk );
	PokeStatus_ConditionParamInit( wk );
}

static void PageChange( PST_WORK * wk, s8 mv )
{
	s8	next = wk->page;

	while(1){
		next += mv;
		if( next < PST_PAGE_INFO ){
			next = PST_PAGE_RET;
		}else if( next > PST_PAGE_RET ){
			next = PST_PAGE_INFO;
		}
		if( ( wk->dat->page_flg & (1<<next) ) != 0 ){
			break;
		}
	}

	Snd_SePlay( PST_SE_PAGE_MOVE );
	PokeStatus_SubButtonBgInit( wk );

	PageChangeWrite( wk, next );
}

static u8 SubButtonPageChg( PST_WORK * wk )
{
	u8	ret = PokeStatus_SubBottonPageGet( wk );

	if( ret == 0xff ){
		return FALSE;
	}

/*
	PokeStatus_PageBmpFree( wk );
	wk->page = ret;

	PokeStatusTabPosChg( wk );
	PokeStatus_PageTypeIconDrow( wk );
	PokeStatus_KezuyaInit( wk );
	PokeStatus_PageBmpAdd( wk );
	PokeStatus_PageBmpPut( wk );

	PageBgScreenChg( wk );
	PokeStatus_ConditionParamInit( wk );
*/

	wk->btn_cnt = 0;
	wk->btn_pos = ret;
	wk->btn_seq = 0;

	return TRUE;
}


static void PageBgScreenChg( PST_WORK * wk )
{
	{
		NNSG2dScreenData * dat;
		void * buf;

		buf = ArchiveDataLoadMalloc(
				ARC_PSTATUS_GRA, NARC_pst_gra_p_st_s0_NSCR+wk->page, HEAPID_POKESTATUS );
		NNS_G2dGetUnpackedScreenData( buf, &dat );
		GF_BGL_ScrWrite( wk->bgl, GF_BGL_FRAME3_M, dat->rawData, 0, 0, 32, 32 );
		GF_BGL_LoadScreenV_Req( wk->bgl, GF_BGL_FRAME3_M );
		sys_FreeMemoryEz( buf );
	}

	if( wk->page == PST_PAGE_INFO ){
		ExpGaugeSet( wk );
	}else if( wk->page == PST_PAGE_PARAM ){
		HPGaugeSet( wk );
	}
}

#define	HP_CHR_PALETTE	( 0xa000 )
#define	HP_CHR_GREEN	( 0xc0 )
#define	HP_CHR_YELLOW	( 0xe0 )
#define	HP_CHR_RED		( 0x100 )
#define	HP_DOT_CHR_MAX	( 6 )
#define	HP_DOT_MAX		( 48 )
#define	HP_GAUGE_SCR_PX	( 24 )
#define	HP_GAUGE_SCR_PY	( 6 )

static void HPGaugeSet( PST_WORK * wk )
{
	u16	chr;
	u16	cgx;
	u8	dot;
	u8	i;

	switch( GetHPGaugeDottoColor( wk->pp.hp, wk->pp.mhp, HP_DOT_MAX ) ){
	case HP_DOTTO_MAX:		// 
	case HP_DOTTO_GREEN:
	case HP_DOTTO_NULL:
		chr = HP_CHR_GREEN | HP_CHR_PALETTE;
		break;
	case HP_DOTTO_YELLOW:	// 
		chr = HP_CHR_YELLOW | HP_CHR_PALETTE;
		break;
	case HP_DOTTO_RED:		// 
		chr = HP_CHR_RED | HP_CHR_PALETTE;
		break;
	}

	dot = GetNumDotto( wk->pp.hp, wk->pp.mhp, HP_DOT_MAX );

	for( i=0; i<HP_DOT_CHR_MAX; i++ ){
		if( dot >= 8 ){
			cgx = chr + 8;
		}else{
			cgx = chr + dot;
		}

		GF_BGL_ScrFill(
			wk->bgl, GF_BGL_FRAME3_M, cgx,
			HP_GAUGE_SCR_PX+i, HP_GAUGE_SCR_PY, 1, 1, GF_BGL_SCRWRT_PALIN );

		if( dot < 8 ){
			dot = 0;
		}else{
			dot -= 8;
		}
	}

	GF_BGL_LoadScreenV_Req( wk->bgl, GF_BGL_FRAME3_M );
}


#define	EXP_CGX				( 0xac )
#define	EXP_DOT_CHR_MAX		( 7 )
#define	EXP_DOT_MAX			( 56 )
#define	EXP_GAUGE_SCR_PX	( 23 )
#define	EXP_GAUGE_SCR_PY	( 23 )

static void ExpGaugeSet( PST_WORK * wk )
{
	u32	max;
	u32	now;
	u16	cgx;
	u8	dot;
	u8	i;

	max = wk->pp.next_lv_exp - wk->pp.now_lv_exp;
	now = wk->pp.now_exp - wk->pp.now_lv_exp;

	dot = GetNumDotto( now, max, EXP_DOT_MAX );

	for( i=0; i<EXP_DOT_CHR_MAX; i++ ){
		if( dot >= 8 ){
			cgx = EXP_CGX + 8;
		}else{
			cgx = EXP_CGX + dot;
		}

		GF_BGL_ScrFill(
			wk->bgl, GF_BGL_FRAME3_M, cgx,
			EXP_GAUGE_SCR_PX+i, EXP_GAUGE_SCR_PY, 1, 1, GF_BGL_SCRWRT_PALIN );

		if( dot < 8 ){
			dot = 0;
		}else{
			dot -= 8;
		}
	}

	GF_BGL_LoadScreenV_Req( wk->bgl, GF_BGL_FRAME3_M );
}



static void PokeChange( PST_WORK * wk, s8 mv )
{
	s8	pos = PokeSearch( wk, mv );

	if( pos == -1 ){
		return;
	}

	Snd_SePlay( PST_SE_PAGE_MOVE );

	wk->dat->pos = (u8)pos;

	PokeDataSet( wk );

	PokeStatus_NickNamePut( wk );
	PokeStatus_LvPut( wk );
	PokeStatus_ItemPut( wk );

	if( wk->page == PST_PAGE_INFO ){
		ExpGaugeSet( wk );
	}else if( wk->page == PST_PAGE_PARAM ){
		HPGaugeSet( wk );
	}else if( wk->page == PST_PAGE_CONDITION ){
		PokeStatus_ConditionParamMake( wk );
	}

	PokeStatus_SoftSpriteReset( wk );

	PokeStatus_PokeTypeChange( wk );
	PokeStatus_PageTypeIconDrow( wk );
	PokeStasus_PokeIconChange( wk );
	PokeStatusBallAnmChg( wk );
	PokeStatus_StIconChg( wk );
	PokeStatus_KezuyaInit( wk );
	PokeStatus_MarkChg( wk );

	PokeStatus_PageBmpPut( wk );
}

static s8 PokeSearch( PST_WORK * wk, s8 mv )
{
	switch( wk->dat->ppt ){
	case PST_PP_TYPE_POKEPARAM:	// POKEMON_PARAM
		return PokeSearch_PokemonParam( wk, mv );
	case PST_PP_TYPE_POKEPARTY:	// POKEPARTY
		return PokeSearch_PokeParty( wk, mv );
	case PST_PP_TYPE_POKEPASO:	// POKEMON_PASO_PARAM
		return PokeSearch_PokemonPasoParam( wk, mv );
	}
	return -1;
}

static s8 PokeSearch_PokemonParam( PST_WORK * wk, s8 mv )
{
	POKEMON_PARAM * pp;
	s8	pos;

	pos = (s8)wk->dat->pos;
	pp  = (POKEMON_PARAM *)wk->dat->ppd;

	while(1){
		pos += mv;
		if( pos < 0 || pos >= wk->dat->max ){
			return -1;
		}
		pp = (POKEMON_PARAM *)( (u32)wk->dat->ppd + PokemonParamSizeGet()*pos );
		if( PokeParaGet( pp, ID_PARA_monsno, NULL ) != 0 ){
			break;
		}
	}

	return pos;
}

static s8 PokeSearch_PokeParty( PST_WORK * wk, s8 mv )
{
	POKEMON_PARAM * pp;
	s8	pos;

	pos = (s8)wk->dat->pos;

	while(1){
		pos += mv;
		if( pos < 0 || pos >= wk->dat->max ){
			return -1;
		}
		pp = PokeParty_GetMemberPointer( wk->dat->ppd, pos );
		if( PokeParaGet( pp, ID_PARA_monsno, NULL ) != 0 ){
			break;
		}
	}

	return pos;
}

static s8 PokeSearch_PokemonPasoParam( PST_WORK * wk, s8 mv )
{
	POKEMON_PASO_PARAM * ppp;
	s8	pos;

	pos = (s8)wk->dat->pos;
	ppp = (POKEMON_PASO_PARAM *)wk->dat->ppd;

	while(1){
		pos += mv;
		if( pos < 0 || pos >= wk->dat->max ){
			return -1;
		}
		ppp = (POKEMON_PASO_PARAM *)( (u32)wk->dat->ppd + PokemonPasoParamSizeGet()*pos );
		if( PokePasoParaGet( ppp, ID_PARA_monsno, NULL ) != 0 ){
			break;
		}
	}

	return pos;
}

void * PokeStatusPokeParamGet( PST_WORK * wk )
{
	switch( wk->dat->ppt ){
	case PST_PP_TYPE_POKEPARAM:	// POKEMON_PARAM
		return (void *)( (u32)wk->dat->ppd + ( PokemonParamSizeGet()*wk->dat->pos ) );

	case PST_PP_TYPE_POKEPARTY:	// POKEPARTY
		return (void *)PokeParty_GetMemberPointer( wk->dat->ppd, wk->dat->pos );

	case PST_PP_TYPE_POKEPASO:	// POKEMON_PASO_PARAM
		return (void *)((u32)wk->dat->ppd+(PokemonPasoParamSizeGet()*wk->dat->pos));
	}
	return NULL;
}



//--------------------------------------------------------------------------------------------
//	Z
//--------------------------------------------------------------------------------------------
static u8 WazaInfoScrollIn( PST_WORK * wk )
{
	switch( wk->sub_seq ){
	case 0:
		PokeStatus_TabArrowVanish( wk, 0 );
		GF_BGL_BmpWinOffVReq( &wk->def_win[WIN_LV] );
		GF_BGL_BmpWinOffVReq( &wk->def_win[WIN_MOTIMONO] );
		GF_BGL_BmpWinOffVReq( &wk->def_win[WIN_ITEMNAME] );
		GF_BGL_BmpWinOffVReq( &wk->def_win[WIN_P5_KUWASIKU] );
		wk->cur_pos = 0;
		wk->sub_seq = 1;
		break;
	case 1:
		{
			int	cnt = GF_BGL_ScrollGetX( wk->bgl, GF_BGL_FRAME2_M );

			if( cnt <= 64 ){
				GF_BGL_ScrollReq( wk->bgl, GF_BGL_FRAME2_M, GF_BGL_SCROLL_X_SET, 0 );
				wk->sub_seq = 2;
			}else{
				GF_BGL_ScrollReq( wk->bgl, GF_BGL_FRAME2_M, GF_BGL_SCROLL_X_DEC, 64 );
			}
		}
		break;
	case 2:
		PokeStatus_SkillCancelPut( wk );
		GF_BGL_BmpWinOnVReq( &wk->def_win[WIN_P5_IREKAE] );
		WazaInfoChange( wk );
		CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_POKE1], 1 );
		CATS_ObjectEnable( wk->awp[PST_OBJ_POKEICON], 1 );
		if( wk->pp.type1 != wk->pp.type2 ){
			CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_POKE2], 1 );
		}
		CLACT_SetDrawFlag( wk->awp[PST_OBJ_WAZACURSOR1], 1 );
		return TRUE;
	}
	return FALSE;
}

static u8 WazaInfoScrollOut( PST_WORK * wk )
{
	switch( wk->sub_seq ){
	case 0:
		CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_POKE1], 0 );
		CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_POKE2], 0 );
		CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_KIND], 0 );
		CATS_ObjectEnable( wk->awp[PST_OBJ_POKEICON], 0 );
		CLACT_SetDrawFlag( wk->awp[PST_OBJ_WAZACURSOR1], 0 );
		GF_BGL_BmpWinOffVReq( &wk->def_win[WIN_P5_IREKAE] );
		PokeStatus_SkillCancelDel( wk );
		PokeStatus_WazaInfoOff( wk );
		wk->sub_seq = 1;
		break;
	case 1:
		{
			int	cnt = GF_BGL_ScrollGetX( wk->bgl, GF_BGL_FRAME2_M );

			if( cnt >= 128 ){
				GF_BGL_ScrollReq( wk->bgl, GF_BGL_FRAME2_M, GF_BGL_SCROLL_X_SET, 136 );
				wk->sub_seq = 2;
			}else{
				GF_BGL_ScrollReq( wk->bgl, GF_BGL_FRAME2_M, GF_BGL_SCROLL_X_INC, 64 );
			}
		}
		break;
	case 2:
		GF_BGL_BmpWinOnVReq( &wk->def_win[WIN_LV] );
		GF_BGL_BmpWinOnVReq( &wk->def_win[WIN_MOTIMONO] );
		GF_BGL_BmpWinOnVReq( &wk->def_win[WIN_ITEMNAME] );
		GF_BGL_BmpWinOnVReq( &wk->def_win[WIN_P5_KUWASIKU] );
		PokeStatus_TabArrowVanish( wk, 1 );
		return TRUE;
	}
	return FALSE;
}


static u8 WazaCursorMove( PST_WORK * wk, s8 mv )
{
	s8	pos = wk->cur_pos;

	while(1){
		pos += mv;
		if( pos < 0 ){
			pos = 4;
		}else if( pos == 5 ){
			pos = 0;
		}
		if( wk->pp.waza[pos] != 0 || pos == 4 ){
			break;
		}
	}
	if( pos != wk->cur_pos ){
		wk->cur_pos = pos;
		return TRUE;
	}
	return FALSE;	// ԂAɂ͂Ȃ
}

static void WazaInfoChange( PST_WORK * wk )
{
	PokeStatus_WazaCursorMove( wk );

	if( wk->cur_pos == 4 ){
		if( wk->dat->waza != 0 ){
			PokeStatus_WazaInfoPut( wk );
			PokeStatus_KindIconChange( wk, wk->dat->waza );
			CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_KIND], 1 );
		}else{
			PokeStatus_WazaInfoOff( wk );
			CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_KIND], 0 );
		}
	}else{
		PokeStatus_WazaInfoPut( wk );
		PokeStatus_KindIconChange( wk, wk->pp.waza[wk->cur_pos] );
		CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_KIND], 1 );
	}
}



static void WazaChangeMain( PST_WORK * wk )
{
	void * pp;
	u16	tmp_waza;
	u8	tmp_wpp;
	u8	tmp_mpp;

	pp = PokeStatusPokeParamGet( wk );

	if( wk->dat->ppt == PST_PP_TYPE_POKEPASO ){
		PokePasoParaWazaReplace( pp, wk->cur_pos, wk->sub_cur );
	}else{
		PokeParaWazaReplace( pp, wk->cur_pos, wk->sub_cur );
	}

	tmp_waza = wk->pp.waza[wk->cur_pos];
	wk->pp.waza[wk->cur_pos] = wk->pp.waza[wk->sub_cur];
	wk->pp.waza[wk->sub_cur] = tmp_waza;

	tmp_wpp = wk->pp.wpp[wk->cur_pos];
	wk->pp.wpp[wk->cur_pos] = wk->pp.wpp[wk->sub_cur];
	wk->pp.wpp[wk->sub_cur] = tmp_wpp;

	tmp_mpp = wk->pp.mpp[wk->cur_pos];
	wk->pp.mpp[wk->cur_pos] = wk->pp.mpp[wk->sub_cur];
	wk->pp.mpp[wk->sub_cur] = tmp_mpp;
}


static void WazaInfoPut( PST_WORK * wk )
{
//	PokeStatus_TabArrowVanish( wk, 0 );
	GF_BGL_BmpWinOffVReq( &wk->def_win[WIN_LV] );
	GF_BGL_BmpWinOffVReq( &wk->def_win[WIN_MOTIMONO] );
	GF_BGL_BmpWinOffVReq( &wk->def_win[WIN_ITEMNAME] );

	GF_BGL_ScrollReq( wk->bgl, GF_BGL_FRAME2_M, GF_BGL_SCROLL_X_SET, 0 );

	PokeStatus_SkillCancelPut( wk );
	WazaInfoChange( wk );

	CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_POKE1], 1 );
	CATS_ObjectEnable( wk->awp[PST_OBJ_POKEICON], 1 );
	if( wk->pp.type1 != wk->pp.type2 ){
		CATS_ObjectEnable( wk->awp[PST_OBJ_TYPE_POKE2], 1 );
	}
	CLACT_SetDrawFlag( wk->awp[PST_OBJ_WAZACURSOR1], 1 );
}


//--------------------------------------------------------------------------------------------
/**
 * ԈُACR̃A[JCuCfbNX擾iLj
 *
 * @param	none
 *
 * @return	A[JCuCfbNX
 */
//--------------------------------------------------------------------------------------------
u32 BadStatusIconCharArcGet(void)
{
	return NARC_pst_gra_p_st_ijou_NCGR;
}

//--------------------------------------------------------------------------------------------
/**
 * ԈُACR̃A[JCuCfbNX擾ipbgj
 *
 * @param	none
 *
 * @return	A[JCuCfbNX
 */
//--------------------------------------------------------------------------------------------
u32 BadStatusIconPlttArcGet(void)
{
	return NARC_pst_gra_p_st_ijou_NCLR;
}

//--------------------------------------------------------------------------------------------
/**
 * ԈُACR̃A[JCuCfbNX擾iZj
 *
 * @param	none
 *
 * @return	A[JCuCfbNX
 */
//--------------------------------------------------------------------------------------------
u32 BadStatusIconCellArcGet(void)
{
	return NARC_pst_gra_p_st_ijou_NCER;
}

//--------------------------------------------------------------------------------------------
/**
 * ԈُACR̃A[JCuCfbNX擾iZAjj
 *
 * @param	none
 *
 * @return	A[JCuCfbNX
 */
//--------------------------------------------------------------------------------------------
u32 BadStatusIconCanmArcGet(void)
{
	return NARC_pst_gra_p_st_ijou_NANR;
}

//--------------------------------------------------------------------------------------------
/**
 * ԈُACR̃Ajԍ擾
 *
 * @param	pp		|Pf[^
 *
 * @return	Ajԍ
 */
//--------------------------------------------------------------------------------------------
u32 BadStatusIconAnmGet( POKEMON_PARAM * pp )
{
	u32	prm = PokeParaGet( pp, ID_PARA_condition, NULL );

	// 
	if( ( prm & (CONDITION_DOKU|CONDITION_DOKUDOKU) ) != 0 ){
		return ST_ICON_DOKU;
	// 
	}else if( ( prm & CONDITION_NEMURI ) != 0 ){
		return ST_ICON_NEMURI;
	// Ώ
	}else if( ( prm & CONDITION_YAKEDO ) != 0 ){
		return ST_ICON_YAKEDO;
	// X
	}else if( ( prm & CONDITION_KOORI ) != 0 ){
		return ST_ICON_KOORI;
	// 
	}else if( ( prm & CONDITION_MAHI ) != 0 ){
		return ST_ICON_MAHI;
	// m
	}else if( PokeParaGet( pp, ID_PARA_hp, NULL ) == 0 ){
		return ST_ICON_HINSI;
	}
	// Ȃ
	return ST_ICON_NONE;
}

//--------------------------------------------------------------------------------------------
/**
 * ԈُACR̃Ajԍ擾iXe[^Xʐpj
 *
 * @param	wk		[N
 *
 * @return	Ajԍ
 */
//--------------------------------------------------------------------------------------------
u32 PokeStatus_BadStatusIconAnmGet( PST_WORK * wk )
{
	// 
	if( ( wk->pp.st & (CONDITION_DOKU|CONDITION_DOKUDOKU) ) != 0 ){
		return ST_ICON_DOKU;
	// 
	}else if( ( wk->pp.st & CONDITION_NEMURI ) != 0 ){
		return ST_ICON_NEMURI;
	// Ώ
	}else if( ( wk->pp.st & CONDITION_YAKEDO ) != 0 ){
		return ST_ICON_YAKEDO;
	// X
	}else if( ( wk->pp.st & CONDITION_KOORI ) != 0 ){
		return ST_ICON_KOORI;
	// 
	}else if( ( wk->pp.st & CONDITION_MAHI ) != 0 ){
		return ST_ICON_MAHI;
	// m
	}else if( wk->pp.hp == 0 ){
		return ST_ICON_HINSI;
	}
	// Ȃ
	return ST_ICON_NONE;
}
