#include <mbstring.h>

#pragma hdrstop

#include "token.h"

#define	TOKEN_BLOCK_SIZE		(256)

static char			*st_buf, *st_end_p;
static ST_TOKEN		*st_a_token;
static int			st_size;

static char			*st_cur_p;
static char			*st_beg_line_p;
static int			st_set_line;
static AnsiString	st_line;

static int			st_start_pos;
static int			st_cur_pos;
static int			st_analized_pos;

static int			st_skip_space;

static ST_TOKEN *analyze_token(void);
static int skip_space(void);
static bool next_is_cr(void);
static ST_TOKEN *add_token_ident(void);
static ST_TOKEN *add_token(int token);
static AnsiString get_line(void);
static ST_TOKEN *get_token_area(void);
static int token_pos_inc(int pos);
static int token_pos_dec(int pos);

void token_init(void)
{
	ST_TOKEN		*t;

	st_size = TOKEN_BLOCK_SIZE;
	st_a_token = new ST_TOKEN[st_size];
}

void token_start(AnsiString path)
{
	TFileStream		*pf;

	pf = new TFileStream(path, fmOpenRead);
	st_buf = new char[pf->Size];
	pf->Read(st_buf, pf->Size);
	st_end_p = st_buf + (int)pf->Size;
	delete pf;

	st_cur_p = st_buf;
	st_beg_line_p = st_cur_p;
	st_set_line = 0;

	st_cur_pos = 1;
	st_analized_pos = 1;
	st_start_pos = 0;
	st_a_token[st_start_pos].token = TOKEN_OTHER;
}

void token_end(void)
{
	delete[] st_buf;
}

void token_finish(void)
{
	delete[] st_a_token;
}

int bookmark_token(void)
{
	int		pos;

	if (st_start_pos < st_cur_pos) {
		pos = st_cur_pos - st_start_pos;
	} else {
		pos = st_size - st_start_pos + st_cur_pos;
	}

	return pos;
}

void go_bookmark_token(int pos)
{
	int		cur;

	cur = st_start_pos + pos;
	if (st_size <= cur) {
		cur -= st_size;
	}
	st_cur_pos = cur;
}

bool token_is_end(void)
{
	int		pos;

	pos = token_pos_dec(st_cur_pos);
	return st_a_token[pos].token == TOKEN_EOF;
}

ST_TOKEN *token_prev(ST_TOKEN *t)
{
	int		pos = t - st_a_token;

	return &st_a_token[token_pos_dec(pos)];
}

void token_set_pos_to_start(void)
{
	int		cur;

	cur = st_start_pos + 1;
	if (cur == st_size) {
		cur = 0;
	}
	st_cur_pos = cur;
}

void token_restart(void)
{
	if (st_cur_pos != 0) {
		st_start_pos = st_cur_pos - 1;
	} else {
		st_start_pos = st_size - 1;
	}
}

ST_TOKEN *get_next_token(void)
{
	ST_TOKEN		*t;

	if (st_cur_pos != st_analized_pos) {
		t = &st_a_token[st_cur_pos];
		st_cur_pos = token_pos_inc(st_cur_pos);
	} else if (token_is_end()) {
		t = &st_a_token[token_pos_dec(st_cur_pos)];
	} else {
		t = analyze_token();
		st_cur_pos = token_pos_inc(st_cur_pos);
	}
	return t;
}

static ST_TOKEN *analyze_token(void)
{
	ST_TOKEN	*t;
	int			skip;

	st_skip_space = skip_space();
	if (st_end_p <= st_cur_p) {
		t = add_token(TOKEN_EOF);
	} else if (__iscsymf(*st_cur_p)) {
		t = add_token_ident();
	} else if (next_is_cr()) {
		t = add_token(TOKEN_NEW_LINE);
	} else {
		switch (*st_cur_p++) {
		case '#':
			t = add_token(TOKEN_SHARP);
			break;
		case '*':
			if (st_cur_p < st_end_p && *st_cur_p == '/') {
				t = add_token(TOKEN_CLOSE_COMMENT);
				st_cur_p++;
			} else {
				t = add_token(TOKEN_OTHER);
			}
			break;
		case '(':
			t = add_token(TOKEN_OPEN_PAR);
			break;
		case ')':
			t = add_token(TOKEN_CLOSE_PAR);
			break;
		case '{':
			t = add_token(TOKEN_OPEN_PAR2);
			break;
		case '}':
			t = add_token(TOKEN_CLOSE_PAR2);
			break;
		case '[':
			t = add_token(TOKEN_OPEN_PAR3);
			break;
		case ']':
			t = add_token(TOKEN_CLOSE_PAR3);
			break;
		case ';':
			t = add_token(TOKEN_SEMICOLON);
			break;
		case ':':
			t = add_token(TOKEN_COLON);
			break;
		case ',':
			t = add_token(TOKEN_COMMA);
			break;
		case '/':
			if (st_cur_p < st_end_p) {
				if (*st_cur_p == '/') {
					t = add_token(TOKEN_DBL_SLASH);
					st_cur_p++;
				} else if (*st_cur_p == '*') {
					t = add_token(TOKEN_OPEN_COMMENT);
					st_cur_p++;
				} else {
					t = add_token(TOKEN_OTHER);
				}
			} else {
				t = add_token(TOKEN_OTHER);
			}
			break;
		case '=':
			// !=A==AEEEȂǂ邯ǁA܁[AۂǕςȏȂsł傤;;B
			t = add_token(TOKEN_EQUAL);
			break;
		case '"':
			t = add_token(TOKEN_DQUART);
			break;
		case '.':
			t = add_token(TOKEN_DOT);
			break;
		default:
			t = add_token(TOKEN_OTHER);
		}
	}

	return t;
}

static int skip_space(void)
{
	int		skip;
	int		is_skip;

	is_skip = 0;
	while (st_cur_p < st_end_p) {
		if (_MBC_LEAD == _mbbtype((unsigned char)*st_cur_p, 0)) {
			st_cur_p += 2;
		} else if (*st_cur_p == ' ' || *st_cur_p == '\t') {
			st_cur_p++;
		} else if (*st_cur_p == '\\') {
			// \gGXP[vn͋󔒈B
			st_cur_p++;
			if (st_cur_p < st_end_p) {
				if (!next_is_cr()) {
					st_cur_p++;
				}
			}
		} else {
			break;
		}
		is_skip = 1;
	}

	return is_skip;
}

static bool next_is_cr(void)
{
	bool	ret = false;
	char	*p = st_cur_p;
	int		skip = 0;

	if (*p == '\r') {
		if (p + 1 < st_end_p && p[1] == '\n') {
			skip = 2;
		} else {
			skip = 1;
		}
		ret = true;
	} else if (*p == '\n') {
		skip = 1;
		ret = true;
	}

	if (ret) {
		st_cur_p += skip;
		st_beg_line_p = st_cur_p;
		st_set_line = 0;
	}
	return ret;
}

static ST_TOKEN *add_token_ident(void)
{
	ST_TOKEN	*t = get_token_area();

	t->token = TOKEN_IDENT;
	t->skip_space = st_skip_space;
	t->line = get_line();
	t->start_p = st_cur_p;

	st_cur_p++;
	while (st_cur_p < st_end_p && __iscsym(*st_cur_p)) {
		st_cur_p++;
	}
	t->len = st_cur_p - t->start_p;
	st_analized_pos = token_pos_inc(st_analized_pos);

	return t;
}

static ST_TOKEN *add_token(int token)
{
	ST_TOKEN	*t = get_token_area();

	t->token = token;
	t->skip_space = st_skip_space;
	st_analized_pos = token_pos_inc(st_analized_pos);

	return t;
}

static AnsiString get_line(void)
{
	char			*p;
	int				len;

	if (!st_set_line) {
		for (p = st_beg_line_p; p < st_end_p; p++) {
			if (*p == '\r' || *p == '\n') {
				break;
			}
		}
		len = p - st_beg_line_p;
		st_line = AnsiString(st_beg_line_p, len);
		st_set_line = 1;
	}

	return st_line;
}

static ST_TOKEN *get_token_area(void)
{
	if (st_cur_pos == st_start_pos) {
		ST_TOKEN	*p;

		p = new ST_TOKEN[st_size + TOKEN_BLOCK_SIZE];
		for (int i = 0; i < st_cur_pos; i++) {
			p[i] = st_a_token[i];
		}
		for (int i = st_size - 1; i > st_start_pos; i--) {
			p[i + TOKEN_BLOCK_SIZE] = st_a_token[i];
		}
		delete[] st_a_token;
		st_a_token = p;
		st_start_pos += TOKEN_BLOCK_SIZE;
		st_size += TOKEN_BLOCK_SIZE;
	}
	return &st_a_token[st_cur_pos];
}

static int token_pos_inc(int pos)
{
	pos++;
	if (pos == st_size) {
		pos = 0;
	}

	return pos;
}

static int token_pos_dec(int pos)
{
	if (pos != 0) {
		pos--;
	} else {
		pos = st_size - 1;
	}

	return pos;
}
