﻿use strict;
use XML::DOM;
use utf8;

#***********************************************************
# パッケージoutputer
#***********************************************************
package outputer;

my $HEAD_SIZE = 0x0008;		# メッセージ１つに対するヘッダサイズ。

#===========================================================
# オブジェクト生成。
#===========================================================
sub new {
	my ($class, $fname, $ext) = @_;

	if ($ext eq "") {
		$ext = "normal";
	}
	bless {
		fname => $fname,
		ext => $ext,
		msg_id_check => {},
		msg_id_list => [],
		msg_len => "",
		msg_body => "",
		msg_count => 0,
		res => ""
	}, $class;
}

#===========================================================
# メッセージ追加。
#===========================================================
sub add {
	my ($self, $msg_id, $res) = @_;

	if (defined($self->{msg_id_check}{$msg_id})) {
		# ここでエラーを出力する。データとしてはEOMのみにしておく。
		# 日本語では、このケースは発生していないので好きにすればよい。
		# 海外では今まで;をEOMとしてきた。日本語では常にEOMを付けているので、
		# 今回からは、;は不要かもしれない。
	}

	# メッセージを記録。
	$self->{msg_id_check}{$msg_id} = 1;
	push @{$self->{msg_id_list}}, $msg_id;
	$self->{msg_len} .= $res->{length} . ",";
	$self->{msg_body} .= $res->{data};
	$self->{msg_count}++;
}

#===============================================================
# ファイル出力実行。
#===============================================================
sub exec {
	my ($self, $header_dir, $res_dir) = @_;

	if ($self->{ext} eq "normal") {
		$self->output_header($header_dir);
	}
	$self->output_res($res_dir);
}

#===============================================================
# 保存結果データを取得。
#===============================================================
sub getResult {
	my ($self) = @_;

	if ($self->{ext} ne "result") {
		die "結果を保存していない。\n";
	}
	return $self->{res};
}

#===============================================================
# メッセージIDのリストを取得。
#===============================================================
sub getMsgIdList {
	my ($self) = @_;

	return $self->{msg_id_list};
}

#---------------------------------------------------------------
# private
#---------------------------------------------------------------

#===============================================================
# ヘッダファイル出力。
#===============================================================
sub output_header {
	my ($self, $dir) = @_;
	my ($h_path, $h_name, $h_define, $str, $now_str, $out_str);
	my $cnt;

	$str = $self->{fname};
	$str =~ s/\.gmm$/.h/;
	$str =~ /([^\\\/]+)$/;
	$h_name = "msg_" . $1;
	$h_path = "$dir/$h_name";
	$str = $h_name;
	$str =~ s/\./_/;
	$h_define = "__" . uc($str) . "__";

	if (open(HEADER, "$h_path")) {
		my @line;

		binmode(HEADER, "encoding(shiftjis)");
		@line = <HEADER>;
		close HEADER;
		$now_str = join '', @line;
	}

	$out_str .= "//==============================================================================\n";
	$out_str .= "/**\n";
	$out_str .= " * \@file		$h_name\n";
	$out_str .= " * \@brief		メッセージID参照用ヘッダファイル\n";
	$out_str .= " *\n";
	$out_str .= " * このファイルはコンバータにより自動生成されています\n";
	$out_str .= " */\n";
	$out_str .= "//==============================================================================\n";
	$out_str .= "#ifndef $h_define\n";
	$out_str .= "#define $h_define\n\n";

	$cnt = 0;
	for my $i (@{$self->{msg_id_list}}) {
		$out_str .= "#define\t$i\t\t($cnt)\n";
		$cnt++;
	}

	$out_str .= "\n#endif\n";

	# 現在の内容を確認して、一致していれば上書きしない。
	# これによって、ヘッダ変更による再コンパイルの発生を抑制する。
	if ($out_str ne $now_str) {
		if (!open(HEADER, ">$h_path")) {
			die "error: ファイルが開けません\n";
		}
		binmode(HEADER, "encoding(shiftjis)");
		print HEADER $out_str;
		close HEADER;
	}
}

#===============================================================
# データ出力。
#===============================================================
sub output_res {
	my ($self, $dir) = @_;
	my ($r_path, $enc_path, $str, $dat, @len, $val, $ofs);

	$str = $self->{fname};
	$str =~ s/\.gmm$/.dat/;
	$str =~ /([^\\\/]+)$/;
	$r_path = "$dir/$1";
	# 乱数の種に使われる。フォルダを固定しないと、実行場所によって
	# 生成物の内容が変わってしまうので、日本語版に固定した。
	$enc_path = "./data/" . $1;

	# 先頭２バイトにメッセージ件数。
	$dat .= pack('S', $self->{msg_count});

	# 次の２バイトに暗号用乱数。
	$val = $self->calc_crc($enc_path);
	$dat .= pack('S', $val);

	# 次にオフセットテーブル（１件４バイト）。
	@len = split(/,/, $self->{msg_len});
	$ofs = 0;
	for (my $i = 0; $i < $self->{msg_count}; $i++) {
		# データファイル先頭からのオフセット
		$val = 4 + ($self->{msg_count}  * $HEAD_SIZE) + $ofs;
		$dat .= pack('L', $val);
		$dat .= pack('L', $len[$i]);

		$ofs += $len[$i] * 2;
	}

	# 最後にメッセージデータ
	$dat .= $self->{msg_body};

	if ($self->{ext} ne "result") {
		if (!open(RES, ">$r_path")) {
			die "error: ファイルが開けません\n";
		}

		binmode(RES);
		print RES $dat;
		close(RES);

		system("msgenc $r_path\n");
	} else {
		$self->{res} = $dat;
	}
}

#===============================================================
# ファイル名文字列からCRC計算。
#===============================================================
sub calc_crc {
	my ($self, $fname) = @_;
	my ($crc, $p, $d, $len, $count);

	$len = length($fname);
	$p = 0;
	$crc = 0;
	while ($len--) {
		$d = ord(substr($fname, $p++, 1));
		$count = 8;
		while ($count--) {
			$crc ^= 0x8003 if ($crc <<= 1) & 0x10000;
			$crc ^= 1 if ($d <<= 1) & 0x100;
		}
	}
	return $crc & 0xffff;
}

1;
