﻿use strict;
use XML::DOM;
use utf8;

#***********************************************************
# パッケージgmm
#***********************************************************
package gmm;

our %STR_TBL = (
	'japanese' => {
		LANG_JAPAN => '日本語',
		LANG_ENGLISH => 'English',
		ENG => 'English',
		WIN_CNT => 'ウィンドウコンテキスト名',
		CMT_ENG => 'コメント（英語）'
	},
	'english' => {
		LANG_JAPAN => "Japanese",
		LANG_ENGLISH => 'English',
		ENG => 'English',
		WIN_CNT => 'WindowContextName',
		CMT_ENG => 'Comment(English)'
	}
);

my $parser = new XML::DOM::Parser;

#===========================================================
# オブジェクト生成。
#===========================================================
sub new {
	my ($class, $fname) = @_;
	my ($doc, $body, @row, $lang, $tmp);

	$doc = $parser->parsefile($fname);
	($tmp) = $doc->getElementsByTagName('gmml', 0);
	($body) = $tmp->getElementsByTagName('body', 0);
	@row = $body->getElementsByTagName('row', 0);
	$lang = $body->getAttribute('language');

	bless {
		doc => $doc,
		row => \@row,
		lang => $lang
	}, $class;
}

#===========================================================
# ファイル出力。
#===========================================================
sub print {
	my ($self, $fname) = @_;

	if (!open(OUT, ">$fname")) {
		die "error: open";
	}
	binmode(OUT, "utf8");
	$self->{doc}->printToFileHandle(\*OUT);
	close(OUT);
}

#===========================================================
# 新規イテレータ取得。
#===========================================================
sub newIter {
	my ($self, $lang) = @_;

	return gmm::iterator->new($self, $lang);
}

#***********************************************************
# パッケージgmm::iterator
#***********************************************************
package gmm::iterator;

#===========================================================
# オブジェクト生成。
#===========================================================
sub new {
	my ($class, $gmm, $lang) = @_;
	my $over = 0;

	if (@{$gmm->{row}} == 0) {
		$over = 1;
	}
	return bless {
		gmm => $gmm,
		id => 0,
		over => $over,
		cache => {},
		lang => $lang
	}, $class;
}

#===========================================================
# 次に進める。
#===========================================================
sub next {
	my ($self) = @_;

	$self->{id}++;
	$self->{cache} = {};

	if (@{$self->{gmm}{row}} <= $self->{id}) {
		$self->{over} = 1;
	}
}

#===========================================================
# 範囲オーバーしていないか？
#===========================================================
sub over {
	my ($self) = @_;

	return $self->{over};
}

#===========================================================
# テキストを取得。
#===========================================================
sub getText {
	my ($self, $lang) = @_;
	my $msg_node;

	if (!defined($lang)) {
		$lang = $self->{lang};
	}
	$msg_node = $self->getMsgNode($lang);
	if ($msg_node == 0) {
		return "";
	}
	return $msg_node->getData();
}

#===========================================================
# メッセージIDを取得。
#===========================================================
sub getMsgId {
	my ($self) = @_;
	my $row = $self->{gmm}{row}[$self->{id}];

	return $row->getAttribute('id');
}

#===========================================================
# 行数を取得。
#===========================================================
sub getLineNum {
	my ($self) = @_;
}

#===========================================================
# テキストを設定。
#===========================================================
sub setText {
	my ($self, $text, $lang) = @_;
	my $msg_node;

	if (!defined($lang)) {
		$lang = $self->{lang};
	}
	$msg_node = $self->getMsgNode($lang);
	if ($msg_node == 0) {
		# 基本的にコンバータに使用するので、追加の必要はない。
		# ただし、日本語のノードがない場合には対処する必要がある。
		# 本当にあり得るかはコンバータをチェックすること。
		die "error: no msg_node";
	}
	$msg_node->setData($text);
}

#===========================================================
# 元ファイル名を取得。
#===========================================================
sub getOrgFileName {
	my ($self) = @_;

	return $self->getAttr('org_file');
}

#-----------------------------------------------------------
# private
#-----------------------------------------------------------

#===========================================================
# テキストノードを取得。
#===========================================================
sub getTextNodes {
	my ($self, $parent) = @_;
	my (@node, @text_node);

	@node = $parent->getChildNodes();
	for my $i (@node) {
		if ($i->getNodeType() == ::TEXT_NODE) {
			push @text_node, $i;
		}
	}
	return @text_node;
}

#===========================================================
# メッセージノードを取得、キャッシュする。
#===========================================================
sub getMsgNode {
	my ($self, $lang) = @_;
	my $gmm = $self->{gmm};
	my $cache = $self->{cache};

	if (!defined($cache->{text}{$lang})) {
		my $lang_node = $self->getLangNode($lang);

		if ($lang_node == 0) {
			$cache->{text}{$lang} = 0;
		} else {
			($cache->{text}{$lang}) = $self->getTextNodes($lang_node);
			if (!defined($cache->{text}{$lang})) {
				$cache->{text}{$lang} = $lang_node->addText("");
			}
		}
	}
	return $cache->{text}{$lang};
}

#===========================================================
# 言語ノードを取得、キャッシュする。
#===========================================================
sub getLangNode {
	my ($self, $lang) = @_;
	my $gmm = $self->{gmm};
	my $cache = $self->{cache};

	if (!defined($cache->{lang}{$lang})) {
		my $row = $gmm->{row}[$self->{id}];
		my @lang;

		@lang = $row->getElementsByTagName('language', 0);
		for my $i (@lang) {
			if ($i->getAttribute('name') ne $gmm::STR_TBL{$gmm->{lang}}{$lang}) {
				next;
			}
			$cache->{lang}{$lang} = $i;
		}
		if (!defined($cache->{lang}{$lang})) {
			$cache->{lang}{$lang} = 0;
		}
	}
	return $cache->{lang}{$lang};
}

#===========================================================
# アトリビュートを取得、キャッシュする。
#===========================================================
sub getAttr {
	my ($self, $attr) = @_;
	my $text_node;

	$text_node = $self->getAttrNode($attr);
	if ($text_node == 0) {
		return "";
	} else {
		return $text_node->getData();
	}
}

#===========================================================
# アトリビュートを設定、キャッシュする。
#===========================================================
sub setAttr {
	my ($self, $attr, $val) = @_;
	my $text_node;

	$text_node = $self->getAttrNode($attr);
	if ($text_node == 0) {
		# 無いので作る。
		my $attr_node = $self->{gmm}{doc}->createElement('attribute');
		my $row = $self->{gmm}{row}[$self->{id}];

		$attr_node->setAttribute('name', $attr);
		$text_node = $self->{gmm}{doc}->createTextNode('');
		$attr_node->appendChild($text_node);
		$row->addText("\t");
		$row->appendChild($attr_node);
		$row->addText("\n\t");
		$self->{cache}{attr}{$attr} = $text_node;
	}
	$text_node->setData($val);
}

#===========================================================
# アトリビュートのテキストノードを取得する。
#===========================================================
sub getAttrNode {
	my ($self, $attr) = @_;
	my $gmm = $self->{gmm};
	my $cache = $self->{cache};

	if (!defined($cache->{attr}{$attr})) {
		my $row = $gmm->{row}[$self->{id}];
		my @attr;

		@attr = $row->getElementsByTagName('attribute', 0);
		for my $i (@attr) {
			if ($i->getAttribute('name') ne $attr) {
				next;
			}
			($cache->{attr}{$attr}) = $self->getTextNodes($i);
		}
		if (!defined($cache->{attr}{$attr})) {
			$cache->{attr}{$attr} = 0;
		}
	}
	return $cache->{attr}{$attr};
}

1;
