/*
 *               Copyright (C) 2005, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */


#ifndef __EC_H__
#define __EC_H__

#include "vng.h"

#ifdef __cplusplus
extern "C" {
#endif

#include "estypes.h"

#if !defined(_GBA) && !defined(_SC)
#pragma pack(push, 4)
#endif



/* ECError values */
#define EC_ERROR_OK                      0  /* No error */
#define EC_ERROR_FAIL                   -1  /* Generic error */
#define EC_ERROR_NOT_SUPPORTED          -2  /* Feature not implemented */
#define EC_ERROR_INSUFICIENT_RESOURCE   -3
#define EC_ERROR_INVALID                -4
#define EC_ERROR_NOMEM                  -5
#define EC_ERROR_NOT_FOUND              -6
#define EC_ERROR_NOT_BUSY               -7  /* no active async operation */
#define EC_ERROR_BUSY                   -8
#define EC_ERROR_NOT_DONE               -9
#define EC_ERROR_NO_PROXY              -10
#define EC_ERROR_NOT_CACHED            -11
#define EC_ERROR_NOT_OPEN              -12
#define EC_ERROR_NET_NA                -13  /* Internet access not available */
#define EC_ERROR_ECS_NA                -14  /* ECommerce sevice not available */
#define EC_ERROR_ECS_RESP              -15  /* ECS reports a problem */
#define EC_ERROR_ECP                   -16  /* EC proxy reports an error */
#define EC_ERROR_ECARD                 -17  /* Invalid eCard */
#define EC_ERROR_OVERFLOW              -18  /* Output too big for buf provided */



/* ECTitleKind values */
#define EC_TITLE_KIND_GAME              1
#define EC_TITLE_KIND_MANUAL            2
#define EC_TITLE_KIND_GAME_AND_MANUAL   3

/* ECPricingKind values
 *      - only EC_PK_SUBSCRIPTION is applicable NetC
 */
#define EC_PK_SUBSCRIPTION      1
#define EC_PK_PURCHASE          2

/* ECPricingCategory values
 *      - only EC_PC_SUBSCRIPTION is applicable NetC
 */
#define EC_PC_SUBSCRIPTION      1
#define EC_PC_PERMANENT         2
#define EC_PC_RENTAL            3
#define EC_PC_BONUS             4
#define EC_PC_TRIAL             5

/* ECLicenseKind values
 *      - only EC_SR is applicable to NetC
 */
#define EC_SR                   1 /* Subscription */
#define EC_PR                   2 /* Permanent */
#define EC_LR                   3 /* limited play */
#define EC_TR                   4 /* time limited play */

/* ECTimeUnit values */
#define EC_DAY      1
#define EC_MONTH    2

/* miscellaneous defines */
#define EC_TITLE_NAME_BUF_SIZE         64
#define EC_TITLE_DESC_BUF_SIZE        256
#define EC_TITLE_CAT_BUF_SIZE          32
#define EC_MAX_CONTENTS_PER_TITLE      10
#define EC_MAX_TITLE_PRICINGS          10
#define EC_MAX_RATINGS                  5
#define EC_MAX_CONSUMPTIONS           (VN_MAX_MSG_LEN/sizeof(ECConsumption))

#define EC_MAX_URL_LEN                  255
#define EC_URL_BUF_SIZE                (EC_MAX_URL_LEN + 1)
#define EC_MAX_CRL_VERSION_LEN          47
#define EC_CRL_VERSION_BUF_SIZE        (EC_MAX_CRL_VERSION_LEN +1)
#define EC_MAX_CURRENCY_LEN             31
#define EC_CURRENCY_BUF_SIZE           (EC_MAX_CURRENCY_LEN + 1)

#define EC_MAX_SUB_CHANNEL_NAME_LEN     31  
#define EC_SUB_CHANNEL_NAME_BUF_SIZE   (EC_MAX_SUB_CHANNEL_NAME_LEN + 1)
#define EC_MAX_SUB_CHANNEL_DESC_LEN     31
#define EC_SUB_CHANNEL_DESC_BUF_SIZE   (EC_MAX_SUB_CHANNEL_DESC_LEN + 1)

#define EC_MAX_RATING_NAME_LEN          15
#define EC_RATING_NAME_BUF_SIZE        (EC_MAX_RATING_NAME_LEN + 1)
#define EC_MAX_RATING_LEN               15
#define EC_RATING_BUF_SIZE             (EC_MAX_RATING_LEN + 1)

typedef s32 ECTitleKind;
typedef s32 ECPricingKind;
typedef s32 ECPricingCategory;
typedef s32 ECLicenseKind;
typedef s64 ECTimeStamp;         /* msec since 1970-01-01 00:00:00 UTC */
typedef s32 ECError;
typedef s32 ECTimeUnit;




/*  ECCacheStatus values refer to the cache status of files
 *  cached on the USB attached PC.
 *
 *  An ECCacheStatus values haave the following meaning:
 *
 *      - a negative value for cachedSize or totalSize is an ECError code
 *
 *      - cachedSize == EC_ERROR_NOT_CACHED means the file is not cached
 *        and download of the file is not in progress.
 *
 *      - totalSize is the size the file will be when fully downloaded or
 *        EC_ERROR_INVALID if the totalSize is not yet known.
 *
 *      - cachedSize >= 0 and cachedSize != totalSize indicates the file
 *        is currently being downloaded.  The number of bytes that have
 *        been downloaded is the cachedSize.
 *
 *      - cachedSize >= 0 and cachedSize == totalSize indicates the file
 *        is in the cache and available to be opened.
 *
 *      - cachedSize < 0 and cachedSize != EC_ERROR_NOT_CACHED indicates
 *        the ECCacheStatus is not valid.  This will be the state of an
 *        ECCacheStatus if EC_GetCachedContentStatus fails before the
 *        cache status has been obtained.
 *
 */

typedef struct {
    s32     cachedSize;
    s32     totalSize;
} ECCacheStatus;


typedef struct {
    s64             size;
    ESTitleId       id;
    char            name[EC_TITLE_NAME_BUF_SIZE];

} ECListTitlesInfo;


/* The only valid NetC currency value is "EUNITS"
 *
 */
typedef struct {
    s32        amount;
    char       currency[EC_CURRENCY_BUF_SIZE];

} ECPrice;

typedef struct
{
    s32                itemId;
    ECLicenseKind      licenseKind;
    s32                limits;
    ECPrice            price;
    ECPricingCategory  pricingCategory;

} ECPricing;


typedef struct {
    ESTitleId       titleId;
    ECTitleKind     titleKind;
    char            titleName   [EC_TITLE_NAME_BUF_SIZE];
    char            description [EC_TITLE_DESC_BUF_SIZE];
    char            category    [EC_TITLE_CAT_BUF_SIZE];
    u32             nContentIds;
    ESContentId     contentIds  [EC_MAX_CONTENTS_PER_TITLE];
    ECCacheStatus   cacheStatus [EC_MAX_CONTENTS_PER_TITLE];
    ESContentId     tmdContentId;
    s64             titleSize;
    ECTimeStamp     version;

} ECTitleInfo;


typedef struct {
    char            name[EC_RATING_NAME_BUF_SIZE];
    char            rating[EC_RATING_BUF_SIZE];
} ECRating;


typedef struct {
    ECTitleInfo     info;
    u32             nRatings;
    ECRating        ratings[EC_MAX_RATINGS];
    u32             nPricings;
    ECPricing       pricings[EC_MAX_TITLE_PRICINGS];

} ECTitleDetails;



typedef struct {
    s32            itemId;
    s32            channelId;
    char           channelName [EC_SUB_CHANNEL_NAME_BUF_SIZE];
    char           channelDescription [EC_SUB_CHANNEL_DESC_BUF_SIZE];
    s32            subscriptionLength;
    ECTimeUnit     subscriptionTimeUnit;
    ECPrice        price;
    s32            maxCheckouts; /* -1 means no max specified */

} ECSubscriptionPricing;



typedef struct {
    ECTimeStamp   timestamp;
    
} ECSigned;

typedef u8 ECSignature[20];

typedef struct {
    ECTitleInfo         secureKernel;
    ECTitleInfo         viewer;
    char                crlVersion[EC_CRL_VERSION_BUF_SIZE];
    ESContentId         crlContentId;
    ECSigned            signedInfo;
    ECSignature         signature;

} ECMeta;



typedef struct {
    ESTitleId       titleId;
    s32             minutes;

} ECConsumption;

      






/*
 *  Asynchronous vs. Synchronous EC APIs
 *
 *  The eCommerce library functions are either synchronous or asynchronous.
 *
 *  Synchronous functions perform an operation and do not return until the
 *  operation completes or returns due to an error.. Asynchronous functions
 *  start an operation and return without waiting for completion.
 *
 *  The return value of an asynchronous function can be
 *      1. an ECError,
 *      2. an integer value that is an ECError when negative and
 *         a call specific return value when positve.
 *
 *  A zero return value indicates successful completion and may convey other
 *  call specific information like "no data returned".
 *
 *  If an asynchronous function returns before the operation is complete,
 *  it will return EC_ERROR_NOT_DONE.
 *
 *  The caller is expected to poll for completion until the
 *  operation completes or is terminated by an error.
 *
 *  Only one eCommerce asynchronous operation can be active at a time.
 *
 *  Completion status of an active asynchronous operation is retrieved by
 *  calling EC_GetReturnVlaue().  The integer return value of
 *  EC_GetReturnVlaue() will have the same meaning as the the most recently
 *  called asynchronous function.
 *
 *  For example, if the function would return a positive count on success
 *  when called synchronously, EC_GetReturnVlaue() will return a positive
 *  count on successful completion of the asynchronous operation.
 *  EC_GetReturnVlaue() will return EC_ERROR_NOT_DONE if the operation
 *  has not finished.
 *
 *  EC_ERROR_BUSY will be returned by a call to start an asynchronous
 *  operation if the current one has not finished.  The existing
 *  operation will continue.  Use EC_CancelOperation() to terminate an
 *  on-going asynchronous operation before it has completed.
 *
 *  A call to EC_CancelOperation is not needed if EC_GetReturnValue
 *  returns other than EC_ERROR_NOT_DONE.
 *
 */


/*  EC_GetETicketSize
 *
 *  Returns the size of an ETicket.
 *
 */
s32 EC_GetETicketSize ();



/*  EC_GetReturnValue
 *
 *  Gets the return value or status of an on-going asynchronous operation.
 *
 *  Only one eCommerce asynchronous operation can be active at a time.
 *
 *  The current operation is complete if EC_GetReturnValue returns other
 *  than EC_ERROR_NOT_DONE.  EC_GetReturnValue will retrieve the return
 *  value of the last asynchronous operation until a new one is started.
 *
 *  EC_ERROR_BUSY will be returned by a call to start an asynchronous
 *  operation if the current one has not finished.  The existing
 *  operation will continue.  Use EC_CancelOperation() to terminate an
 *  on-going asynchronous operation before it has completed.
 *
 *  The return value of an asynchronous function can be
 *       1. an ECError value,
 *       2. an integer value that is an ECError when negative and
 *          a call specific value when positive.
 *
 *  A zero return value indicates successful completion of the
 *  asynchronous operation and may convey additional
 *  call specific information like no data returned.
 *
 *  If the current asynchronous operation has completed,
 *     the return value of the asynchronous operation is returned.
 *
 *  If the operation associated is still in progress,
 *      EC_ERROR_NOT_DONE is returned.
 *
 *  Error codes returned include:
 *
 *      EC_ERROR_NOT_BUSY  - No asynchronous operation is active.
 *
 */
ECError  EC_GetReturnValue ();



/*  EC_CancelOperation
 *
 *  Cancels the current asynchronous operation.
 *
 *  Only one eCommerce asynchronous operation can be active at a time.
 *
 *  EC_ERROR_BUSY will be returned by a call to start an asynchronous
 *  operation if the current one has not finished.  The existing
 *  operation will continue.  Use EC_CancelOperation() to terminate an
 *  on-going asynchronous operation before it has completed.
 *
 *  A call to EC_CancelOperation is not needed if EC_GetReturnValue
 *  returns other than EC_ERROR_NOT_DONE.
 *
 *  EC_CancelOperation indicates that no further actions associated
 *  with the current asynchronous operation are required and the
 *  return value is not needed.  It may not actually stop the
 *  the operation, but will allow another operation to be started.
 *
 *  After EC_CancelOperation(), calls to EC_GetReturnValue will
 *  return EC_ERROR_NOT_BUSY until another asynchronous operation
 *  is started.
 *
 *  Error codes returned include:
 *
 *      EC_ERROR_OK
 *
 */
ECError EC_CancelOperation ();



/*  NAME
 *
 *      EC_ListTitles
 *
 *  Fills in a list of data structures describing titles that match
 *  the specified title kind and pricing category.
 *
 *  The caller must provide a buffer at *titles and set *nTitles to
 *  the number of titles that can be returned in the buffer.
 *
 *  On completion the actual number of titles returned at *titles will
 *  be stored at *nTitles.
 *
 *  For NetC, the only valid value for pricingKind is EC_PK_SUBSCRIPTION.
 *
 *  Error codes returned include:
 *
 *      EC_ERROR_OK       - operation was successful or started successfully
 *      EC_ERROR_BUSY     - previous async operation has not finished
 *      EC_ERROR_INVALID  - an argument was not valid
 *      EC_ERROR_NOT_DONE - asynchronous operation is not complete.
 *
 *  If EC_ERROR_NOT_DONE is returned,the caller is expected to poll
 *  for completion by calling EC_GetReturnVlaue().
 *
 */
ECError  EC_ListTitles (ECTitleKind         titleKind,
                        ECPricingKind       pricingKind,
                        ECListTitlesInfo   *titles,
                        u32                *nTitles);
                                              
/*  NAME
 *
 *      EC_GetTitleDetails
 *
 *  Fills a data structure describing the title indicated by titleId.
 *
 *  On completion the title details will be returned at *details.
 *
 *  Error codes returned include:
 *
 *      EC_ERROR_OK       - operation was successful
 *      EC_ERROR_BUSY     - previous async operation has not finished
 *      EC_ERROR_INVALID  - an argument was not valid
 *      EC_ERROR_NOT_DONE - asynchronous operation is not complete.
 *
 *  If EC_ERROR_NOT_DONE is returned,the caller is expected to poll
 *  for completion by calling EC_GetReturnVlaue().
 *
 */
ECError  EC_GetTitleDetails (ESTitleId        titleId,
                             ECTitleDetails  *details);
                                              
                                              
/*  NAME
 *
 *      EC_GetECMeta
 *
 *  Fills in a data structure with information from the
 *  infrastructure server.  See the definition of ECMeta.
 *
 *  On completion the EC meta data will be stored at *meta.
 *
 *  Error codes returned include:
 *
 *      EC_ERROR_OK       - operation was successful
 *      EC_ERROR_BUSY     - previous async operation has not finished
 *      EC_ERROR_INVALID  - an argument was not valid
 *      EC_ERROR_NOT_DONE - asynchronous operation is not complete.
 *
 *  If EC_ERROR_NOT_DONE is returned,the caller is expected to poll
 *  for completion by calling EC_GetReturnVlaue().
 *  
 */
ECError  EC_GetECMeta (ECMeta        *meta);



/*  NAME
 *
 *      EC_ListSubscriptionPricings
 *
 *  Fills in a list of data structures describing subscription
 *  pricing options.
 *
 *  The caller must provide a buffer at *pricings and set *nPricings
 *  to the number of ECSubscriptionPricing data structures that can be
 *  returned in the buffer.
 *
 *  On completion the actual number of pricings returned at *pricings will
 *  be stored at *nPricings.
 *
 *  Error codes returned include:
 *
 *      EC_ERROR_OK       - operation was successful or started successfully
 *      EC_ERROR_BUSY     - previous async operation has not finished
 *      EC_ERROR_INVALID  - an argument was not valid
 *      EC_ERROR_NOT_DONE - asynchronous operation is not complete.
 *
 *  If EC_ERROR_NOT_DONE is returned,the caller is expected to poll
 *  for completion by calling EC_GetReturnVlaue().
 *
 */
ECError  EC_ListSubscriptionPricings (ECSubscriptionPricing  *pricings,
                                      u32                    *nPricings);



/*  NAME
 *
 *      EC_Subscribe
 *
 *  Requests that the subscription for the device be
 *  extended for the amount of time indicated using the
 *  ECard provided to pay for the subscription.
 *
 *  The time can be indicated by months or days.  The
 *  time units is indicated by subscriptionTimeUnit and
 *  the number of units is indicated by subscriptionLength.
 *
 *  The itemId, channelId, priceAmount, currency, subscriptionLength,
 *  and subscriptionTimeUnit should correspond to the values returned in
 *  an ECSubscriptionPricing returned by EC_ListSubscriptionPricings.
 *
 *  On completion returns the ETicket, certs, and certsSize.
 *
 *  The ETicket is returned at *ticket and has the length returned by
 *  EC_GetETicketSize().  The byte array at *tickets must be at
 *  least EC_GetETicketSize() bytes.
 *
 *  The certificate chain is returned at *certs.  The caller should initialize
 *  *certsSize to the buffer size at *certs.   On completion *certsSize
 *  will be set to the size of the certs returned.
 *
 *  Error codes returned include:
 *
 *      EC_ERROR_OK       - operation was successful
 *      EC_ERROR_BUSY     - previous async operation has not finished
 *      EC_ERROR_<TBD>    - TBD ECard eror codes
 *      EC_ERROR_INVALID  - an argument was not valid
 *      EC_ERROR_NOT_DONE - asynchronous operation is not complete.
 *
 *  If EC_ERROR_NOT_DONE is returned,the caller is expected to poll
 *  for completion by calling EC_GetReturnVlaue().
 *  
 */
ECError  EC_Subscribe  (s32                itemId,
                        s32                channelId,
                        s32                priceAmount,
                        char              *currency,
                        char              *eCard,
                        s32                subscriptionLength,
                        ECTimeUnit         subscriptionTimeUnit,
                        void              *ticket,
                        void              *certs,
                        u32               *certsSize);






/*  NAME
 *
 *      EC_UpdateStatus
 *
 *  Provides consumption information on played games.
 *
 *  The caller passes an array of consumption information consisting
 *  of a titleId and the number of minutes played for each played game.
 *
 *  The number of elements in the consumption array is specified
 *  in nConsumptions.
 *
 *  Error codes returned include:
 *
 *      EC_ERROR_OK       - operation was successful
 *      EC_ERROR_BUSY     - previous async operation has not finished
 *      EC_ERROR_INVALID  - an argument was not valid
 *      EC_ERROR_NOT_DONE - asynchronous operation is not complete.
 *
 *  If EC_ERROR_NOT_DONE is returned,the caller is expected to poll
 *  for completion by calling EC_GetReturnVlaue().
 *  
 */
ECError  EC_UpdateStatus (ECConsumption     *consumptions,
                          u32               nConsumptions);







/*  EC_OpenCachedContent
 *
 *  Opens a read only content file cached on a USB attached PC.
 *
 *  Only one cached content file can be opened at a time.
 *
 *  If the file is present on a PC attached via USB, it will be used
 *  as the source of subsequent EC_ReadCachedContent requests.
 *
 *  If the file is not present on the attached PC, an error is returned.
 *
 *  Download of content files from an ifrastructure server can be
 *  initiated via EC_CacheContent().  Status of cached content
 *  files can be retrieved via EC_GetCachedContentStatus().
 *
 *  Error codes include:
 *
 *      EC_ERROR_OK         - operation was successful
 *      EC_ERROR_PATH       - the path is not valid
 *      EC_ERROR_NOT_FOUND  - the filename did not correspond to an
 *                            existing file on a USB attached PC.
 *      EC_ERROR_BUSY       - A cached file is already open.
 *                            Only one can be open at a time.
 *      EC_ERROR_INVALID    - an argument was not valid
 *
 */
ECError  EC_OpenCachedContent (ESContentId cid);




/*  EC_ReadCachedContent
 *
 *  Attempt to read count bytes into buf from the current file position
 *  of the cached content file indicated by cid.  The content cache file
 *  associated with cid must be currently opened.
 *
 *  On success returns the number of bytes copied to buf and
 *  updates the file position by that amount.
 *
 *  If the operation completes with less then count bytes read, it is not
 *  an error.
 *
 *  EC_ReadCachedContent should be called repeatedly until the end of file or desired
 *  number of bytes have been read.
 *
 *  If the end of file is reached, the number of bytes that were copied
 *  to buf is returned.
 *
 *  If 0 is returned, the file position is at end of file.
 *
 *  Error codes returned include
 *
 *      EC_ERROR_NOT_OPEN  - cached content file is not open
 *      EC_ERROR_INVALID   - an argument was not valid
 */
s32 EC_ReadCachedContent (ESContentId   cid,
                          u8           *buf,
                          u32           count);





/*  EC_SeekCachedContent
 *
 *  Set the file position for the cached content file indicated by cid
 *  to offset from the start of file.   The content cache file associated
 *  with cid must currently be opened.
 *
 *  If the file position is set past the end of file, subsequent reads
 *  will indicate zero bytes read.
 *
 *  Error codes include:
 *
 *      EC_ERROR_OK        - operation was successful
 *      EC_ERROR_NOT_OPEN  - cached content file is not open
 *
 */
ECError  EC_SeekCachedContent (ESContentId  cid,
                               u32          offset);



/*  EC_CloseCachedContent
 *
 *  Close the cached content file indicated by cid.
 *
 *  Resources will be released.
 *
 *  Error codes include:
 *
 *      EC_ERROR_OK        - operation was successful
 *      EC_ERROR_NOT_OPEN  - cached content file is not open
 *
 */
ECError EC_CloseCachedContent (ESContentId  cid);


/*  EC_DeleteCachedContent
 *
 *  Delete the content file cached on the USB attached PC
 *  that is associated with cid.
 *
 *  It is not treated as an error if the file is not present.
 *
 *  The file does not need to be open.  If the file is open,
 *  it will be closed and removed.
 *
 *  Error codes include:
 *
 *      EC_ERROR_OK       - operation was successful
 *      <TBD>
 *
 */
ECError  EC_DeleteCachedContent (ESContentId  cid);



/*  EC_CacheContent
 *
 *  Start download of a set of content files from an
 *  infrastructure server to the USB attached PC file cache.
 * 
 *  For each content id in cids[]
 *      if the content file associated with cids[i] is not
 *      in the cache of the attached PC, download it from an
 *      infrastructure server and store it in the PC content cache.
 *
 *  nCids is the number of content ids in cids[]
 *
 *  This function starts the operation of checking and
 *  downloading the content files but does not wait for completion.
 *
 *  Use EC_GetCachedContentStatus() to get the cache status of the files.
 *
 *  Error codes include:
 *
 *      EC_ERROR_OK       - operation was successfully started
 *
 */
ECError EC_CacheContent (ESContentId  *cids,
                         u32           nCids);



/*  EC_GetCachedContentStatus
 *
 *  Get the status of content files cached on the USB attached PC.
 *
 *  Stores the ECCacheStatus value corresponding to the content id
 *  passed in at cids[i] in the caller provided output buffer at
 *  cacheStatus[i].
 *
 *  The number of content ids in cids[] is indicated by nCids.
 *
 *  A buffer large enough to hold status for nCids must be provided
 *  at *cacheStatus.
 *
 *  See the description of ECCacheStatus for the meaning of the
 *  cache status values.
 *
 *  Error codes include:
 *
 *      EC_ERROR_OK      - operation was successful
 *      EC_ERROR_INVALID - an argument was not valid
 *                         cids, nCids, and cacheStatus must be non-zero
 *
 */
ECError EC_GetCachedContentStatus (ESContentId        *cids,
                                   u32                 nCids,
                                   ECCacheStatus      *cacheStatus);




#if !defined(_GBA) && !defined(_SC)
#pragma pack(pop)
#endif




#ifdef __cplusplus
}
#endif //__cplusplus



#endif /*__EC_H__*/
