/********************************************************************/
/*          AgbFlash1M.h                                            */
/*            1Mbit FLASH Library Header    ver. 1.0.2              */
/*                                                                  */
/*                                         last modified 2002.11.15 */
/*                                                                  */
/*          Copyright (C) 2000-2002 NINTENDO Co.,Ltd.               */
/********************************************************************/
#ifndef __AGB_FLASH_1M_H__
#define __AGB_FLASH_1M_H__

#ifdef __cplusplus
extern "C" {
#endif



#ifndef __AGB_FLASH_512K_H__

#include <Agb.h>


// define data----------------------------------
#define FLASH_ADR           0x0e000000      // Flash Start Address
#define FLASH_RETRY_MAX     3               // Maximum retry number for the 
                                            // ProgramFlashSectorEx function 


typedef struct flashSectorTag {
    u32         size;           // Sector Size
    u8          shift;          // Value converted from sector
    u16         count;          // Sector Count
    u16         top;            // Usable start sector No.
}flashSector;

typedef struct flashTypeTag {
    u32         romSize;        // ROM Size
    flashSector sector;         // Sector Information (* See flashSector above)
    u16         agbWait[2];     // read/write wait value for AGB Game Pak bus
    u8          makerID;        // Vendor ID
    u8          deviceID;       // Device ID
}flashType;


/*------------------------------------------------------------------*/
/*          Global Variabl                                          */
/*------------------------------------------------------------------*/
extern const flashType *flash;  // Pointer to data showing content of flash
                                // (See definition of above data for details)
extern u16 flash_remainder;     // Barometer to indicate the progress when writing flash
    // * With Atmel's flash device, flash_remainder decreases in increments
     // of 128. For all other devices, flash_remainder decreases in increments of 1.

#endif	/*__AGB_FLASH_512K_H__*/


/*------------------------------------------------------------------*/
/*          Device Identification and Initial Settings              */
/*------------------------------------------------------------------*/

extern u16 IdentifyFlash_1M();

/* Reads flash ID. Identifies what flash is in Game Pak. Gets flash 
   capacity and sector size, sets up access speed, and sets the corresponding
   access functions for flash. The flash data obtained can be referenced with
   the global variable [flashType *flash]. (See definition above for details
   on flashType.)
   Call this function once before accessing flash (including read).
   If the device cannot be identified, an error is returned and the subsequent
   access function cannot be used.
    <Arguments>
    None
    <Return Values>
    u16 result  : 0 is a normal end, Not 0 is an error
*/


#ifndef __AGB_FLASH_512K_H__
/*------------------------------------------------------------------*/
/*          Timer Interrupt Settings for Access                     */
/*------------------------------------------------------------------*/

extern u16 SetFlashTimerIntr(u8 timerNo, void (**IntrFunc)(void));

/*  Selects timer interrupt for determining timeouts during access to flash.
    This routine needs to be called at least once before the library functions 
    EraseFlashChip, EraseFlashSector, and ProgramFlashSector are called. Once 
    the timer interrupt used with the routine has been set, there is no need 
    to call this routine again, unless the timer interrupt whose number has 
    been set is used by other processes, or unless another interrupt table 
    is used.
    If timer No. is not between 0 an 3, an error is returned.
    * According to the library specifications, when this routing is called, a 
      specific timer interrupt routine is forcibly set for the library functions
      so the interrupt table when accessing flash must be in the RAM area.

    <Arguments>
    u8 timerNo  : Number of the timer used by the timeout routine.
    void (**IntrFunc)(void): Pointer to compatible timer interrupt address
                             in interrupt table.
    <Return Values>
    u16 result  : 0 is a normal end. Not 0 is an error.
*/

/*------------------------------------------------------------------*/
/*          Read Data                                               */
/*------------------------------------------------------------------*/

extern void ReadFlash(u16 secNo,u32 offset,u8 *dst,u32 size);

/*   From "offset" byte start address in flash target sector number, read 
     data equivalent to "size" bytes to area starting from "dst" address 
     in work area.
    <Arguments>
    u16 secNo   : Target sector number
    u32 offset  : Offset bytes in sector
    u8 *dst     : Address of work area where read data is stored
    u32 size    : Read size in bytes
    <Return Values>
    None
*/

/*------------------------------------------------------------------*/
/*          Erase Chip                                              */
/*------------------------------------------------------------------*/

extern u16 (*EraseFlashChip)();

/*  Completely erase all of chip.
     * SetEepromTimerIntr must be called prior to this routine.
    <Arguments>
     None
    <Return Values>
     u16 result      : 0 is a normal end, Not 0 is an error
*/

/*------------------------------------------------------------------*/
/*          Erase Sector                                            */
/*------------------------------------------------------------------*/

extern u16 (*EraseFlashSector)(u16 secNo);

/*   Erase sector of target sector number.
     This routine is called in the write routine so it is usually not 
     necessary to call it before a write. If the target sector number is 
     outside the range, an error is returned. 
     * SetEepromTimerIntr must be called prior to this routine.
     <Arguments>
     u16 secNo  : Target sector number 
     <Return Values>
     u16 result : 0 is a normal end, Not 0 is an error

     * Please note that all interrupts are prohibited during a certain 
       period when using Flash manufactured by Atmel. See "Cautions 
       when Using Flash Access Function" in the "AGB Back-up Library Manual" 
       for details. 
     Do not use DMAs which start up automatically with specified timing  
     such as direct sound, V/H blank synchronization, display synchronization
     or Game Pak requests when this function is called.
*/

/*------------------------------------------------------------------*/
/*          Write of Sector Unit Data                               */
/*------------------------------------------------------------------*/

extern u16 (*ProgramFlashSector)(u16 secNo,u8 *src);

/*   Write 1 sector (4 kbytes) to target sector number from src address.
     Call EraseFlashSector (described above) in this routine, and write after 
     erasing sector. If the target sector number is outside the range, an 
     error is returned.
     When executing this routine, you can determine the remaining byte 
     count by referencing the global variable flash_remainder.
     * SetEepromTimerIntr must be called prior to this routine.
     <Arguments>
      u16 secNo :Target sector number
      u8 *src   :Write source address
     <Return Values>
      u16 result    :0 is anormal end, Not 0 is an error

     * Please note that all interrupts are prohibited during a certain 
       period when using Flash manufactured by Atmel. See "Cautions 
       when Using Flash Access Function" in the "AGB Back-up Library Manual" 
       for details. 
     Do not use DMAs which start up automatically with specified timing  
     such as direct sound, V/H blank synchronization, display synchronization
     or Game Pak requests when this function is called.
*/

/*------------------------------------------------------------------*/
/*          Verify data in sector units                             */
/*------------------------------------------------------------------*/

extern u32 VerifyFlashSector(u16 secNo,u8 *src);

/*   Verify one sector (4 KB) of data at the target sector number in the 
     flash with data in "src" address. If verify ends normally, 0 is returned. 
     If an error occurs, returns the address where the error occurred.
    <Arguments>
    u16 secNo   : Target sector number
    u8 *src     : Verify source address (original data)
    <Return Values>
    u32 errorAdr: 0 is a normal end. Returns the error address on the device side
                  where the verify error occurred.
                  
*/

/*------------------------------------------------------------------*/
/*          Verify data in bytes units                              */
/*------------------------------------------------------------------*/

extern u32 VerifyFlash(u16 secNo,u8 *src,u32 size);

/*  Verify data from "src" address and data of target sector number of 
    flash for the value (in bytes) of "size".
    If verify ends normally, 0 is returned.
    If a verify error occurs, returns the address where the error occurred.
    <Arguments>
    u16 secNo   : Target sector number
    u8 *src     : Verify source address (original data)
    u32 size    : Verify size (bytes)
    <Return Values>
    u32 errorAdr: 0 is a normal end. Returns the error address on the device side
                  where the verify error occurred.
                   
*/

/*------------------------------------------------------------------*/
/*    Write data in sector units and verify in sector units         */
/*------------------------------------------------------------------*/

extern u32 ProgramFlashSectorEx(u16 secNo,u8 *src);

/* This function writes internally using ProgramFlashSector and then 
   verifies using VeriyFlashSector. In case of an error, it retries a
   maximum of FLASH_RETRY_MAX times (defined by AgbFlash.h).
   * This function returns an error code in 16 bits out of 32 bits during a write
     error. However, it returns a 32-bit error address on the device side during a 
     verify error. So, be aware of this when checking an error code.
   <Arguments>
    u16 secNo   : Target sector number
    u8 *src     : Write source address
   <Return Value>
    u32 result  : 0 is a normal end.
                 16-bit error code when a write error occurs.
                 Error address on device side when a verify error occurs.

     * Please note that all interrupts are prohibited during a certain 
       period when using Flash manufactured by Atmel. See "Cautions 
       when Using Flash Access Function" in the "AGB Back-up Library Manual" 
       for details. 
    Do not use DMAs which start up automatically with specified timing  
    such as direct sound, V/H blank synchronization, display synchronization
    or Game Pak requests when this function is called.
*/

/*------------------------------------------------------------------*/
/*       Write data in sector units and verify in bytes units       */
/*------------------------------------------------------------------*/

extern u32 ProgramFlashSectorEx2(u16 secNo,u8 *src,u32 verifySize);

/*  This function writes internally using ProgramFlashSector and then
    verifies verifySize bytes worth using VerifyFlash.
    In case of an error, it retries a maximum of FLASH_RETRY_MAX times (defined 
    by AgbFlash.h).
    * This function returns an error code in 16 bits out of 32 bits during a write
      error. However, it returns a 32-bit error address on the device side during a
      verify error.  So, be aware of this when checking an error code.
    <Arguments>
    u16 secNo   : Target sector number
    u8 *src     : Write source address
    u32 verifySize : Verify size (byte)
    <Return Values>
    u32 result  : 0 is a normal end.
                  16-bit error code when a write error occurs.
                  Error address on device side when a verify error occurs.

     * Please note that all interrupts are prohibited during a certain 
       period when using Flash manufactured by Atmel. See "Cautions 
       when Using Flash Access Function" in the "AGB Back-up Library Manual" 
       for details. 
     Do not use DMAs which start up automatically with specified timing 
     such as direct sound, V/H blank synchronization, display synchronization
     or Game Pak requests when this function is called.
*/

#endif	/*__AGB_FLASH_512K_H__*/


#ifdef __cplusplus
}      /* extern "C" */
#endif

#endif  // __AGB_FLASH_1M_H__
