/********************************************************************/
/*         AgbSramFast.h                                            */
/*     256kbit SRAM High Speed Access Library Header     ver. 1.0.3 */
/*                                                                  */
/*                                         last modified 2002.11.15 */
/*                                                                  */
/*          Copyright (C) 2000-2002 NINTENDO Co.,Ltd.               */
/********************************************************************/
#ifndef __AGB_SRAM_FAST_H__
#define __AGB_SRAM_FAST_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <Agb.h>

//#define   __SRAM_DEBUG

#ifndef __SRAM_DEBUG
#define SRAM_ADR                0x0e000000      // SRAM Start Address
#define SRAM_SIZE_256K          0x00008000      // 256KSRAM
#define SRAM_SIZE_512K          0x00010000      // 512KSRAM
#else
#define SRAM_ADR                0x02018000
#define SRAM_SIZE_256K          0x00000400
#define SRAM_SIZE_512K          0x00000800
#endif
#define SRAM_F_RETRY_MAX        3               // Maximum retry number for the 
                                                // WriteSramEx function 

/*------------------------------------------------------------------
  Compared to the function group for AgbSram.h, the function group 
  in this header file can access SRAM at a high speed.
  However, note that it uses a static variable area of approximately
  300 bytes.
--------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/*          Access Routine Settings                                 */
/*------------------------------------------------------------------*/
extern void SetSramFastFunc();

/*   According to the specifications for the routines ReadSram() and
     VerifySram(), these functions need to operate on WRAM, so these 
     routines are transfered to the WRAM static area.
     Call this function once before accessing SRAM (including reads).
     <Arguments>
      None
     <Return Values>
      None
*/


/*------------------------------------------------------------------*/
/*          Read Data                                               */
/*------------------------------------------------------------------*/

extern void (*ReadSramFast)(u8 *src,u8 *dst,u32 size);

/*   Read "size" bytes of data from SRAM address specified by arguments 
     to the area starting from the "dst" address in the work area.
     <Arguments>
         u8 *src : Read source SRAM address (Address on AGB memory map)
         u8 *dst : Address of work area where read data is stored
                   (Address on AGB memory map)
        u32 size : Read size in bytes
     <Return Values>
        None
*/


/*------------------------------------------------------------------*/
/*          Write Data                                              */
/*------------------------------------------------------------------*/

extern void WriteSramFast(u8 *src,u8 *dst,u32 size);

/*   Write "size" byte data from work area address specified with argument 
     to area starting from "dst" address in SRAM.
     <Arguments>
      u8 *src  : Write source work area address
      u8 *dst  : Write destination SRAM address (Address on AGB memory map)
      u32 size : Write size in bytes
     <Return Values>
      None
*/

/*------------------------------------------------------------------*/
/*          Verify Data                                             */
/*------------------------------------------------------------------*/

extern u32 (*VerifySramFast)(u8 *src,u8 *tgt,u32 size);

/*   Verify "size" byte of data from the "src" address in the work area 
     and "tgt" address in SRAM.
     If verify ends normally this function returns a 0, if a verify 
     error occurs, the address where it occurred is returned.
     <Arguments>
     u8 *src : Pointer to verify source work area address (original data)
     u8 *tgt : Pointer to verify destination SRAM address
                (Write destination data, address on AGB memory map)
     u32 size: Verify size in bytes
     <Return Values>
     u32 errorAdr: If normal, returns 0. 
                   If a verify error occurs, returns error address.
*/


/*------------------------------------------------------------------*/
/*          Write data & Verify                                     */
/*------------------------------------------------------------------*/

extern u32 WriteSramFastEx(u8 *src, u8 *dst, u32 size);

/* This function writes internally using WriteSramFast and then 
   verifies using VerifySramFast. In case of an error, it retries a 
   maximum of SRAM_F_RETRY_MAX times (defined by AgbSramFast.h).
     
    <Argument>
     u8 *src    : Work area address of write source  
     u8 *dst    : SRAM address of write destination (address on AGB memory map)  
     u32 size   : Write size in number of bytes 
 
    <Return value> 
     u32 errorAdr  : Normal end  -> 0
                     Verify error  -> Error address on device side 
*/

#ifdef __cplusplus
}      /* extern "C" */
#endif

#endif  // __AGB_SRAM_FAST_H__
