/*----------------------------------------------------------------------*/
/*		LHarc Archiver Driver for UNIX				*/
/*		This is part of LHarc UNIX Archiver Driver		*/
/*									*/
/*		Copyright(C) MCMLXXXIX  Yooichi.Tagawa			*/
/*		Thanks to H.Yoshizaki. (MS-DOS LHarc)			*/
/*									*/
/*  V0.00  Original				1988.05.23  Y.Tagawa	*/
/*  V0.01  Alpha Version (for 4.2BSD)		1989.05.28  Y.Tagawa	*/
/*  V0.02  Alpha Version Rel.2			1989.05.29  Y.Tagawa	*/
/*  V0.03  Release #3  Beta Version		1989.07.02  Y.Tagawa	*/
/*  V0.03a Debug				1989.07.03  Y.Tagawa	*/
/*  V0.03b Modified				1989.07.13  Y.Tagawa	*/
/*  V0.03c Debug (Thanks to void@rena.dit.junet)1989.08.09  Y.Tagawa	*/
/*  V0.03d Modified (quiet and verbose)		1989.09.14  Y.Tagawa	*/
/*  V1.00  Fixed				1989.09.22  Y.Tagawa	*/
/*  V1.01  Bug Fixed				1989.12.25  Y.Tagawa	*/
/*----------------------------------------------------------------------*/

#include "lharc.h"


/*----------------------------------------------------------------------*/
/*		PROGRAM 						*/
/*----------------------------------------------------------------------*/


#define CMD_UNKNOWN	0
#define CMD_EXTRACT	1
#define CMD_ADD		2
#define CMD_LIST	3
#define CMD_DELETE	4

static int	cmd = CMD_UNKNOWN;
char	**cmd_filev;
int	cmd_filec;

char	*archive_name;
char	expanded_archive_name[FILENAME_LENGTH];
char	temporary_name[FILENAME_LENGTH];
char	backup_archive_name[FILENAME_LENGTH];


/* options */
boolean	quiet = FALSE;
boolean	text_mode = FALSE;
boolean	verbose = FALSE;
boolean	noexec = FALSE;	/* debugging option */
boolean	force = FALSE;
boolean	prof = FALSE;


/* view command flags */
boolean	verbose_listing = FALSE;

/* extract command flags */
boolean	output_to_stdout = FALSE;

/* append command flags */
boolean	new_archive = FALSE;
boolean	update_if_newer = FALSE;
boolean	delete_after_append = FALSE;
boolean	generic_format = FALSE;

boolean	remove_temporary_at_error = FALSE;
boolean	recover_archive_when_interrupt = FALSE;
boolean	remove_extracting_file_when_interrupt = FALSE;


/*----------------------------------------------------------------------*/
/* NOTES :	Text File Format					*/
/*	GENERATOR		NewLine					*/
/*	[generic]		0D 0A					*/
/*	[MS-DOS]		0D 0A					*/
/*	[MacOS]			0D					*/
/*	[UNIX]			0A					*/
/*----------------------------------------------------------------------*/


static void
print_tiny_usage_and_exit ()
{
  fprintf (stderr, "\
LHarc for UNIX  V1.02   Copyright(C) 1989  Y.Tagawa\n\
usage: lharc {axelvudmcp}[qvnftgvd] archive_file [files or directories...]\n\
commands:				options:\n\
 a   Add(or replace) to archive		 q  quiet\n\
 x,e EXtract from archive		 v  verbose\n\
 l,v List / Verbose List		 n  no execute\n\
 u   Update newer files to archive	 f  force (over write at extract)\n\
 d   Delete from archive		 t  FILES are TEXT file\n\
 m   Move to archive (means 'ad')	 g  [Generic] format (compatiblity)\n\
 c   re-Construct new archive\n\
 p   Print to STDOUT from archive	 d  delete FILES after a/u/c command\n\
");
  exit (1);
}

main (argc, argv)
     int argc;
     char *argv[];
{
  char *p;

  if (argc < 3)
    print_tiny_usage_and_exit ();

  /* commands */
  switch (argv[1][0])
    {
    case 'x':
    case 'e':
      cmd = CMD_EXTRACT;
      break;

    case 'p':
      output_to_stdout = TRUE;
      cmd = CMD_EXTRACT;
      break;

    case 'c':
      new_archive = TRUE;
      cmd = CMD_ADD;
      break;

    case 'a':
      cmd = CMD_ADD;
      break;

    case 'd':
      cmd = CMD_DELETE;
      break;

    case 'u':
      update_if_newer = TRUE;
      cmd = CMD_ADD;
      break;

    case 'm':
      delete_after_append = TRUE;
      cmd = CMD_ADD;
      break;

    case 'v':
      verbose_listing = TRUE;
      cmd = CMD_LIST;
      break;

    case 'l':
      cmd = CMD_LIST;
      break;

    case 'h':
    default:
      print_tiny_usage_and_exit ();
    }

  /* options */
  p = &argv[1][1];
  for (p = &argv[1][1]; *p; )
    {
      switch (*p++)
	{
	case 'q':	quiet = TRUE; break;
	case 'f':	force = TRUE; break;
	case 'p':	prof = TRUE; break;
	case 'v':	verbose = TRUE; break;
	case 't':	text_mode = TRUE; break;
	case 'n':	noexec = TRUE; break;
	case 'g':	generic_format = TRUE; break;
	case 'd':	delete_after_append = TRUE; break;

	default:
	  fprintf (stderr, "LHarc: Unknown option '%c'.\n", p[-1]);
	  exit (1);
	}
    }

  /* archive file name */
  archive_name = argv[2];

  /* target file name */
  cmd_filec = argc - 3;
  cmd_filev = argv + 3;
  sort_files ();

  switch (cmd)
    {
    case CMD_EXTRACT:	cmd_extract ();	break;
    case CMD_ADD:	cmd_add ();	break;
    case CMD_LIST:	cmd_list ();	break;
    case CMD_DELETE:	cmd_delete ();	break;
    }

#ifdef USE_PROF
  if (!prof)
    exit (0);
#endif

  exit (0);
}

static
message_1 (title, subject, name)
     char *title, *subject, *name;
{
  fprintf (stderr, "LHarc: %s%s ", title, subject);
  fflush (stderr);

  if (errno == 0)
    fprintf (stderr, "%s\n", name);
  else
    perror (name);
}

void
message (subject, name)
     char *subject, *name;
{
  message_1 ("", subject, name);
}

void
warning (subject, name)
     char *subject, *name;
{
  message_1 ("Warning: ", subject, name);
}

void
error (subject, msg)
     char *subject, *msg;
{
  message_1 ("Error: ", subject, msg);
}

void
fatal_error (msg)
     char *msg;
{
  message_1 ("Fatal error:", "", msg);

  if (remove_temporary_at_error)
    unlink (temporary_name);

  exit (1);
}

char *writting_filename;
char *reading_filename;

write_error ()
{
  fatal_error (writting_filename);
}

read_error ()
{
  fatal_error (reading_filename);
}

interrupt (signo)
     int signo;
{
  errno = 0;
  message ("Interrupted\n", "");

  unlink (temporary_name);
  if (recover_archive_when_interrupt)
    rename (backup_archive_name, archive_name);
  if (remove_extracting_file_when_interrupt)
    {
      errno = 0;
      message ("Removing", writting_filename);
      unlink (writting_filename);
    }
  signal (SIGINT, SIG_DFL);
  signal (SIGHUP, SIG_DFL);
  kill (getpid (), signo);
}



/*----------------------------------------------------------------------*/
/*									*/
/*----------------------------------------------------------------------*/

static boolean
expand_archive_name (dst, src)
     char *dst, *src;
{
  register char *p, *dot;

  strcpy (dst, src);

  for (p = dst, dot = (char*)0; ; p ++, dot = (char*)0)
    {
      while (*p == '.')		/* skip leading dots of base name */
	p ++;
      for (; *p != '/'; p ++)
	{
	  if (*p == '.')
	    dot = p;
	  else if (*p == '\0')
	    goto _Done;
	}
    }
  _Done:

  if (dot)
    p = dot;

  strcpy (p, ARCHIVENAME_EXTENTION); /* ".lzh" */
  return (strcmp (dst, src) != 0);
}

static int
sort_by_ascii (a, b)
     char **a, **b;
{
  register char *p, *q;
  register int c1, c2;

  p = *a, q = *b;
  if (generic_format)
    {
      do
	{
	  c1 = *(unsigned char*)p ++;
	  c2 = *(unsigned char*)q ++;
	  if (!c1 || !c2)
	    break;
	  if (islower (c1))
	    c1 = toupper (c1);
	  if (islower (c2))
	    c2 = toupper (c2);
	}
      while (c1 == c2) ;
      return c1 - c2;
    }
  else
    {
      while (*p == *q && *p != '\0')
	p ++, q ++;
      return *(unsigned char*)p - *(unsigned char*)q;
    }
}

sort_files ()
{
  if (cmd_filec > 1)
    qsort (cmd_filev, cmd_filec, sizeof (char*), sort_by_ascii);
}

char *xmalloc (size)
     int size;
{
  char *p = malloc (size);
  if (!p)
    fatal_error ("Not enough memory");
  return p;
}

char *xrealloc (old, size)
     char *old;
     int size;
{
  char *new = realloc (old, size);
  if (!new)
    fatal_error ("Not enough memory");
  return new;
}

/*----------------------------------------------------------------------*/
/*		STRING POOL 						*/
/*----------------------------------------------------------------------*/

/*
 * string pool :
 *	+-------------+-------------+---     ---+-------------+----------+
 *	| N A M E 1 \0| N A M E 2 \0|    ...    | N A M E n \0|	         |
 *	+-------------+-------------+---     ---+-------------+----------+
 *	^						      ^		 ^
 * buffer+0					    buffer+used  buffer+size
 */

/*
 * vector :
 *	+---------------+---------------+-------------     -------------+
 *	| pointer to	| pointer to	| pointer to   ...  pointer to	|
 *	|  string pool	|  N A M E 1	|  N A M E 2   ...   N A M E n	|
 *	+---------------+---------------+-------------     -------------+
 *	^		^
 *   malloc base      returned
 */

void
init_sp (sp)
     struct string_pool *sp;
{
  sp->size = 1024 - 8;		/* any ( >=0 ) */
  sp->used = 0;
  sp->n = 0;
  sp->buffer = (char*)xmalloc (sp->size * sizeof (char));
}

void
add_sp (sp, name, len)
     struct string_pool *sp;
     char *name;		/* stored '\0' at tail */
     int len;			/* include '\0' */
{
  while (sp->used + len > sp->size)
    {
      sp->size *= 2;
      sp->buffer = (char*) xrealloc (sp->buffer, sp->size * sizeof (char));
    }
  bcopy (name, sp->buffer + sp->used, len);
  sp->used += len;
  sp->n ++;
}

void
finish_sp (sp, v_count, v_vector)
     register struct string_pool *sp;
     int *v_count;
     char ***v_vector;
{
  int i;
  register char *p;
  char **v;

  v = (char**) xmalloc ((sp->n + 1) * sizeof (char*));
  *v++ = sp->buffer;
  *v_vector = v;
  *v_count = sp->n;
  p = sp->buffer;
  for (i = sp->n; i; i --)
    {
      *v++ = p;
      if (i - 1)
	p += strlen (p) + 1;
    }
}

void
free_sp (vector)
     char **vector;
{
  vector --;
  free (*vector);		/* free string pool */
  free (vector);
}


/*----------------------------------------------------------------------*/
/*		READ DIRECTORY FILES					*/
/*----------------------------------------------------------------------*/

static boolean
include_path_p (path, name)
     char *path, *name;
{
  char *n = name;
  while (*path)
    if (*path++ != *n++)
      return (path[-1] == '/' && *n == '\0');
  return (*n == '/' || (n != name && path[-1] == '/' && n[-1] == '/'));
}

#define STREQU(a,b)	(((a)[0] == (b)[0]) ? (strcmp ((a),(b)) == 0) : FALSE)
void
cleaning_files (v_filec, v_filev)
     int *v_filec;
     char ***v_filev;
{
  char *flags;
  struct stat stbuf;
  register char **filev = *v_filev;
  register int filec = *v_filec;
  register char *p;
  register int i, j;

  if (filec == 0)
    return;

  flags = xmalloc (filec * sizeof (char));

  /* flags & 0x01 :	1: ignore */
  /* flags & 0x02 :	1: directory, 0 : regular file */
  /* flags & 0x04 :	1: need delete */

  for (i = 0; i < filec; i ++)
    if (stat (filev[i], &stbuf) < 0)
      {
	flags[i] = 0x04;
	fprintf (stderr,
		 "LHarc: Cannot access \"%s\", ignored.\n", filev[i]);
      }
    else
      {
	if (is_regularfile (&stbuf))
	  flags[i] = 0x00;
	else if (is_directory (&stbuf))
	  flags[i] = 0x02;
	else
	  {
	    flags[i] = 0x04;
	    fprintf (stderr,
		     "LHarc: Cannot archive \"%s\", ignored.\n", filev[i]);
	  }
      }
  errno = 0;

  for (i = 0; i < filec; i ++)
    {
      p = filev[i];
      if ((flags[i] & 0x07) == 0x00)
	{			/* regular file, not deleted/ignored */
	  for (j = i + 1; j < filec; j ++)
	    {
	      if ((flags[j] & 0x07) == 0x00)
		{		/* regular file, not deleted/ignored */
		  if (STREQU (p, filev[j]))
		    flags[j] = 0x04; /* delete */
		}
	    }
	}
      else if ((flags[i] & 0x07) == 0x02)
	{			/* directory, not deleted/ignored */
	  for (j = i + 1; j < filec; j ++)
	    {
	      if ((flags[j] & 0x07) == 0x00)
		{		/* regular file, not deleted/ignored */
		  if (include_path_p (p, filev[j]))
		    flags[j] = 0x04; /* delete */
		}
	      else if ((flags[j] & 0x07) == 0x02)
		{		/* directory, not deleted/ignored */
		  if (include_path_p (p, filev[j]))
		    flags[j] = 0x04; /* delete */
		}
	    }
	}
    }

  for (i = j = 0; i < filec; i ++)
    {
      if ((flags[i] & 0x04) == 0)
	{
	  if (i != j)
	    filev[j] = filev[i];
	  j ++;
	}
    }
  *v_filec = j;

  free (flags);
}

#ifdef NODIRECTORY
/* please need your imprementation */
boolean
find_files (name, v_filec, v_filev)
     char *name;
     int *v_filec;
     char ***v_filev;
{
  return FALSE;			/* DUMMY */
}

void
free_files (filec, filev)
     int filec;
     char **filev;
{
  /* do nothing */
}
#else
boolean
find_files (name, v_filec, v_filev)
     char *name;
     int *v_filec;
     char ***v_filev;
{
  struct string_pool sp;
  char newname[FILENAME_LENGTH];
  int len, n;
  DIR *dirp;
  DIRENTRY *dp;

  strcpy (newname, name);
  len = strlen (name);
  if (len > 0 && newname[len-1] != '/')
    newname[len++] = '/';

  dirp = opendir (name);
  if (!dirp)
    return FALSE;

  init_sp (&sp);

  for (dp = readdir (dirp); dp != NULL; dp = readdir (dirp))
    {
      n = NAMLEN (dp);
      if ((dp->d_ino != 0) &&
	  /* exclude '.' and '..' */
	  ((dp->d_name[0] != '.') ||
	   ((n != 1) &&
	    ((dp->d_name[1] != '.') ||
	     (n != 2)))) &&
	  (strcmp (dp->d_name, temporary_name) != 0) &&
	  (strcmp (dp->d_name, archive_name) != 0))
	{
	  strncpy (newname+len, dp->d_name, n);
	  newname[len + n] = '\0';
	  add_sp (&sp, newname, len + n + 1);
	}
    }
  closedir (dirp);
  finish_sp (&sp, v_filec, v_filev);
  if (*v_filec > 1)
    qsort (*v_filev, *v_filec, sizeof (char*), sort_by_ascii);
  cleaning_files (v_filec, v_filev);

  return TRUE;
}

void
free_files (filec, filev)
     int filec;
     char **filev;
{
  free_sp (filev);
}
#endif



/*----------------------------------------------------------------------*/
/*									*/
/*----------------------------------------------------------------------*/

static int
calc_sum (p, len)
     register char *p;
     register int len;
{
  register int sum;

  for (sum = 0; len; len --)
    sum += *p++;

  return sum & 0xff;
}

static char *get_ptr;
#define setup_get(PTR)		(get_ptr = (PTR))
#define get_byte()		(*get_ptr++ & 0xff)
#define put_ptr			get_ptr
#define setup_put(PTR)		(put_ptr = (PTR))
#define put_byte(c)		(*put_ptr++ = (c))

static unsigned short
get_word ()
{
  int b0, b1;

  b0 = get_byte ();
  b1 = get_byte ();
  return (b1 << 8) + b0;
}

static
put_word (v)
     unsigned int v;
{
  put_byte (v);
  put_byte (v >> 8);
}

static long
get_longword ()
{
  long b0, b1, b2, b3;

  b0 = get_byte ();
  b1 = get_byte ();
  b2 = get_byte ();
  b3 = get_byte ();
  return (b3 << 24) + (b2 << 16) + (b1 << 8) + b0;
}

static
put_longword (v)
     long v;
{
  put_byte (v);
  put_byte (v >> 8);
  put_byte (v >> 16);
  put_byte (v >> 24);
}


static
msdos_to_unix_filename (name, len)
     register char *name;
     register int len;
{
  register int i;

#ifdef MULTIBYTE_CHAR
  for (i = 0; i < len; i ++)
    {
      if (MULTIBYTE_FIRST_P (name[i]) &&
	  MULTIBYTE_SECOND_P (name[i+1]))
	i ++;
      else if (name[i] == '\\')
	name[i] = '/';
      else if (isupper (name[i]))
	name[i] = tolower (name[i]);
    }
#else
  for (i = 0; i < len; i ++)
    {
      if (name[i] == '\\')
	name[i] = '/';
      else if (isupper (name[i]))
	name[i] = tolower (name[i]);
    }
#endif
}

static
generic_to_unix_filename (name, len)
     register char *name;
     register int len;
{
  register int i;
  boolean lower_case_used = FALSE;

#ifdef MULTIBYTE_CHAR
  for (i = 0; i < len; i ++)
    {
      if (MULTIBYTE_FIRST_P (name[i]) &&
	  MULTIBYTE_SECOND_P (name[i+1]))
	i ++;
      else if (islower (name[i]))
	{
	  lower_case_used = TRUE;
	  break;
	}
    }
  for (i = 0; i < len; i ++)
    {
      if (MULTIBYTE_FIRST_P (name[i]) &&
	  MULTIBYTE_SECOND_P (name[i+1]))
	i ++;
      else if (name[i] == '\\')
	name[i] = '/';
      else if (!lower_case_used && isupper (name[i]))
	name[i] = tolower (name[i]);
    }
#else
  for (i = 0; i < len; i ++)
    if (islower (name[i]))
      {
	lower_case_used = TRUE;
	break;
      }
  for (i = 0; i < len; i ++)
    {
      if (name[i] == '\\')
	name[i] = '/';
      else if (!lower_case_used && isupper (name[i]))
	name[i] = tolower (name[i]);
    }
#endif
}

static
macos_to_unix_filename (name, len)
     register char *name;
     register int len;
{
  register int i;

  for (i = 0; i < len; i ++)
    {
      if (name[i] == ':')
	name[i] = '/';
      else if (name[i] == '/')
	name[i] = ':';
    }
}

static
unix_to_generic_filename (name, len)
     register char *name;
     register int len;
{
  register int i;

  for (i = 0; i < len; i ++)
    {
      if (name[i] == '/')
	name[i] = '\\';
      else if (islower (name[i]))
	name[i] = toupper (name[i]);
    }
}

/*----------------------------------------------------------------------*/
/*									*/
/*	Generic stamp format:						*/
/*									*/
/*	 31 30 29 28 27 26 25 24 23 22 21 20 19 18 17 16		*/
/*	|<-------- year ------->|<- month ->|<-- day -->|		*/
/*									*/
/*	 15 14 13 12 11 10  9  8  7  6  5  4  3  2  1  0		*/
/*	|<--- hour --->|<---- minute --->|<- second*2 ->|		*/
/*									*/
/*----------------------------------------------------------------------*/


static long
gettz ()
{
   struct timeval tp;
   struct timezone tzp;
   gettimeofday (&tp, &tzp);	/* specific to 4.3BSD */
/* return (tzp.tz_minuteswest * 60L + (tzp.tz_dsttime != 0 ? 60L * 60L : 0));*/
   return (tzp.tz_minuteswest * 60L);
}

#ifdef NOT_USED
static struct tm *
msdos_to_unix_stamp_tm (a)
     long a;
{
  static struct tm t;

  t.tm_sec	= ( a          & 0x1f) * 2;
  t.tm_min	=  (a >>    5) & 0x3f;
  t.tm_hour	=  (a >>   11) & 0x1f;
  t.tm_mday	=  (a >>   16) & 0x1f;
  t.tm_mon	=  (a >> 16+5) & 0x0f - 1;
  t.tm_year	= ((a >> 16+9) & 0x7f) + 80;
  return &t;
}
#endif

static time_t
generic_to_unix_stamp (t)
     long t;
{
  int year, month, day, hour, min, sec;
  long longtime;
  static unsigned int dsboy[12] = { 0, 31, 59, 90, 120, 151,
				      181, 212, 243, 273, 304, 334};
  unsigned int days;

  year  = ((int)(t >> 16+9) & 0x7f) + 1980;
  month =  (int)(t >> 16+5) & 0x0f;	/* 1..12 means Jan..Dec */
  day   =  (int)(t >> 16)   & 0x1f;	/* 1..31 means 1st,...31st */

  hour  =  ((int)t >> 11) & 0x1f;
  min   =  ((int)t >> 5)  & 0x3f;
  sec   = ((int)t         & 0x1f) * 2;

				/* Calculate days since 1970.01.01 */
  days = (365 * (year - 1970) + /* days due to whole years */
	  (year - 1970 + 1) / 4 + /* days due to leap years */
	  dsboy[month-1] +	/* days since beginning of this year */
	  day-1);		/* days since beginning of month */

  if ((year % 4 == 0) &&
      (year % 400 != 0) &&
      (month >= 3))		/* if this is a leap year and month */
    days ++;			/* is March or later, add a day */

  /* Knowing the days, we can find seconds */
  longtime = (((days * 24) + hour) * 60 + min) * 60 + sec;
  longtime += gettz ();      /* adjust for timezone */

  /* special case:  if MSDOS format date and time were zero, then we set
     time to be zero here too. */
  if (t == 0)
    longtime = 0;

  /* LONGTIME is now the time in seconds, since 1970/01/01 00:00:00.  */
  return (time_t)longtime;
}

static long
unix_to_generic_stamp (t)
     time_t t;
{
  struct tm *tm = localtime (&t);

  return ((((long)(tm->tm_year - 80)) << 25) +
	  (((long)(tm->tm_mon + 1)) << 21) +
	  (((long)tm->tm_mday) << 16) +
	  (long)((tm->tm_hour << 11) +
		 (tm->tm_min << 5) +
		 (tm->tm_sec / 2)));
}

/*----------------------------------------------------------------------*/
/*									*/
/*----------------------------------------------------------------------*/

boolean
get_header (fp, hdr)
     FILE *fp;
     register LzHeader *hdr;
{
  int header_size;
  int name_length;
  char data[LZHEADER_STRAGE];
  int checksum;
  int i;

  bzero (hdr, sizeof (LzHeader));

  if (((header_size = getc (fp)) == EOF) || (header_size == 0))
    {
      return FALSE;		/* finish */
    }

  if (fread (data + I_HEADER_CHECKSUM,
		  sizeof (char), header_size + 1, fp) < header_size + 1)
    {
      fatal_error ("Invalid header (LHarc file ?)");
      return FALSE;		/* finish */
    }

  setup_get (data + I_HEADER_CHECKSUM);
  checksum = calc_sum (data + I_METHOD, header_size);
  if (get_byte () != checksum)
    warning ("Checksum error (LHarc file?)", "");

  hdr->header_size = header_size;
  bcopy (data + I_METHOD, hdr->method, METHOD_TYPE_STRAGE);
#ifdef OLD
  if ((bcmp (hdr->method, LZHUFF1_METHOD, METHOD_TYPE_STRAGE) != 0) &&
      (bcmp (hdr->method, LZHUFF0_METHOD, METHOD_TYPE_STRAGE) != 0) &&
      (bcmp (hdr->method, LARC5_METHOD, METHOD_TYPE_STRAGE) != 0) &&
      (bcmp (hdr->method, LARC4_METHOD, METHOD_TYPE_STRAGE) != 0))
    {
      warning ("Unknown method (LHarc file ?)", "");
      return FALSE;		/* invalid method */
    }
#endif
  setup_get (data + I_PACKED_SIZE);
  hdr->packed_size	= get_longword ();
  hdr->original_size	= get_longword ();
  hdr->last_modified_stamp = get_longword ();
  hdr->attribute	= get_word ();
  name_length		= get_byte ();
  for (i = 0; i < name_length; i ++)
    hdr->name[i] =(char)get_byte ();
  hdr->name[name_length] = '\0';

  /* defaults for other type */
  hdr->unix_mode	= UNIX_FILE_REGULAR | UNIX_RW_RW_RW;
  hdr->unix_gid 	= 0;
  hdr->unix_uid		= 0;

  if (header_size - name_length >= 24)
    {				/* EXTEND FORMAT */
      hdr->crc				= get_word ();
      hdr->extend_type			= get_byte ();
      hdr->minor_version		= get_byte ();
      hdr->has_crc = TRUE;
    }
  else if (header_size - name_length == 22)
    {				/* Generic with CRC */
      hdr->crc				= get_word ();
      hdr->extend_type			= EXTEND_GENERIC;
      hdr->has_crc = TRUE;
    }
  else if (header_size - name_length == 20)
    {				/* Generic no CRC */
      hdr->extend_type			= EXTEND_GENERIC;
      hdr->has_crc = FALSE;
    }
  else
    {
      warning ("Unknown header (LHarc file ?)", "");
      return FALSE;
    }

  switch (hdr->extend_type)
    {
    case EXTEND_MSDOS:
      msdos_to_unix_filename (hdr->name, name_length);
      hdr->unix_last_modified_stamp	=
	generic_to_unix_stamp (hdr->last_modified_stamp);
      break;

    case EXTEND_UNIX:
      hdr->unix_last_modified_stamp	= (time_t)get_longword ();
      hdr->unix_mode			= get_word ();
      hdr->unix_uid			= get_word ();
      hdr->unix_gid			= get_word ();
      break;

    case EXTEND_MACOS:
      macos_to_unix_filename (hdr->name, name_length);
      hdr->unix_last_modified_stamp	=
	generic_to_unix_stamp (hdr->last_modified_stamp);
      break;

    default:
      generic_to_unix_filename (hdr->name, name_length);
      hdr->unix_last_modified_stamp	=
	generic_to_unix_stamp (hdr->last_modified_stamp);
    }

  return TRUE;
}

void
init_header (name, v_stat, hdr)
     char *name;
     struct stat *v_stat;
     LzHeader *hdr;
{
  int len;

  bcopy (LZHUFF1_METHOD, hdr->method, METHOD_TYPE_STRAGE);
  hdr->packed_size		= 0;
  hdr->original_size		= v_stat->st_size;
  hdr->last_modified_stamp	= unix_to_generic_stamp (v_stat->st_mtime);
  hdr->attribute		= GENERIC_ATTRIBUTE;
  strcpy (hdr->name, name);
  len = strlen (name);
  hdr->crc			= 0x0000;
  hdr->extend_type		= EXTEND_UNIX;
  hdr->unix_last_modified_stamp	= v_stat->st_mtime;
				/* since 00:00:00 JAN.1.1970 */
#ifdef NOT_COMPATIBLE_MODE
  Please need your modification in this space.
#else
  hdr->unix_mode		= v_stat->st_mode;
#endif

  hdr->unix_uid			= v_stat->st_uid;
  hdr->unix_gid			= v_stat->st_gid;

  if (is_directory (v_stat))
    {
      bcopy (LZHUFF0_METHOD, hdr->method, METHOD_TYPE_STRAGE);
      hdr->attribute = GENERIC_DIRECTORY_ATTRIBUTE;
      hdr->original_size = 0;
      if (len > 0 && hdr->name[len-1] != '/')
	strcpy (&hdr->name[len++], "/");
    }
  if (generic_format)
    unix_to_generic_filename (hdr->name, len);
}

/* Write unix extended header or generic header. */
void
write_header (nafp, hdr)
     FILE *nafp;
     LzHeader *hdr;
{
  int header_size;
  int name_length;
  char data[LZHEADER_STRAGE];

  bzero (data, LZHEADER_STRAGE);
  bcopy (hdr->method, data + I_METHOD, METHOD_TYPE_STRAGE);
  setup_put (data + I_PACKED_SIZE);
  put_longword (hdr->packed_size);
  put_longword (hdr->original_size);
  put_longword (hdr->last_modified_stamp);
  put_word (hdr->attribute);
  name_length = strlen (hdr->name);
  put_byte (name_length);
  bcopy (hdr->name, data + I_NAME, name_length);
  setup_put (data + I_NAME + name_length);
  put_word (hdr->crc);
  if (generic_format)
    {
      header_size = I_GENERIC_HEADER_BOTTOM - 2 + name_length;
    }
  else
    {
      put_byte (EXTEND_UNIX);
      put_byte (CURRENT_UNIX_MINOR_VERSION);
      put_longword ((long)hdr->unix_last_modified_stamp);
      put_word (hdr->unix_mode);
      put_word (hdr->unix_uid);
      put_word (hdr->unix_gid);
      header_size = I_UNIX_EXTEND_BOTTOM - 2 + name_length;
    }
  data[I_HEADER_SIZE] = header_size;
  data[I_HEADER_CHECKSUM] = calc_sum (data + I_METHOD, header_size);
  if (fwrite (data, sizeof (char), header_size + 2, nafp) == NULL)
    fatal_error ("Cannot write to temporary file");
}

/* If TRUE, archive file name is msdos SFX file name. */
boolean
archive_is_msdos_sfx1 (name)
     char *name;
{
  int len = strlen (name);

  return ((len >= 4) &&
	  (strcmp (name + len - 4, ".com") == 0 ||
	   strcmp (name + len - 4, ".exe") == 0 ||
	   strcmp (name + len - 4, ".COM") == 0 ||
	   strcmp (name + len - 4, ".EXE") == 0));
}

boolean
skip_msdos_sfx1_code (fp)
     FILE *fp;
{
  unsigned char buffer[2048];
  unsigned char *p, *q;
  int n;

  n = fread (buffer, sizeof (char), 2048, fp);

  for (p = buffer + 2, q = buffer + n - 5; p < q; p ++)
    {
      /* found "-l??-" keyword (as METHOD type string) */
      if (p[0] == '-' && p[1] == 'l' && p[4] == '-')
	{
	  /* size and checksum validate check */
	  if (p[-2] > 20 && p[-1] == calc_sum (p, p[-2]))
	    {
	      fseek (fp, ((p - 2) - buffer) - n, SEEK_CUR);
	      return TRUE;
	    }
	}
    }

  fseek (fp, -n, SEEK_CUR);
  return FALSE;
}


/*----------------------------------------------------------------------*/
/*									*/
/*----------------------------------------------------------------------*/

/* Build temporary file name and store to TEMPORARY_NAME */
void
build_temporary_name ()
{
#ifdef TMP_FILENAME_TEMPLATE
  /* "/tmp/lhXXXXXX" etc. */
  strcpy (temporary_name, TMP_FILENAME_TEMPLATE);
  mktemp (temporary_name);
#else
  char *p, *s;

  strcpy (temporary_name, archive_name);
  for (p = temporary_name, s = (char*)0; *p; p ++)
    if (*p == '/')
      s = p;
  strcpy ((s ? s+1 : temporary_name), "#..lhXXXXXX");
  mktemp (temporary_name);
#endif
}

static void
modify_filename_extention (buffer, ext)
     char *buffer;
     char *ext;
{
  register char *p, *dot;

  for (p = buffer, dot = (char*)0; *p; p ++)
    {
      if (*p == '.')
	dot = p;
      else if (*p == '/')
	dot = (char*)0;
    }

  if (dot)
    p = dot;

  strcpy (p, ext);
}

/* build backup file name */
void
build_backup_name (buffer, original)
     char *buffer;
     char *original;
{
  strcpy (buffer, original);
  modify_filename_extention (buffer, BACKUPNAME_EXTENTION); /* ".bak" */
}

void
build_standard_archive_name (buffer, orginal)
     char *buffer;
     char *orginal;
{
  strcpy (buffer, orginal);
  modify_filename_extention (buffer, ARCHIVENAME_EXTENTION); /* ".lzh" */
}

/*----------------------------------------------------------------------*/
/*									*/
/*----------------------------------------------------------------------*/


boolean
need_file (name)
     char *name;
{
  int i;

  if (cmd_filec == 0)
    return TRUE;

  for (i = 0; i < cmd_filec; i ++)
    {
      if (STREQU (cmd_filev[i], name))
	return TRUE;
      if (include_path_p (cmd_filev[i], name))
	return TRUE;
    }

  return FALSE;
}

FILE *
xfopen (name, mode)
     char *name, *mode;
{
  FILE *fp;

  if ((fp = fopen (name, mode)) == NULL)
    fatal_error (name);

  return fp;
}


/*----------------------------------------------------------------------*/
/*									*/
/*----------------------------------------------------------------------*/
int	archive_file_mode;
int	archive_file_gid;

static boolean
open_old_archive_1 (name, v_fp)
     char *name;
     FILE **v_fp;
{
  FILE *fp;
  struct stat stbuf;

  if (stat (name, &stbuf) >= 0 &&
      is_regularfile (&stbuf) &&
      (fp = fopen (name, READ_BINARY)) != NULL)
    {
      *v_fp = fp;
      archive_file_gid = stbuf.st_gid;
      archive_file_mode = stbuf.st_mode;
      return TRUE;
    }

  *v_fp = NULL;
  archive_file_gid = -1;
  return FALSE;
}

FILE *
open_old_archive ()
{
  FILE *fp;

  if (!open_old_archive_1 (archive_name, &fp))
    {
      if (expand_archive_name (expanded_archive_name, archive_name))
	{
	  archive_name = expanded_archive_name;
	  if (open_old_archive_1 (archive_name, &fp))
	    errno = 0;
	}
    }
  return fp;
}

int
inquire (msg, name, selective)
     char *msg, *name, *selective;
{
  char buffer[1024];
  char *p, *q;

  for (;;)
    {
      fprintf (stderr, "%s %s ", name, msg);
      fflush (stderr);

      fgets (buffer, 1024, stdin);

      for (p = selective; *p; p++)
	if (buffer[0] == *p)
	  return p - selective;
    }
  /*NOTREACHED*/
}

void
write_archive_tail (nafp)
     FILE *nafp;
{
  putc (0x00, nafp);
}

void
copy_old_one (oafp, nafp, hdr)
     FILE *oafp, *nafp;
     LzHeader *hdr;
{
  if (noexec)
    {
      fseek (oafp, (long)(hdr->header_size + 2) + hdr->packed_size, SEEK_CUR);
    }
  else
    {
      reading_filename = archive_name;
      writting_filename = temporary_name;
      copy_file (oafp, nafp, (long)(hdr->header_size + 2) + hdr->packed_size);
    }
}

