/*----------------------------------------------------------------------*/
/*		LHarc Extract Command					*/
/*		This is part of LHarc UNIX Archiver Driver		*/
/*									*/
/*		Copyright(C) MCMLXXXIX  Yooichi.Tagawa			*/
/*									*/
/*  V0.00  Original				1988.05.23  Y.Tagawa	*/
/*  V1.00  Fixed				1989.09.22  Y.Tagawa	*/
/*----------------------------------------------------------------------*/

#include "lharc.h"

extern int decode_lzhuf (), decode_larc ();
extern int decode_stored_crc (), decode_stored_nocrc ();



static boolean
inquire_extract (name)
     char *name;
{
  struct stat stbuf;

  if (stat (name, &stbuf) >= 0)
    {
      if (!is_regularfile (&stbuf))
	{
	  error ("Already exist (not a file)", name);
	  return FALSE;
	}

      if (noexec)
	{
	  printf ("EXTRACT %s but file is exist.\n", name);
	  return FALSE;
	}
      else if (!force)
	{
	  switch (inquire ("OverWrite ?(Yes/No/All)", name, "YyNnAa"))
	    {
	    case 0: case 1:	/* Y/y */
	      break;
	    case 2: case 3:	/* N/n */
	      return FALSE;
	    case 4: case 5:	/* A/a */
	      force = TRUE;
	      break;
	    }
	}
    }

  if (noexec)
    printf ("EXTRACT %s\n", name);

  return TRUE;
}

static boolean
make_parent_path (name)
     char *name;
{
  char path[FILENAME_LENGTH];
  struct stat stbuf;
  register char *p;

  /* make parent directory name into PATH for recursive call */
  strcpy (path, name);
  for (p = path + strlen (path); p > path; p --)
    if (p[-1] == '/')
      {
	*--p = '\0';
	break;
      }

  if (p == path)
    {
      message ("Why?", "ROOT");
      return FALSE;		/* no more parent. */
    }

  if (stat (path, &stbuf) >= 0)
    {
      if (is_directory (&stbuf))
	return TRUE;
      error ("Not a directory", path);
      return FALSE;
    }
  errno = 0;

  if (verbose)
    printf ("Making directory \"%s\".", path);

  if (mkdir (path, 0777) >= 0)	/* try */
    return TRUE;		/* successful done. */
  errno = 0;

  if (!make_parent_path (path))
    return FALSE;

  if (mkdir (path, 0777) < 0)	/* try again */
    {
      message ("Cannot make directory", path);
      return FALSE;
    }

  return TRUE;
}

static FILE *
open_with_make_path (name)
     char *name;
{
  FILE *fp;

  if ((fp = fopen (name, WRITE_BINARY)) == NULL)
    {
      errno = 0;
      if (!make_parent_path (name) ||
	  (fp = fopen (name, WRITE_BINARY)) == NULL)
	error ("Cannot extract", name);
      errno = 0;
    }

  return fp;
}

static int (*analyze_method (hdr))()
     LzHeader *hdr;
{
  int (*decode_proc)();

  if (bcmp (hdr->method, LZHUFF1_METHOD, METHOD_TYPE_STRAGE) == 0)
    decode_proc = decode_lzhuf;
  else if ((bcmp (hdr->method, LZHUFF0_METHOD, METHOD_TYPE_STRAGE) == 0) ||
	   (bcmp (hdr->method, LARC4_METHOD, METHOD_TYPE_STRAGE) == 0))
    decode_proc = (hdr->has_crc) ? decode_stored_crc : decode_stored_nocrc;
  else if (bcmp (hdr->method, LARC5_METHOD, METHOD_TYPE_STRAGE) == 0)
    decode_proc = decode_larc;
  else
    decode_proc = (int (*)())0;
  return decode_proc;
}

static void
adjust_info (hdr)
     LzHeader *hdr;
{
  time_t utimebuf[2];

  /* adjust file stamp */
  utimebuf[0] = utimebuf[1] = hdr->unix_last_modified_stamp;
  utime (hdr->name, utimebuf);

  if (hdr->extend_type == EXTEND_UNIX)
    {
#ifdef NOT_COMPATIBLE_MODE
      Please need your modification in this space.
#else
      chmod (hdr->name, hdr->unix_mode);
#endif
      chown (hdr->name, hdr->unix_uid, hdr->unix_gid);
      errno = 0;
    }
}

static void
extract_one (afp, hdr)
     FILE *afp;			/* archive file */
     LzHeader *hdr;
{
  FILE *fp;			/* output file */
  char *name;
  int crc;
  int (*decode_proc)();		/* (ifp,ofp,original_size,name) */
  boolean save_quiet, save_verbose;

  name = hdr->name;
  if ((hdr->unix_mode & UNIX_FILE_TYPEMASK) == UNIX_FILE_REGULAR)
    {
      decode_proc = analyze_method (hdr);
      if (!decode_proc)
	{
	  error ("Sorry, cannot extract this method, skipped.", name);
	  return;
	}
      reading_filename = archive_name;
      writting_filename = name;
      if (output_to_stdout)
	{
	  if (noexec)
	    {
	      printf ("EXTRACT %s\n", name);
	      return;
	    }

	  if (!quiet)
	    printf ("::::::::\n%s\n::::::::\n", name);
	  
	  save_quiet = quiet;
	  save_verbose = verbose;
	  quiet = TRUE;
	  verbose = FALSE;
	  crc = (*decode_proc) (afp, stdout, hdr->original_size, name);
	  quiet = save_quiet;
	  verbose = save_verbose;
	}
      else
	{
	  if (!inquire_extract (name))
	    return;

	  if (noexec)
	    return;

	  signal (SIGINT, interrupt);
	  signal (SIGHUP, interrupt);
	  unlink (name);
	  errno = 0;
	  remove_extracting_file_when_interrupt = TRUE;
	  if ((fp = open_with_make_path (name)) != NULL)
	    {
	      crc = (*decode_proc) (afp, fp, hdr->original_size, name);
	      fclose (fp);
	    }
	  remove_extracting_file_when_interrupt = FALSE;
	  signal (SIGINT, SIG_DFL);
	  signal (SIGHUP, SIG_DFL);
	  if (!fp)
	    return;
	}

      errno = 0;
      if (hdr->has_crc && crc != hdr->crc)
	error ("CRC error", name);

    }
  else if ((hdr->unix_mode & UNIX_FILE_TYPEMASK) == UNIX_FILE_DIRECTORY)
    {
      if (noexec)
	{
	  printf ("EXTRACT %s (directory)\n", name);
	  return;
	}
      /* NAME has trailing SLASH '/', (^_^) */
      if (!output_to_stdout && !make_parent_path (name))
	return;
    }
  else
    {
      error ("Unknown information", name);
    }

  if (!output_to_stdout)
    adjust_info (hdr);
}


/*----------------------------------------------------------------------*/
/*		EXTRACT COMMAND MAIN					*/
/*----------------------------------------------------------------------*/

void
cmd_extract ()
{
  LzHeader hdr;
  long pos;
  FILE *afp;

  /* open archive file */
  if ((afp = open_old_archive ()) == NULL)
    fatal_error (archive_name);
  if (archive_is_msdos_sfx1 (archive_name))
    skip_msdos_sfx1_code (afp);

  /* extract each files */
  while (get_header (afp, &hdr))
    {
      if (need_file (hdr.name))
	{
	  pos = ftell (afp);
	  extract_one (afp, &hdr);
	  fseek (afp, pos + hdr.packed_size, SEEK_SET);
	} else {
	  fseek (afp, hdr.packed_size, SEEK_CUR);
	}
    }

  /* close archive file */
  fclose (afp);

  return;
}

